/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.selenium.page.imp.vaadin;

import net.sourceforge.jbizmo.commons.selenium.data.PageActionResult;

import org.openqa.selenium.WebElement;

/**
 * <p>
 * Page object component that represents a pop-up dialog
 * </p>
 * <p>
 * Copyright 2017 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public class PopUpDialog extends AbstractVaadinPageComponent
{
	public static final String POPUP_XPATH = "//div[@class='popupContent']//";
	public static final String POPUP_BUTTON_YES_XPATH = POPUP_XPATH + "div[@id='cmdYes']";
	public static final String POPUP_BUTTON_OK_XPATH = POPUP_XPATH + "div[@id='cmdOK']";
	public static final String POPUP_FILE_INPUT_XPATH = POPUP_XPATH + "input[@type='file']";
	public static final String POPUP_BUTTON_UPLOAD_XPATH = POPUP_XPATH + "form[@id='upload']//div[@class='v-button']";

	private static final String IMG_DLG_INFO = "dialog_information.png";
	private static final String IMG_DLG_WARN = "dialog_warning.png";
	private static final String IMG_DLG_ERROR = "dialog_error.png";

	/**
	 * Constructor
	 * @param pageObject
	 */
	public PopUpDialog(AbstractPageObject pageObject)
	{
		super(pageObject.getTestContext());

		this.logger = pageObject.getLogger();
	}

	/**
	 * Search for a visible pop-up dialog and compare its status with the expected action result
	 * @param actionResult
	 * @throws AssertionError if the status validation either has failed, or an element could not be found
	 */
	public void validateStatus(PageActionResult actionResult)
	{
		logger.debug("Read information from '{}' pop-up dialog", actionResult.getStatus());

		final String statusExpr = POPUP_XPATH + "img[@id='imgStatus']";
		final WebElement statusIcon = findWebElementByXPath(statusExpr);
		final String statusIconSource = statusIcon.getAttribute("src");
		PageActionResult.ActionResultStatus currentStatus = null;

		// Initialize status by searching for the respective image
		if(statusIconSource.endsWith(IMG_DLG_INFO))
			currentStatus = PageActionResult.ActionResultStatus.INFO;
		else if(statusIconSource.endsWith(IMG_DLG_WARN))
			currentStatus = PageActionResult.ActionResultStatus.WARNING;
		else if(statusIconSource.endsWith(IMG_DLG_ERROR))
			currentStatus = PageActionResult.ActionResultStatus.ERROR;

		if(currentStatus == null)
			fail("Could not determine message status!");

		// Search for the message text
		final String msgExpr = POPUP_XPATH + "div[@id='lblMessage']";
		final WebElement divMsg = findWebElementByXPath(msgExpr);

		// Search for the text area that contains the detail message
		final String detailExpr = POPUP_XPATH + "textarea[@id='txtStackTrace']";
		final WebElement txtDetail = findWebElementByXPath(detailExpr);

		// Validate the dialog's status!
		assertEquals("Pop-up dialog has an unexpected status (" + currentStatus.name() + ")!", currentStatus, actionResult.getStatus());

		if(actionResult.getMessage() != null && !actionResult.getMessage().isEmpty())
		{
			final String detailMessage = txtDetail.getAttribute(ATTR_NAME_VALUE);
			String message = divMsg.getText();

			if(detailMessage != null && !detailMessage.isEmpty())
				message += " " + txtDetail.getAttribute(ATTR_NAME_VALUE);

			if(logger.isTraceEnabled())
				logger.trace("Pop-up dialog contains following message: '{}'", message.replace('\n', ' '));

			// Check if the expected message is available!
			assertTrue("Pop-up dialog doesn't contain expected message!", message.contains(actionResult.getMessage()));
		}
	}

	/**
	 * Close the dialog
	 * @throws AssertionError if the button for closing the dialog could not be found
	 */
	public void closeDialog()
	{
		logger.debug("Close pop-up dialog");

		// Click on the 'OK' button in order to close the dialog
		findWebElement("cmdOK").click();
	}

}
