/**
 * Copyright Openmind http://www.openmindonline.it
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sourceforge.openutils.mgnlcontrols.dialog;

import freemarker.template.TemplateException;
import info.magnolia.cms.core.Content;
import info.magnolia.cms.core.ItemType;
import info.magnolia.cms.core.NodeData;
import info.magnolia.cms.core.SystemProperty;
import info.magnolia.cms.gui.control.ControlImpl;
import info.magnolia.cms.gui.control.FreemarkerControl;
import info.magnolia.cms.gui.dialog.DialogBox;
import info.magnolia.cms.gui.dialog.DialogFreemarker;
import info.magnolia.cms.security.AccessDeniedException;
import info.magnolia.cms.util.NodeDataUtil;

import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.jcr.PathNotFoundException;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.exception.NestableRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * similar to DialogFreemarker but with a fixed template, needed till MAGNOLIA-2175 is done
 * @author fgiust
 * @version $Id: ConfigurableFreemarkerDialog.java 1197 2009-05-11 17:22:11Z fgiust $
 */
public abstract class ConfigurableFreemarkerDialog extends DialogBox
{

    /**
     * Logger.
     */
    private static Logger log = LoggerFactory.getLogger(DialogFreemarker.class);

    private Map configuration;

    private String uuid;

    /**
     * Get a recursive map view of a content node
     * @param node content node
     * @return recursive map view on content node properties and children
     * @throws PathNotFoundException
     * @throws RepositoryException
     * @throws AccessDeniedException
     */
    protected Map getSubNodes(Content node) throws RepositoryException, AccessDeniedException
    {

        this.uuid = node.getUUID();
        Map values = new LinkedHashMap();

        // cycles on properties and stores them in map
        Collection properties = node.getNodeDataCollection();

        if (properties != null && properties.size() > 0)
        {
            Iterator propertiesIt = properties.iterator();
            while (propertiesIt.hasNext())
            {
                NodeData property = (NodeData) propertiesIt.next();
                values.put(property.getName(), NodeDataUtil.getValueObject(property));
            }
        }

        // cycle on children
        Collection children = node.getChildren(ItemType.CONTENTNODE);
        if (children != null && children.size() > 0)
        {
            Iterator childrenIt = children.iterator();
            while (childrenIt.hasNext())
            {
                Content child = (Content) childrenIt.next();

                // gets sub map
                Map subValues = getSubNodes(child);
                // stores it in map
                values.put(child.getName(), subValues);
            }
        }

        return values;
    }

    /**
     * {@inheritDoc}
     */

    @Override
    public void init(HttpServletRequest request, HttpServletResponse response, Content websiteNode, Content configNode)
        throws RepositoryException
    {
        super.init(request, response, websiteNode, configNode);
        configuration = this.getSubNodes(configNode);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    @Override
    public void drawHtml(Writer out) throws IOException
    {

        String alreadyRenderedKey = "freemarker/" + getPath();

        Map parameters = new HashMap();
        parameters.put("name", this.getName());
        parameters.put("value", this.getValue());
        parameters.put("values", this.getValues());
        parameters.put("request", this.getRequest());
        parameters.put("debug", SystemProperty.getBooleanProperty("magnolia.develop"));
        parameters.put("configuration", this.configuration);
        parameters.put("uuid", this.uuid);
        parameters.put("alreadyrendered", getRequest().getAttribute(alreadyRenderedKey) != null);
        addToParameters(parameters);

        this.drawHtmlPre(out);

        try
        {
            FreemarkerControl control = new FreemarkerControl("multiple".equals(this.getConfigValue("valueType"))
                ? ControlImpl.VALUETYPE_MULTIPLE
                : ControlImpl.VALUETYPE_SINGLE);
            control.setType(this.getConfigValue("type", PropertyType.TYPENAME_STRING));
            control.setName(this.getName());
            control.drawHtml(out, getPath(), parameters);
        }
        catch (TemplateException ex)
        {
            log.error("Error processing dialog template:", ex);
            throw new NestableRuntimeException(ex);
        }

        this.drawHtmlPost(out);

        getRequest().setAttribute(alreadyRenderedKey, Boolean.TRUE);
    }

    protected void addToParameters(Map parameters)
    {
        // do nothing, can be user in subclasses
    }

    /**
     * @return ftl template path
     */
    protected abstract String getPath();

}
