/**
 * Copyright Openmind http://www.openmindonline.it
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package it.openutils.mgnltasks;

import info.magnolia.cms.core.HierarchyManager;
import info.magnolia.cms.module.ModuleUtil;
import info.magnolia.cms.module.RegisterException;
import info.magnolia.module.InstallContext;
import info.magnolia.module.delta.BootstrapResourcesTask;
import info.magnolia.module.delta.TaskExecutionException;

import java.io.IOException;

import javax.jcr.RepositoryException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * A task to bootstrap a module. Can be used to fully re-bootstrap a custom module after an update (useful to reload
 * templates, dialogs, etc). This task will always ignore bootstrap files for the <code>website</code> repository.
 * @author fgiust
 * @version $Id: ModuleConfigBootstrapTask.java 784 2008-04-27 20:42:00Z fgiust $
 */
public class ModuleConfigBootstrapTask extends BootstrapResourcesTask
{

    /**
     * Logger.
     */
    private Logger log = LoggerFactory.getLogger(ModuleConfigBootstrapTask.class);

    private String modulename;

    public ModuleConfigBootstrapTask(String modulename)
    {
        super("Bootstrap", "Bootstraps module configuration for " + modulename + " (will not overwrite website!).");
        this.modulename = modulename;
    }

    @Override
    protected boolean acceptResource(InstallContext ctx, String name)
    {

        boolean accept = name.startsWith("/mgnl-bootstrap/" + modulename + "/")
            && !name.startsWith("/mgnl-bootstrap/" + modulename + "/website")
            && name.endsWith(".xml");

        if (accept)
        {
            log.debug("Importing file {}", name);
        }
        return accept;
    }

    @Override
    public void execute(final InstallContext installContext) throws TaskExecutionException
    {
        try
        {
            final String[] resourcesToBootstrap = getResourcesToBootstrap(installContext);

            long millis = System.currentTimeMillis();
            log.info("FCOM bootstrap starting...");

            deleteNode(installContext, "/modules/" + modulename + "/dialogs");
            deleteNode(installContext, "/modules/" + modulename + "/templates");
            deleteNode(installContext, "/modules/" + modulename + "/paragraphs");
            deleteNode(installContext, "/modules/" + modulename + "/virtualURIMapping");

            ModuleUtil.bootstrap(resourcesToBootstrap, false);
            log.info("FCOM bootstrap done in {} seconds", (System.currentTimeMillis() - millis) / 1000);
        }
        catch (IOException e)
        {
            throw new TaskExecutionException("Could not bootstrap: " + e.getMessage());
        }
        catch (RegisterException e)
        {
            throw new TaskExecutionException("Could not bootstrap: " + e.getMessage());
        }
        catch (RepositoryException e)
        {
            throw new TaskExecutionException("Could not bootstrap: " + e.getMessage());
        }
    }

    protected void deleteNode(InstallContext installContext, String nodePath) throws RepositoryException,
        TaskExecutionException
    {

        HierarchyManager hm = installContext.getHierarchyManager("config");

        if (hm.isExist(nodePath))
        {
            hm.delete(nodePath);
        }
    }
}