package com.phantom.adapter

import android.util.Log
import com.connect.common.AddETHChainCallback
import com.connect.common.AddEthereumChainParameter
import com.connect.common.ConnectCallback
import com.connect.common.ConnectConfig
import com.connect.common.ConnectManager
import com.connect.common.DisconnectCallback
import com.connect.common.IConnectAdapter
import com.connect.common.RequestCallback
import com.connect.common.SignAllCallback
import com.connect.common.SignCallback
import com.connect.common.SwitchETHChainCallback
import com.connect.common.TransactionCallback
import com.connect.common.activeConnectSuccess
import com.connect.common.activeSignSuccess
import com.connect.common.eip4361.Eip4361Message
import com.connect.common.model.Account
import com.connect.common.model.ConnectError
import com.connect.common.model.WalletReadyState
import com.connect.common.utils.AppUtils
import com.particle.base.ParticleNetwork
import com.particle.base.iaa.FeeMode
import com.particle.base.model.ChainType
import com.particle.base.model.IconUrl
import com.particle.base.model.MobileWCWalletName
import com.particle.base.model.WalletName
import com.particle.base.model.WebsiteUrl
import com.particle.base.utils.Base58Utils
import com.phantom.adapter.model.AccountDao
import com.phantom.adapter.service.ConnectOutput
import com.phantom.adapter.service.DisConnectOutput
import com.phantom.adapter.service.PhantomConnectService
import com.phantom.adapter.service.PhantomServiceCallback
import com.phantom.adapter.service.PhantomServiceError
import com.phantom.adapter.service.SignAllTransactionsOutput
import com.phantom.adapter.service.SignAndSendTransactionOutput
import com.phantom.adapter.service.SignMessageOutput
import com.phantom.adapter.service.SignTransactionOutput
import network.particle.chains.ChainInfo
import org.p2p.solanaj.core.ITransactionData
import org.p2p.solanaj.core.SolanaTransaction
import org.p2p.solanaj.core.Transaction
import org.p2p.solanaj.utils.TweetNaclFast
import org.p2p.solanaj.utils.crypto.Base64Utils

class PhantomConnectAdapter : IConnectAdapter {

    override val name: WalletName = MobileWCWalletName.Phantom.name

    override val icon: IconUrl = "https://static.particle.network/wallet-icons/Phantom.png"

    override val url: WebsiteUrl = "https://phantom.app"

    override val supportChains: List<ChainType> = listOf(ChainType.Solana)

    override val readyState: WalletReadyState
        get() {
            return if (AppUtils.isAppInstalled(
                    ConnectManager.context,
                    "app.phantom"
                )
            ) WalletReadyState.Installed else WalletReadyState.NotDetected
        }

    override fun getAccounts(): List<Account> {
        return AccountDao.getAdapterAccount()
            .map { Account(it.publicAddress, name, url, null, listOf(icon)) }
    }

    override fun <T : ConnectConfig> connect(config: T?, callback: ConnectCallback) {
        PhantomConnectService.connect(object : PhantomServiceCallback<ConnectOutput> {
            override fun success(output: ConnectOutput) {
                activeConnectSuccess(this@PhantomConnectAdapter, output.publicKey)
                callback.onConnected(Account(output.publicKey, name, url, null, listOf(icon)))
            }

            override fun failure(errMsg: PhantomServiceError) {
                callback.onError(ConnectError.Custom(errMsg.code, errMsg.message))
            }
        })
    }

    override fun disconnect(publicAddress: String, callback: DisconnectCallback) {
        PhantomConnectService.disconnect(publicAddress,
            object : PhantomServiceCallback<DisConnectOutput> {
                override fun success(output: DisConnectOutput) {
                    callback.onDisconnected()
                }

                override fun failure(errMsg: PhantomServiceError) {
                    callback.onError(ConnectError.Custom(errMsg.code, errMsg.message))
                }

            })
    }

    override fun connected(publicAddress: String): Boolean {
        return AccountDao.getAccount(publicAddress) != null
    }

    override fun login(publicAddress: String, message: Eip4361Message, callback: SignCallback) {
        signMessage(publicAddress, message.toBase58(), callback)
    }

    override fun verify(
        publicAddress: String,
        signedMessage: String,
        originalMessage: String
    ): Boolean {
        val account = AccountDao.getAccount(publicAddress) ?: return false
        val signatureProvider =
            TweetNaclFast.Signature(
                Base58Utils.decode(account.address),
                ByteArray(0)
            )
        val base58 = Base64Utils.decodeToString(signedMessage)
        val sign = Base58Utils.decode(base58)
        return signatureProvider.detached_verify(originalMessage.toByteArray(), sign)
    }


    override fun request(
        publicAddress: String,
        method: String,
        params: List<Any>?,
        callback: RequestCallback
    ) {
        callback.onError(ConnectError.UnSupportMethod())
    }

    override fun signAndSendTransaction(
        publicAddress: String,
        transaction: ITransactionData,
        callback: TransactionCallback, chainId: Long?
    ) {
        if (!supportChains.contains(ConnectManager.chainType)) {
            callback.onError(ConnectError.UnSupportChain())
            return
        }

        if (!connected(publicAddress)) {
            callback.onError(ConnectError.Unauthorized())
            return
        }

        if (transaction !is SolanaTransaction) {
            callback.onError(ConnectError.Params())
            return
        }

        PhantomConnectService.signAndSendTransaction(publicAddress, transaction.encode(),
            object : PhantomServiceCallback<SignAndSendTransactionOutput> {
                override fun success(output: SignAndSendTransactionOutput) {
                    activeSignSuccess(this@PhantomConnectAdapter, publicAddress)
                    callback.onTransaction(Base64Utils.encode(Base58Utils.decode(output.signature)))
                }

                override fun failure(errMsg: PhantomServiceError) {
                    callback.onError(ConnectError.Custom(errMsg.code, errMsg.message))
                }
            })
    }

    override fun signAndSendTransaction(
        publicAddress: String,
        transaction: String,
        callback: TransactionCallback, chainId: Long?
    ) {
        signAndSendTransaction(
            publicAddress,
            Transaction.from(Base58Utils.decode(transaction)),
            callback
        )
    }

    override fun signAndSendTransaction(
        publicAddress: String,
        transaction: String,
        feeMode: FeeMode,
        callback: TransactionCallback,
        chainId: Long?
    ) {
        throw UnsupportedOperationException("Not supported")
    }


    override fun signTransaction(
        publicAddress: String,
        transaction: ITransactionData,
        callback: SignCallback, chainId: Long?
    ) {
        if (!supportChains.contains(ConnectManager.chainType)) {
            callback.onError(ConnectError.UnSupportChain())
            return
        }

        if (!connected(publicAddress)) {
            callback.onError(ConnectError.Unauthorized())
            return
        }

        if (transaction !is SolanaTransaction) {
            callback.onError(ConnectError.Params())
            return
        }

        PhantomConnectService.signTransaction(publicAddress, transaction.encode(),
            object : PhantomServiceCallback<SignTransactionOutput> {
                override fun success(output: SignTransactionOutput) {
                    activeSignSuccess(this@PhantomConnectAdapter, publicAddress)
                    callback.onSigned(Base64Utils.encode(Base58Utils.decode(output.transaction)))
                }

                override fun failure(errMsg: PhantomServiceError) {
                    callback.onError(ConnectError.Custom(errMsg.code, errMsg.message))
                }

            })
    }

    override fun signTransaction(
        publicAddress: String,
        transaction: String,
        callback: SignCallback, chainId: Long?
    ) {
        signTransaction(
            publicAddress,
            Transaction.from(Base58Utils.decode(transaction)),
            callback
        )
    }

    override fun signAllTransactions(
        publicAddress: String,
        transactions: List<ITransactionData>,
        callback: SignAllCallback, chainId: Long?
    ) {
        if (!supportChains.contains(ConnectManager.chainType)) {
            callback.onError(ConnectError.UnSupportChain())
            return
        }

        if (!connected(publicAddress)) {
            callback.onError(ConnectError.Unauthorized())
            return
        }

        if (transactions.isEmpty()) {
            callback.onError(ConnectError.Params())
            return
        }

        if (transactions.any { it !is SolanaTransaction }) {
            callback.onError(ConnectError.Params())
            return
        }

        PhantomConnectService.signAllTransaction(publicAddress, transactions.map { it.encode() },
            object : PhantomServiceCallback<SignAllTransactionsOutput> {
                override fun success(output: SignAllTransactionsOutput) {
                    activeSignSuccess(this@PhantomConnectAdapter, publicAddress)
                    callback.onSigned(output.transactions.map {
                        Base64Utils.encode(
                            Base58Utils.decode(
                                it
                            )
                        )
                    })
                }

                override fun failure(errMsg: PhantomServiceError) {
                    callback.onError(ConnectError.Custom(errMsg.code, errMsg.message))
                }
            })
    }

    override fun signAllTransactions(
        publicAddress: String,
        transactions: Array<String>,
        callback: SignAllCallback, chainId: Long?
    ) {
        signAllTransactions(
            publicAddress,
            transactions.map { Transaction.from(Base58Utils.decode(it)) },
            callback
        )
    }

    override fun signMessage(
        publicAddress: String,
        message: String,
        callback: SignCallback,
        chainId: Long?
    ) {
        if (!supportChains.contains(ConnectManager.chainType)) {
            callback.onError(ConnectError.UnSupportChain())
            return
        }

        if (!connected(publicAddress)) {
            callback.onError(ConnectError.Unauthorized())
            return
        }

        PhantomConnectService.signMessage(publicAddress, message,
            object : PhantomServiceCallback<SignMessageOutput> {
                override fun success(output: SignMessageOutput) {
                    val sign58 = output.signature
                    val sign64 = Base64Utils.encode(Base58Utils.decode(sign58))
                    activeSignSuccess(this@PhantomConnectAdapter, publicAddress)
                    callback.onSigned(sign64)
                }

                override fun failure(errMsg: PhantomServiceError) {
                    callback.onError(ConnectError.Custom(errMsg.code, errMsg.message))
                }

            })
    }

    override fun signTypedData(
        publicAddress: String,
        data: String,
        callback: SignCallback,
        chainId: Long?
    ) {
        callback.onError(ConnectError.UnSupportMethod())
    }

}


