package com.phantom.adapter.service

import android.content.Intent
import android.net.Uri
import android.util.Log
import androidx.annotation.Keep
import com.connect.common.ConnectManager
import com.connect.common.ConnectManager.getRedirectUrl
import com.particle.base.utils.Base58Utils
import com.particle.base.utils.GsonUtils
import com.phantom.adapter.model.AccountDao
import com.phantom.adapter.model.AccountInfo
import com.phantom.adapter.model.TransactionsResp
import com.phantom.adapter.tools.TweetNacl
import org.json.JSONObject

@Keep
open class PhantomOutput

@Keep
data class ConnectOutput(
    var publicKey: String = "",
    var error: ErrorInfo? = null
) : PhantomOutput()

@Keep
data class DisConnectOutput(
    var result: Boolean
) : PhantomOutput()

@Keep
data class SignMessageOutput(
    var signature: String
) : PhantomOutput()

@Keep
data class SignTransactionOutput(
    var transaction: String
) : PhantomOutput()

@Keep
data class SignAllTransactionsOutput(
    var transactions: List<String>
) : PhantomOutput()

@Keep
data class SignAndSendTransactionOutput(
    var signature: String
) : PhantomOutput()

@Keep
data class ErrorInfo(
    val code: Int,
    val message: String
)

@Keep
enum class PhantomAction {
    connect,
    disconnect,
    signAndSendTransaction,
    signAllTransactions,
    signTransaction,
    signMessage
}

@Keep
data class PhantomServiceError(val message: String, val code: Int)

@Keep
interface PhantomServiceCallback<T : PhantomOutput> {
    fun success(output: T)

    fun failure(errMsg: PhantomServiceError)
}

@Keep
object PhantomConnectService {
    private val baseUrl = "https://phantom.app/ul/v1"
    private var phantomCallback: PhantomServiceCallback<PhantomOutput>? = null
    private fun buildUri(
        path: String, publicKey: String
    ): Uri {
        val builder = Uri.parse("$baseUrl/$path").buildUpon()
            .appendQueryParameter("dapp_encryption_public_key", publicKey)
            .appendQueryParameter("redirect_link", "${getRedirectUrl()}$path")
        return builder.build()
    }

    private var connectAccountInfo: AccountInfo? = null
    var logicAccount: AccountInfo? = null

    fun connect(phantomCallback: PhantomServiceCallback<ConnectOutput>) {
        PhantomConnectService.phantomCallback =
            phantomCallback as PhantomServiceCallback<PhantomOutput>
        val keyPair = TweetNacl.generateKeyPair()
        connectAccountInfo = AccountInfo(keyPair.publicKey, keyPair.secretKey)
        val uri = buildUri(PhantomAction.connect.name, keyPair.publicKey)
        var cluster = "mainnet-beta"
        if (ConnectManager.chainId == 102L) {
            cluster = "testnet"
        } else if (ConnectManager.chainId == 103L) {
            cluster = "devnet"
        }
        if(ConnectManager.dAppData == null){
            throw RuntimeException("DAppMetadata is null, please init with DAppMetadata")
        }
        val newURI = uri.buildUpon().appendQueryParameter("app_url", ConnectManager.dAppData?.url)
            .appendQueryParameter("cluster", cluster).build()
        startIntentWithURI(newURI)
    }

    fun disconnect(
        publicAddress: String,
        phantomCallback: PhantomServiceCallback<DisConnectOutput>
    ) {
        logicAccount = AccountDao.getAccount(publicAddress)
        if (logicAccount == null) {
            phantomCallback.failure(PhantomServiceError("Account not found", -1))
            return
        }
        AccountDao.delAccount(publicAddress)
        phantomCallback?.success(DisConnectOutput(true))
        /*
        val nonce = TweetNacl.getNonce()
        val box = TweetNacl.getSharedBox(logicAccount!!.otherPublicKey, logicAccount!!.secretKey)

        val map = HashMap<String, String>()
        map["session"] = logicAccount!!.session

        val payload =
            box.box(GsonUtils.toJson(map).toByteArray(Charsets.UTF_8), Base58Utils.decode(nonce))

        val uriBuilder = buildUri(PhantomAction.disconnect.name, logicAccount!!.publicKey)
        val newURI = uriBuilder.buildUpon().appendQueryParameter("nonce", nonce)
            .appendQueryParameter("payload", Base58Utils.encode(payload)).build()

        startIntentWithURI(newURI)*/
    }

    fun signMessage(
        publicAddress: String,
        message: String,
        phantomCallback: PhantomServiceCallback<SignMessageOutput>
    ) {
        logicAccount = AccountDao.getAccount(publicAddress)
        if (logicAccount == null) {
            phantomCallback.failure(PhantomServiceError("Account not found", -1))
            return
        }
        PhantomConnectService.phantomCallback =
            phantomCallback as PhantomServiceCallback<PhantomOutput>
        val box = TweetNacl.getSharedBox(logicAccount!!.otherPublicKey, logicAccount!!.secretKey)
        val nonce = TweetNacl.getNonce()

        val map = HashMap<String, String>()
        map["session"] = logicAccount!!.session
        map["message"] = message
        val json = GsonUtils.toJson(map)

        val payload = box.box(json.toByteArray(Charsets.UTF_8), Base58Utils.decode(nonce))

        val uriBuilder = buildUri(PhantomAction.signMessage.name, logicAccount!!.publicKey)
        val newURI = uriBuilder.buildUpon()
            .appendQueryParameter("nonce", nonce)
            .appendQueryParameter("payload", Base58Utils.encode(payload))
            .build()
        startIntentWithURI(newURI)
    }

    fun signTransaction(
        publicAddress: String,
        transaction: String,
        phantomCallback: PhantomServiceCallback<SignTransactionOutput>
    ) {

        logicAccount = AccountDao.getAccount(publicAddress)
        if (logicAccount == null) {
            phantomCallback?.failure(PhantomServiceError("Account not found", -1))
            return
        }
        PhantomConnectService.phantomCallback =
            phantomCallback as PhantomServiceCallback<PhantomOutput>
        val box = TweetNacl.getSharedBox(logicAccount!!.otherPublicKey, logicAccount!!.secretKey)
        val nonce = TweetNacl.getNonce()

        val map = HashMap<String, String>()
        map["session"] = logicAccount!!.session
        map["transaction"] = transaction
        val payload =
            box.box(GsonUtils.toJson(map).toByteArray(Charsets.UTF_8), Base58Utils.decode(nonce))

        val uriBuilder = buildUri(PhantomAction.signTransaction.name, logicAccount!!.publicKey)
        val newURI = uriBuilder.buildUpon()
            .appendQueryParameter("nonce", nonce)
            .appendQueryParameter("payload", Base58Utils.encode(payload))
            .build()
        startIntentWithURI(newURI)
    }

    fun signAllTransaction(
        publicAddress: String,
        transaction: List<String>,
        phantomCallback: PhantomServiceCallback<SignAllTransactionsOutput>
    ) {

        logicAccount = AccountDao.getAccount(publicAddress)
        if (logicAccount == null) {
            phantomCallback?.failure(PhantomServiceError("Account not found", -1))
            return
        }
        PhantomConnectService.phantomCallback =
            phantomCallback as PhantomServiceCallback<PhantomOutput>
        val box = TweetNacl.getSharedBox(logicAccount!!.otherPublicKey, logicAccount!!.secretKey)
        val nonce = TweetNacl.getNonce()

        val map = HashMap<String, String>()
        map["session"] = logicAccount!!.session
        map["transaction"] = GsonUtils.toJson(transaction)
        val payload =
            box.box(GsonUtils.toJson(map).toByteArray(Charsets.UTF_8), Base58Utils.decode(nonce))

        val uriBuilder = buildUri(PhantomAction.signAllTransactions.name, logicAccount!!.publicKey)
        val newURI = uriBuilder.buildUpon()
            .appendQueryParameter("nonce", nonce)
            .appendQueryParameter("payload", Base58Utils.encode(payload))
            .build()
        startIntentWithURI(newURI)
    }

    fun signAndSendTransaction(
        publicAddress: String,
        transaction: String,
        phantomCallback: PhantomServiceCallback<SignAndSendTransactionOutput>
    ) {
        logicAccount = AccountDao.getAccount(publicAddress)
        if (logicAccount == null) {
            phantomCallback?.failure(PhantomServiceError("Account not found", -1))
            return
        }
        PhantomConnectService.phantomCallback =
            phantomCallback as PhantomServiceCallback<PhantomOutput>
        val box = TweetNacl.getSharedBox(logicAccount!!.otherPublicKey, logicAccount!!.secretKey)
        val nonce = TweetNacl.getNonce()

        val map = HashMap<String, String>()
        map["session"] = logicAccount!!.session
        map["transaction"] = transaction
        val json = GsonUtils.toJson(map)
        val payload =
            box.box(json.toByteArray(Charsets.UTF_8), Base58Utils.decode(nonce))

        val uriBuilder =
            buildUri(PhantomAction.signAndSendTransaction.name, logicAccount!!.publicKey)
        val newURI = uriBuilder.buildUpon()
            .appendQueryParameter("nonce", nonce)
            .appendQueryParameter("payload", Base58Utils.encode(payload))
            .build()
        startIntentWithURI(newURI)
    }

    private fun startIntentWithURI(uri: Uri) {
        Log.d("PhantomConnectService", "startIntentWithURI: $uri")
        ConnectManager.context.startActivity(Intent().apply {
            action = Intent.ACTION_VIEW
            addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            data = uri
        })
        ConnectManager.handleRedirectResult {
            setResultUri(it)
        }
    }

    private fun setResultUri(uri: Uri?) {
        try {
            val errorCode = uri?.getQueryParameter("errorCode")
            val errorMessage = uri?.getQueryParameter("errorMessage")
            if (errorCode != null) {
                if (errorCode.isNotEmpty()) {
                    phantomCallback?.failure(
                        PhantomServiceError(
                            errorMessage as String,
                            errorCode.toInt()
                        )
                    )
                    return
                }
            }

            val strUrl = uri.toString()
            val direct = strUrl.substring(strUrl.lastIndexOf("/") + 1, strUrl.lastIndexOf("?"))
            when (direct) {
                PhantomAction.connect.name -> {
                    val secretKey = connectAccountInfo!!.secretKey

                    val nonce = uri?.getQueryParameter("nonce") as String
                    val source = uri?.getQueryParameter("data") as String
                    val otherPublicKey =
                        uri?.getQueryParameter("phantom_encryption_public_key") as String
                    val decrypted = TweetNacl.getBox(otherPublicKey, secretKey)

                    val sourceByte =
                        decrypted.open(Base58Utils.decode(source), Base58Utils.decode(nonce))
                    val json = JSONObject(String(sourceByte, Charsets.UTF_8))
                    val address = json.getString("public_key")
                    val session = json.getString("session")
                    connectAccountInfo!!.connected(address, session, otherPublicKey)
                    AccountDao.addAccount(connectAccountInfo!!)
                    phantomCallback?.success(ConnectOutput(address))
                }
                PhantomAction.disconnect.name -> {
                    phantomCallback?.success(DisConnectOutput(true))
                    AccountDao.delAccount(logicAccount!!.publicKey)
                }
                PhantomAction.signMessage.name -> {
                    val nonce = uri?.getQueryParameter("nonce") as String
                    val source = uri?.getQueryParameter("data") as String
                    val strData = String(
                        TweetNacl.getSharedBox(
                            logicAccount!!.otherPublicKey,
                            logicAccount!!.secretKey
                        )
                            .open(Base58Utils.decode(source), Base58Utils.decode(nonce))
                    )
                    val signature = JSONObject(strData).getString("signature")
                    phantomCallback?.success(SignMessageOutput(signature))
                }
                PhantomAction.signTransaction.name -> {
                    val nonce = uri?.getQueryParameter("nonce") as String
                    val source = uri?.getQueryParameter("data") as String
                    val strData = String(
                        TweetNacl.getSharedBox(
                            logicAccount!!.otherPublicKey,
                            logicAccount!!.secretKey
                        )
                            .open(Base58Utils.decode(source), Base58Utils.decode(nonce))
                    )
                    val transactions = JSONObject(strData).getString("transaction")
                    phantomCallback?.success(SignTransactionOutput(transactions))
                }
                PhantomAction.signAllTransactions.name -> {
                    val nonce = uri?.getQueryParameter("nonce") as String
                    val source = uri?.getQueryParameter("data") as String
                    val strData = String(
                        TweetNacl.getSharedBox(
                            logicAccount!!.otherPublicKey,
                            logicAccount!!.secretKey
                        )
                            .open(Base58Utils.decode(source), Base58Utils.decode(nonce))
                    )
                    val result = GsonUtils.fromJson(strData, TransactionsResp::class.java)
                    phantomCallback?.success(SignAllTransactionsOutput(result.transactions))
                }
                PhantomAction.signAndSendTransaction.name -> {
                    val nonce = uri?.getQueryParameter("nonce") as String
                    val source = uri?.getQueryParameter("data") as String
                    val strData = String(
                        TweetNacl.getSharedBox(
                            logicAccount!!.otherPublicKey,
                            logicAccount!!.secretKey
                        )
                            .open(Base58Utils.decode(source), Base58Utils.decode(nonce))
                    )
                    val jsonData = JSONObject(strData)
                    val signature = jsonData.getString("signature")
                    phantomCallback?.success(SignAndSendTransactionOutput(signature))
                }
                else -> {

                }
            }
            phantomCallback = null
        } catch (e: Exception) {
            e.printStackTrace()
            phantomCallback?.failure(
                PhantomServiceError(
                    e.message ?: "Unknown error", 100000
                )
            )
        }
    }


}
