package com.particle.connect

import android.app.Application
import android.content.Context
import android.util.Log
import androidx.annotation.Keep
import com.blankj.utilcode.util.LogUtils
import com.connect.common.ConnectManager
import com.connect.common.IConnectAdapter
import com.connect.common.IParticleConnectAdapter
import com.connect.common.model.Chain
import com.particle.base.Env
import com.particle.base.ParticleNetwork
import com.particle.base.analytics.AnalyticsAction
import com.particle.base.analytics.AnalyticsLoginType
import com.particle.base.analytics.AnalyticsService
import com.particle.base.analytics.value
import com.particle.base.model.AnalyticsBody
import com.particle.base.model.ChainType
import com.particle.base.model.DAppMetadata
import com.particle.base.model.MobileWCWalletName
import com.particle.base.utils.ClassUtil
import com.particle.base.utils.UserRepo
import com.particle.connect.model.AdapterAccount
import com.wallet.connect.WalletConnectAdapterManager
import network.particle.chains.ChainInfo

/**
 * Created by chaichuanfa on 2022/7/12
 */
@Keep
object ParticleConnect {
    private var createAdapters: (() -> List<IConnectAdapter>)? = null

    private val adapters: List<IConnectAdapter> by lazy {
        lazyCreateAdapters()
    }

    private var sdkInitialized = false

    val chainId: Long
        get() = ConnectManager.chainId
    val chainInfo: ChainInfo
        get() = ConnectManager.chainInfo

    val chainType: ChainType
        get() = ConnectManager.chainType

    /**
     * @param context: application context.
     * @param env
     * @param chain: support solana and evm chains.
     * [ChainInfo]
     */
    @JvmStatic
    fun init(
        application: Application,
        env: Env,
        chain: ChainInfo,
        dAppData: DAppMetadata?,
        createAdapters: (() -> List<IConnectAdapter>)? = null,
    ) {
        if (sdkInitialized) return
        ParticleNetwork.init(application, env, chain)
        ConnectManager.init(
            application,
            chain,
            Chain(chain.id, getChainTypeByName(chain.name)),
            dAppData,
            env == Env.DEV
        )
        this.createAdapters = createAdapters

        logInit(chain)

        try {
            if (dAppData != null && ClassUtil.isClassExists("com.wallet.connect.WalletConnectAdapterManager")) {
                WalletConnectAdapterManager.init(application, dAppData)
            }
        } catch (e: Exception) {
            LogUtils.e("ParticleConnect", "init WalletConnect error" + e.message)
        }
    }

    @JvmStatic
    fun init(
        application: Context,
        env: Env,
        chain: ChainInfo,
        createAdapters: (() -> List<IConnectAdapter>)? = null,
    ) {
        if (sdkInitialized) return
        ParticleNetwork.init(application, env, chain)
        ConnectManager.init(
            application,
            chain,
            Chain(chain.id, getChainTypeByName(chain.name)),
            null,
            env == Env.DEV
        )
        this.createAdapters = createAdapters

        logInit(chain)
    }

    @JvmStatic
    fun setWalletConnectV2SupportChainInfos(chains: List<ChainInfo>) {
        ConnectManager.setWalletConnectV2SupportChainInfos(chains)
    }


    private fun logInit(chain: ChainInfo) {
        try {
            val adapterAccounts =
                getAccounts(if (chain.isEvmChain()) ChainType.EVM else ChainType.Solana).filter { it.accounts.isNotEmpty() }
            if (adapterAccounts.isNotEmpty()) {
                val address = adapterAccounts[0].accounts[0].publicAddress
                val adapter = adapterAccounts[0].connectAdapter
                val userInfo = if (adapter is IParticleConnectAdapter) {
                    UserRepo.getLogUserInfo()
                } else {
                    null
                }
                AnalyticsService.active(
                    AnalyticsBody(
                        AnalyticsLoginType.activeLoginType(adapter.name).value(),
                        chain.id,
                        address,
                        address,
                        AnalyticsAction.OPEN.value(),
                        userInfo
                    )
                )
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }


    private fun lazyCreateAdapters(): List<IConnectAdapter> {
        val connectAdapters = mutableListOf<IConnectAdapter>()
        val addAdapters = createAdapters?.invoke()
        if (addAdapters.isNullOrEmpty()) {
            throw RuntimeException("adapters can not be empty")
        } else {
            connectAdapters.addAll(addAdapters)
        }
        return connectAdapters
    }

    private fun getChainTypeByName(name: String): ChainType {
        return if (name.equals("solana", true)) ChainType.Solana else ChainType.EVM
    }

    @JvmStatic
    fun setChain(chain: ChainInfo) {
        ConnectManager.setChain(
            Chain(
                chain.id, getChainTypeByName(chain.name)
            )
        )
        ConnectManager.setChainInfo(chain)
        ParticleNetwork.setChainInfo(chain)
    }

    @JvmStatic
    fun getAdapterByAddress(publicAddress: String): List<IConnectAdapter> {
        val adapters = getAdapters()
        return adapters.filter {
            val accounts = it.getAccounts()
            accounts.any { account -> account.publicAddress.equals(publicAddress, true) }
        }
    }

    @JvmStatic
    fun getAdapters(vararg chainTypes: ChainType): List<IConnectAdapter> {
        return if (chainTypes.isNullOrEmpty()) {
            mutableListOf<IConnectAdapter>().apply {
                addAll(adapters)
            }
        } else {
            val supportChains = chainTypes.toList()
            adapters.filter { it.supportChains.containsAll(supportChains) }
        }
    }

    @JvmStatic
    fun getAccounts(vararg chainTypes: ChainType): List<AdapterAccount> {
        val adapters = getAdapters(*chainTypes)
        return adapters.map { AdapterAccount(it, it.getAccounts()) }
    }

    @JvmStatic
    fun isAdapterRegister(walletName: MobileWCWalletName): Boolean {
        return getAdapters().any { it.name == walletName.name }
    }

    @JvmStatic
    fun isAdapterRegister(adapter: IConnectAdapter): Boolean {
        return getAdapters().any { it.name == adapter.name }
    }


}