/*
 * Decompiled with CFR 0.152.
 */
package no.digipost.security.cert;

import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.cert.CertPath;
import java.security.cert.CertPathBuilder;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertSelector;
import java.security.cert.CertStore;
import java.security.cert.CertificateFactory;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.security.auth.x500.X500Principal;
import no.digipost.security.DigipostSecurity;
import no.digipost.security.DigipostSecurityException;
import no.digipost.security.cert.CertHelper;
import no.digipost.security.cert.ReviewedCertPath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Trust {
    private static final Logger LOG = LoggerFactory.getLogger(Trust.class);
    private final Set<X509Certificate> trustedCerts;
    private final Map<X500Principal, List<X509Certificate>> trustedIntermediateCerts;

    public Trust(Stream<X509Certificate> rootCertificates, Stream<X509Certificate> intermediateCertificates) {
        this.trustedCerts = Collections.unmodifiableSet(rootCertificates.collect(Collectors.toSet()));
        this.trustedIntermediateCerts = Collections.unmodifiableMap(intermediateCertificates.collect(Collectors.groupingBy(X509Certificate::getSubjectX500Principal)));
    }

    public ReviewedCertPath resolveCertPath(X509Certificate certificate) {
        try {
            CollectionCertStoreParameters certStoreParams = new CollectionCertStoreParameters(this.getTrustAnchorsAndAnyIntermediateCertificatesFor(certificate.getIssuerX500Principal()).collect(Collectors.toSet()));
            CertStore certStore = CertStore.getInstance("Collection", certStoreParams);
            X509CertSelector certSelector = new X509CertSelector();
            certSelector.setCertificate(certificate);
            certSelector.setSubject(certificate.getSubjectX500Principal());
            PKIXBuilderParameters params = new PKIXBuilderParameters(this.getTrustAnchors(), (CertSelector)certSelector);
            params.addCertStore(certStore);
            params.setSigProvider("BC");
            params.setRevocationEnabled(false);
            CertPath certpath = CertPathBuilder.getInstance("PKIX").build(params).getCertPath();
            if (certpath.getCertificates().size() > 1) {
                return new ReviewedCertPath(certpath, this::trusts);
            }
            CertificateFactory cf = DigipostSecurity.getX509CertificateFactory();
            Optional<X509Certificate> issuer = CertHelper.findTrustAnchorCert(certificate, this.getTrustAnchors());
            return new ReviewedCertPath(cf.generateCertPath(Stream.concat(Stream.of(certificate), issuer.map(Stream::of).orElse(Stream.empty())).collect(Collectors.toList())), path -> issuer.isPresent());
        }
        catch (GeneralSecurityException e) {
            LOG.warn("Error generating cert path. Certificate {} is not issued by trusted issuer. {}: {}", new Object[]{DigipostSecurity.describe(certificate), e.getClass().getSimpleName(), e.getMessage()});
            if (LOG.isDebugEnabled()) {
                LOG.debug(e.getClass().getSimpleName() + ": '" + e.getMessage() + "'", (Throwable)e);
            }
            return new ReviewedCertPath(e);
        }
    }

    public boolean trusts(CertPath certPath) {
        try {
            Set<TrustAnchor> trustAnchors = this.getTrustAnchors();
            PKIXParameters params = new PKIXParameters(trustAnchors);
            params.setSigProvider("BC");
            params.setRevocationEnabled(false);
            CertPathValidator.getInstance("PKIX").validate(certPath, params);
            return true;
        }
        catch (CertPathValidatorException e) {
            return false;
        }
        catch (GeneralSecurityException e) {
            throw new DigipostSecurityException(e);
        }
    }

    public Set<TrustAnchor> getTrustAnchors() {
        return this.getTrustAnchorCertificates().stream().map(c -> new TrustAnchor((X509Certificate)c, null)).collect(Collectors.toSet());
    }

    public Set<X509Certificate> getTrustAnchorCertificates() {
        return this.trustedCerts;
    }

    public KeyStore getTrustAnchorsKeyStore() {
        return DigipostSecurity.asKeyStore(this.getTrustAnchorCertificates());
    }

    public Map<X500Principal, List<X509Certificate>> getTrustedIntermediateCertificates() {
        return this.trustedIntermediateCerts;
    }

    Stream<X509Certificate> getTrustAnchorsAndAnyIntermediateCertificatesFor(X500Principal principal) {
        return Stream.concat(this.getTrustAnchorCertificates().stream(), this.getTrustedIntermediateCertificates().getOrDefault(principal, Collections.emptyList()).stream());
    }
}

