/*
 * Decompiled with CFR 0.152.
 */
package no.digipost.signature.client.core.internal.security;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import no.digipost.signature.client.core.exceptions.ConfigurationException;
import no.digipost.signature.client.core.internal.security.ProvidesCertificateResourcePaths;

public class TrustStoreLoader {
    private static final AtomicInteger aliasSequence = new AtomicInteger();

    public static KeyStore build(ProvidesCertificateResourcePaths trustedCertificates) {
        try {
            KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
            trustStore.load(null, null);
            for (String certificateFolder : trustedCertificates.certificatePaths()) {
                TrustStoreLoader.loadCertificatesInto(certificateFolder, trustStore);
            }
            return trustStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new ConfigurationException("Unable to load certificates into truststore", e);
        }
    }

    private static void loadCertificatesInto(String certificateLocation, KeyStore trustStore) {
        ResourceLoader certificateLoader = certificateLocation.startsWith("classpath:") ? new ClassPathResourceLoader(certificateLocation) : new FileLoader(certificateLocation);
        certificateLoader.forEachFile((fileName, contents) -> {
            X509Certificate ca = (X509Certificate)CertificateFactory.getInstance("X.509").generateCertificate(contents);
            trustStore.setCertificateEntry(fileName, ca);
        });
        try {
            TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            tmf.init(trustStore);
            SSLContext context = SSLContext.getInstance("TLS");
            context.init(null, tmf.getTrustManagers(), null);
        }
        catch (KeyManagementException | KeyStoreException | NoSuchAlgorithmException e) {
            throw new ConfigurationException("Error initializing SSLContext for certification location " + certificateLocation, e);
        }
    }

    static String generateAlias(Path location) {
        return TrustStoreLoader.generateAlias(location.toString());
    }

    static String generateAlias(String resourceName) {
        if (resourceName == null || resourceName.trim().isEmpty()) {
            return "certificate-alias-" + aliasSequence.getAndIncrement();
        }
        String[] splitOnSlashes = resourceName.split("/");
        int size = splitOnSlashes.length;
        if (size == 1) {
            return splitOnSlashes[0];
        }
        return splitOnSlashes[size - 2] + ":" + splitOnSlashes[size - 1];
    }

    private static class ClassPathResourceLoader
    implements ResourceLoader {
        static final String CLASSPATH_PATH_PREFIX = "classpath:";
        private final String resourceName;

        ClassPathResourceLoader(String resourceName) {
            this.resourceName = resourceName.replaceFirst(CLASSPATH_PATH_PREFIX, "");
        }

        @Override
        public void forEachFile(ForFile forEachFile) {
            URL resourceUrl = TrustStoreLoader.class.getResource(this.resourceName);
            if (resourceUrl == null) {
                throw new ConfigurationException(this.resourceName + " not found on classpath");
            }
            try (InputStream inputStream = resourceUrl.openStream();){
                forEachFile.call(TrustStoreLoader.generateAlias(this.resourceName), inputStream);
            }
            catch (Exception e) {
                throw new ConfigurationException("Unable to load certificate from classpath: " + this.resourceName, e);
            }
        }
    }

    private static class FileLoader
    implements ResourceLoader {
        private final Path path;

        FileLoader(String certificateFolder) {
            this.path = Paths.get(certificateFolder, new String[0]);
        }

        @Override
        public void forEachFile(ForFile forEachFile) {
            if (!Files.isDirectory(this.path, new LinkOption[0])) {
                throw new ConfigurationException("Certificate path '" + this.path + "' is not a directory. It should point to a directory containing certificates.");
            }
            try (Stream<Path> files = Files.list(this.path);){
                files.forEach(file -> {
                    try (InputStream contents = Files.newInputStream(file, new OpenOption[0]);){
                        forEachFile.call(TrustStoreLoader.generateAlias(file), contents);
                    }
                    catch (Exception e) {
                        throw new ConfigurationException("Unable to load certificate from file " + file, e);
                    }
                });
            }
            catch (IOException e) {
                throw new ConfigurationException("Error reading certificates from " + this.path, e);
            }
        }
    }

    @FunctionalInterface
    private static interface ForFile {
        public void call(String var1, InputStream var2) throws IOException, GeneralSecurityException;
    }

    private static interface ResourceLoader {
        public void forEachFile(ForFile var1);
    }
}

