/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils;

import com.google.common.collect.AbstractIterator;
import com.google.common.collect.PeekingIterator;
import java.io.Serializable;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.utils.FBUtilities;

public class MerkleTree
implements Serializable {
    private static final long serialVersionUID = 2L;
    public static final byte RECOMMENDED_DEPTH = 126;
    public static final int CONSISTENT = 0;
    public static final int FULLY_INCONSISTENT = 1;
    public static final int PARTIALLY_INCONSISTENT = 2;
    public final byte hashdepth;
    private transient IPartitioner partitioner;
    private long maxsize;
    private long size;
    private Hashable root;

    public MerkleTree(IPartitioner partitioner, byte hashdepth, long maxsize) {
        assert (hashdepth < 127);
        this.partitioner = partitioner;
        this.hashdepth = hashdepth;
        this.maxsize = maxsize;
        this.size = 1L;
        this.root = new Leaf(null);
    }

    static byte inc(byte in) {
        assert (in < 127);
        return (byte)(in + 1);
    }

    public void init() {
        byte sizedepth = (byte)(Math.log10(this.maxsize) / Math.log10(2.0));
        byte depth = (byte)Math.min(sizedepth, this.hashdepth);
        Object mintoken = this.partitioner.getMinimumToken();
        this.root = this.initHelper((Token)mintoken, (Token)mintoken, (byte)0, depth);
        this.size = (long)Math.pow(2.0, depth);
    }

    private Hashable initHelper(Token left, Token right, byte depth, byte max) {
        if (depth == max) {
            return new Leaf();
        }
        Token midpoint = this.partitioner.midpoint(left, right);
        Hashable lchild = this.initHelper(left, midpoint, MerkleTree.inc(depth), max);
        Hashable rchild = this.initHelper(midpoint, right, MerkleTree.inc(depth), max);
        return new Inner(midpoint, lchild, rchild);
    }

    Hashable root() {
        return this.root;
    }

    public IPartitioner partitioner() {
        return this.partitioner;
    }

    public long size() {
        return this.size;
    }

    public long maxsize() {
        return this.maxsize;
    }

    public void maxsize(long maxsize) {
        this.maxsize = maxsize;
    }

    public void partitioner(IPartitioner partitioner) {
        this.partitioner = partitioner;
    }

    public static List<TreeRange> difference(MerkleTree ltree, MerkleTree rtree) {
        ArrayList<TreeRange> diff = new ArrayList<TreeRange>();
        Object mintoken = ltree.partitioner.getMinimumToken();
        TreeRange active = new TreeRange(null, (Token)mintoken, (Token)mintoken, 0, null);
        byte[] lhash = ltree.hash(active);
        byte[] rhash = rtree.hash(active);
        if (lhash != null && rhash != null && !Arrays.equals(lhash, rhash)) {
            if (1 == MerkleTree.differenceHelper(ltree, rtree, diff, active)) {
                diff.add(active);
            }
        } else if (lhash == null || rhash == null) {
            diff.add(active);
        }
        return diff;
    }

    static int differenceHelper(MerkleTree ltree, MerkleTree rtree, List<TreeRange> diff, TreeRange active) {
        boolean rreso;
        boolean lreso;
        Token midpoint = ltree.partitioner().midpoint(active.left, active.right);
        TreeRange left = new TreeRange(null, active.left, midpoint, MerkleTree.inc(active.depth), null);
        TreeRange right = new TreeRange(null, midpoint, active.right, MerkleTree.inc(active.depth), null);
        byte[] lhash = ltree.hash(left);
        byte[] rhash = rtree.hash(left);
        int ldiff = 0;
        boolean bl = lreso = lhash != null && rhash != null;
        if (lreso && !Arrays.equals(lhash, rhash)) {
            ldiff = MerkleTree.differenceHelper(ltree, rtree, diff, left);
        } else if (!lreso) {
            ldiff = 1;
        }
        lhash = ltree.hash(right);
        rhash = rtree.hash(right);
        int rdiff = 0;
        boolean bl2 = rreso = lhash != null && rhash != null;
        if (rreso && !Arrays.equals(lhash, rhash)) {
            rdiff = MerkleTree.differenceHelper(ltree, rtree, diff, right);
        } else if (!rreso) {
            rdiff = 1;
        }
        if (ldiff == 1 && rdiff == 1) {
            return 1;
        }
        if (ldiff == 1) {
            diff.add(left);
            return 2;
        }
        if (rdiff == 1) {
            diff.add(right);
            return 2;
        }
        return 2;
    }

    TreeRange get(Token t) {
        Object mintoken = this.partitioner.getMinimumToken();
        return this.getHelper(this.root, (Token)mintoken, (Token)mintoken, (byte)0, t);
    }

    TreeRange getHelper(Hashable hashable, Token pleft, Token pright, byte depth, Token t) {
        if (hashable instanceof Leaf) {
            return new TreeRange(this, pleft, pright, depth, hashable);
        }
        Inner node = (Inner)hashable;
        if (Range.contains(pleft, node.token, t)) {
            return this.getHelper(node.lchild, pleft, node.token, MerkleTree.inc(depth), t);
        }
        return this.getHelper(node.rchild, node.token, pright, MerkleTree.inc(depth), t);
    }

    public void invalidate(Token t) {
        this.invalidateHelper(this.root, (Token)this.partitioner.getMinimumToken(), t);
    }

    private void invalidateHelper(Hashable hashable, Token pleft, Token t) {
        hashable.hash(null);
        if (hashable instanceof Leaf) {
            return;
        }
        Inner node = (Inner)hashable;
        if (Range.contains(pleft, node.token, t)) {
            this.invalidateHelper(node.lchild, pleft, t);
        } else {
            this.invalidateHelper(node.rchild, node.token, t);
        }
    }

    public byte[] hash(Range range) {
        Object mintoken = this.partitioner.getMinimumToken();
        try {
            return this.hashHelper(this.root, new Range((Token)mintoken, (Token)mintoken), range);
        }
        catch (StopRecursion e) {
            return null;
        }
    }

    private byte[] hashHelper(Hashable hashable, Range active, Range range) throws StopRecursion {
        if (hashable instanceof Leaf) {
            if (!range.contains(active)) {
                throw new StopRecursion.BadRange();
            }
            return hashable.hash();
        }
        Inner node = (Inner)hashable;
        Range leftactive = new Range(active.left, node.token);
        Range rightactive = new Range(node.token, active.right);
        if (range.contains(active)) {
            if (node.hash() != null) {
                return node.hash();
            }
            byte[] lhash = this.hashHelper(node.lchild(), leftactive, range);
            byte[] rhash = this.hashHelper(node.rchild(), rightactive, range);
            node.hash(lhash, rhash);
            return node.hash();
        }
        if (leftactive.contains(range)) {
            return this.hashHelper(node.lchild, leftactive, range);
        }
        if (rightactive.contains(range)) {
            return this.hashHelper(node.rchild, rightactive, range);
        }
        throw new StopRecursion.BadRange();
    }

    public boolean split(Token t) {
        if (this.size >= this.maxsize) {
            return false;
        }
        Object mintoken = this.partitioner.getMinimumToken();
        try {
            this.root = this.splitHelper(this.root, (Token)mintoken, (Token)mintoken, (byte)0, t);
        }
        catch (StopRecursion.TooDeep e) {
            return false;
        }
        return true;
    }

    private Hashable splitHelper(Hashable hashable, Token pleft, Token pright, byte depth, Token t) throws StopRecursion.TooDeep {
        if (depth >= this.hashdepth) {
            throw new StopRecursion.TooDeep();
        }
        if (hashable instanceof Leaf) {
            ++this.size;
            Token midpoint = this.partitioner.midpoint(pleft, pright);
            return new Inner(midpoint, new Leaf(), new Leaf());
        }
        Inner node = (Inner)hashable;
        if (Range.contains(pleft, node.token, t)) {
            node.lchild(this.splitHelper(node.lchild, pleft, node.token, MerkleTree.inc(depth), t));
        } else {
            node.rchild(this.splitHelper(node.rchild, node.token, pright, MerkleTree.inc(depth), t));
        }
        return node;
    }

    public void compact(Token t) {
        this.root = this.compactHelper(this.root, t);
    }

    private Hashable compactHelper(Hashable hashable, Token t) {
        assert (!(hashable instanceof Leaf));
        Inner node = (Inner)hashable;
        int comp = t.compareTo(node.token);
        if (comp == 0) {
            assert (node.lchild() instanceof Leaf && node.rchild() instanceof Leaf) : "Can only compact a subrange evenly split by the given token!";
            --this.size;
            return new Leaf(node.lchild().hash(), node.rchild().hash());
        }
        if (comp < 0) {
            node.lchild(this.compactHelper(node.lchild(), t));
        } else {
            node.rchild(this.compactHelper(node.rchild(), t));
        }
        return node;
    }

    public TreeRangeIterator invalids(Range range) {
        return new TreeRangeIterator(this, range);
    }

    public String toString() {
        StringBuilder buff = new StringBuilder();
        buff.append("#<MerkleTree root=");
        this.root.toString(buff, 8);
        buff.append(">");
        return buff.toString();
    }

    static abstract class StopRecursion
    extends Exception {
        StopRecursion() {
        }

        static class TooDeep
        extends StopRecursion {
        }

        static class InvalidHash
        extends StopRecursion {
        }

        static class BadRange
        extends StopRecursion {
        }
    }

    static abstract class Hashable
    implements Serializable {
        private static final long serialVersionUID = 1L;
        protected byte[] hash;

        protected Hashable(byte[] hash) {
            this.hash = hash;
        }

        public byte[] hash() {
            return this.hash;
        }

        void hash(byte[] hash) {
            this.hash = hash;
        }

        void hash(byte[] lefthash, byte[] righthash) {
            this.hash = Hashable.binaryHash(lefthash, righthash);
        }

        void addHash(byte[] righthash) {
            this.hash = this.hash == null ? righthash : Hashable.binaryHash(this.hash, righthash);
        }

        static byte[] binaryHash(byte[] left, byte[] right) {
            return FBUtilities.xor(left, right);
        }

        public abstract void toString(StringBuilder var1, int var2);

        public static String toString(byte[] hash) {
            if (hash == null) {
                return "null";
            }
            return "[" + FBUtilities.bytesToHex(hash) + "]";
        }
    }

    public static class RowHash {
        public final Token token;
        public final byte[] hash;

        public RowHash(Token token, byte[] hash) {
            this.token = token;
            this.hash = hash;
        }

        public String toString() {
            return "#<RowHash " + this.token + " " + Hashable.toString(this.hash) + ">";
        }
    }

    static class Leaf
    extends Hashable {
        public static final long serialVersionUID = 1L;

        public Leaf() {
            super(null);
        }

        public Leaf(byte[] hash) {
            super(hash);
        }

        public Leaf(byte[] lefthash, byte[] righthash) {
            super(Hashable.binaryHash(lefthash, righthash));
        }

        @Override
        public void toString(StringBuilder buff, int maxdepth) {
            buff.append(this.toString());
        }

        public String toString() {
            return "#<Leaf " + Hashable.toString(this.hash()) + ">";
        }
    }

    static class Inner
    extends Hashable {
        public static final long serialVersionUID = 1L;
        public final Token token;
        private Hashable lchild;
        private Hashable rchild;

        public Inner(Token token, Hashable lchild, Hashable rchild) {
            super(null);
            this.token = token;
            this.lchild = lchild;
            this.rchild = rchild;
        }

        public Hashable lchild() {
            return this.lchild;
        }

        public Hashable rchild() {
            return this.rchild;
        }

        public void lchild(Hashable child) {
            this.lchild = child;
        }

        public void rchild(Hashable child) {
            this.rchild = child;
        }

        @Override
        public void toString(StringBuilder buff, int maxdepth) {
            buff.append("#<").append(this.getClass().getSimpleName());
            buff.append(" ").append(this.token);
            buff.append(" hash=").append(Hashable.toString(this.hash()));
            buff.append(" children=[");
            if (maxdepth < 1) {
                buff.append("#");
            } else {
                if (this.lchild == null) {
                    buff.append("null");
                } else {
                    this.lchild.toString(buff, maxdepth - 1);
                }
                buff.append(" ");
                if (this.rchild == null) {
                    buff.append("null");
                } else {
                    this.rchild.toString(buff, maxdepth - 1);
                }
            }
            buff.append("]>");
        }

        public String toString() {
            StringBuilder buff = new StringBuilder();
            this.toString(buff, 1);
            return buff.toString();
        }
    }

    public static class TreeRangeIterator
    extends AbstractIterator<TreeRange>
    implements Iterable<TreeRange>,
    PeekingIterator<TreeRange> {
        private final ArrayDeque<TreeRange> tovisit;
        private final Range range;
        private final MerkleTree tree;

        TreeRangeIterator(MerkleTree tree, Range range) {
            Object mintoken = tree.partitioner().getMinimumToken();
            this.tovisit = new ArrayDeque();
            this.tovisit.add(new TreeRange(tree, (Token)mintoken, (Token)mintoken, 0, tree.root));
            this.tree = tree;
            this.range = range;
        }

        public TreeRange computeNext() {
            while (!this.tovisit.isEmpty()) {
                TreeRange active = this.tovisit.pop();
                if (active.hashable.hash() != null) continue;
                if (active.hashable instanceof Leaf) {
                    return active;
                }
                Inner node = (Inner)active.hashable;
                TreeRange left = new TreeRange(this.tree, active.left, node.token, MerkleTree.inc(active.depth), node.lchild);
                TreeRange right = new TreeRange(this.tree, node.token, active.right, MerkleTree.inc(active.depth), node.rchild);
                if (right.intersects(this.range)) {
                    this.tovisit.push(right);
                }
                if (!left.intersects(this.range)) continue;
                this.tovisit.push(left);
            }
            return (TreeRange)this.endOfData();
        }

        @Override
        public Iterator<TreeRange> iterator() {
            return this;
        }
    }

    public static class TreeRange
    extends Range {
        public static final long serialVersionUID = 1L;
        private final MerkleTree tree;
        public final byte depth;
        private final Hashable hashable;

        TreeRange(MerkleTree tree, Token left, Token right, byte depth, Hashable hashable) {
            super(left, right);
            this.tree = tree;
            this.depth = depth;
            this.hashable = hashable;
        }

        public void hash(byte[] hash) {
            assert (this.tree != null) : "Not intended for modification!";
            this.hashable.hash(hash);
        }

        public byte[] hash() {
            return this.hashable.hash();
        }

        public void addHash(RowHash entry) {
            assert (this.tree != null) : "Not intended for modification!";
            assert (this.hashable instanceof Leaf);
            this.hashable.addHash(entry.hash);
        }

        public void addAll(Iterator<RowHash> entries) {
            while (entries.hasNext()) {
                this.addHash(entries.next());
            }
        }

        @Override
        public String toString() {
            StringBuilder buff = new StringBuilder("#<TreeRange ");
            buff.append(super.toString()).append(" depth=").append(this.depth);
            return buff.append(">").toString();
        }
    }
}

