/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.collect.Iterables;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.compaction.LeveledCompactionStrategy;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.FileUtils;
import org.codehaus.jackson.JsonEncoding;
import org.codehaus.jackson.JsonFactory;
import org.codehaus.jackson.JsonGenerator;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledManifest {
    private static final Logger logger = LoggerFactory.getLogger(LeveledCompactionStrategy.class);
    static int MAX_COMPACTING_L0 = 32;
    private final ColumnFamilyStore cfs;
    private final List<SSTableReader>[] generations;
    private final DecoratedKey[] lastCompactedKeys;
    private final int maxSSTableSizeInMB;
    private int levelCount;

    private LeveledManifest(ColumnFamilyStore cfs, int maxSSTableSizeInMB) {
        this.cfs = cfs;
        this.maxSSTableSizeInMB = maxSSTableSizeInMB;
        int n = (int)Math.log10(1000000000 / maxSSTableSizeInMB);
        this.generations = new List[n];
        this.lastCompactedKeys = new DecoratedKey[n];
        for (int i = 0; i < this.generations.length; ++i) {
            this.generations[i] = new ArrayList<SSTableReader>();
            this.lastCompactedKeys[i] = new DecoratedKey(cfs.partitioner.getMinimumToken(), null);
        }
    }

    static LeveledManifest create(ColumnFamilyStore cfs, int maxSSTableSize) {
        LeveledManifest manifest = new LeveledManifest(cfs, maxSSTableSize);
        LeveledManifest.load(cfs, manifest);
        for (SSTableReader ssTableReader : cfs.getSSTables()) {
            if (manifest.levelOf(ssTableReader) >= 0) continue;
            manifest.add(ssTableReader);
        }
        return manifest;
    }

    private static void load(ColumnFamilyStore cfs, LeveledManifest manifest) {
        File manifestFile = LeveledManifest.tryGetManifest(cfs);
        if (manifestFile == null) {
            return;
        }
        ObjectMapper m = new ObjectMapper();
        try {
            JsonNode rootNode = (JsonNode)m.readValue(manifestFile, JsonNode.class);
            JsonNode generations = rootNode.get("generations");
            assert (generations.isArray());
            for (JsonNode generation : generations) {
                int level = generation.get("generation").getIntValue();
                JsonNode generationValues = generation.get("members");
                for (JsonNode generationValue : generationValues) {
                    for (SSTableReader ssTableReader : cfs.getSSTables()) {
                        if (ssTableReader.descriptor.generation != generationValue.getIntValue()) continue;
                        logger.debug("Loading {} at L{}", (Object)ssTableReader, (Object)level);
                        manifest.add(ssTableReader, level);
                    }
                }
            }
        }
        catch (Exception e) {
            logger.error("Manifest present but corrupt. Cassandra will compact levels from scratch", (Throwable)e);
        }
    }

    public synchronized void add(SSTableReader reader) {
        logger.debug("Adding {} to L0", (Object)reader);
        this.add(reader, 0);
        this.serialize();
    }

    private int skipLevels(int newLevel, Iterable<SSTableReader> added) {
        while (this.maxBytesForLevel(newLevel) < (double)SSTableReader.getTotalBytes(added) && this.generations[newLevel + 1].isEmpty()) {
            ++newLevel;
        }
        return newLevel;
    }

    public synchronized void promote(Iterable<SSTableReader> removed, Iterable<SSTableReader> added) {
        if (logger.isDebugEnabled()) {
            logger.debug((Iterables.isEmpty(added) ? "Removing [" : "Replacing [") + this.toString(removed) + "]");
        }
        int minimumLevel = Integer.MAX_VALUE;
        int maximumLevel = 0;
        for (SSTableReader sstable : removed) {
            int thisLevel = this.levelOf(sstable);
            maximumLevel = Math.max(maximumLevel, thisLevel);
            minimumLevel = Math.min(minimumLevel, thisLevel);
            this.remove(sstable);
        }
        if (!added.iterator().hasNext()) {
            return;
        }
        int newLevel = minimumLevel == maximumLevel ? maximumLevel + 1 : maximumLevel;
        newLevel = this.skipLevels(newLevel, added);
        assert (newLevel > 0);
        if (logger.isDebugEnabled()) {
            logger.debug("Adding [{}] at L{}", (Object)this.toString(added), (Object)newLevel);
        }
        this.lastCompactedKeys[minimumLevel] = ((SSTableReader)SSTable.sstableOrdering.max(added)).last;
        for (SSTableReader ssTableReader : added) {
            this.add(ssTableReader, newLevel);
        }
        this.serialize();
    }

    private String toString(Iterable<SSTableReader> sstables) {
        StringBuilder builder = new StringBuilder();
        for (SSTableReader sstable : sstables) {
            builder.append(sstable.descriptor.cfname).append('-').append(sstable.descriptor.generation).append("(L").append(this.levelOf(sstable)).append("), ");
        }
        return builder.toString();
    }

    private double maxBytesForLevel(int level) {
        return level == 0 ? (double)(4 * this.maxSSTableSizeInMB * 1024 * 1024) : Math.pow(10.0, level) * (double)this.maxSSTableSizeInMB * 1024.0 * 1024.0;
    }

    public synchronized Collection<SSTableReader> getCompactionCandidates() {
        this.logDistribution();
        for (int i = this.generations.length - 1; i >= 0; --i) {
            List<SSTableReader> sstables = this.generations[i];
            if (sstables.isEmpty()) continue;
            double score = (double)SSTableReader.getTotalBytes(sstables) / this.maxBytesForLevel(i);
            logger.debug("Compaction score for level {} is {}", (Object)i, (Object)score);
            if (!(score > 1.0) && i != 0) continue;
            Collection<SSTableReader> candidates = this.getCandidatesFor(i);
            if (logger.isDebugEnabled()) {
                logger.debug("Compaction candidates for L{} are {}", (Object)i, (Object)this.toString(candidates));
            }
            return candidates;
        }
        return Collections.emptyList();
    }

    public int getLevelSize(int i) {
        return this.generations.length > i ? this.generations[i].size() : 0;
    }

    public void logDistribution() {
        for (int i = 0; i < this.generations.length; ++i) {
            if (this.generations[i].isEmpty()) continue;
            logger.debug("L{} contains {} SSTables ({} bytes) in {}", new Object[]{i, this.generations[i].size(), SSTableReader.getTotalBytes(this.generations[i]), this});
        }
    }

    private int levelOf(SSTableReader sstable) {
        for (int level = 0; level < this.generations.length; ++level) {
            if (!this.generations[level].contains(sstable)) continue;
            return level;
        }
        return -1;
    }

    private void remove(SSTableReader reader) {
        int level = this.levelOf(reader);
        assert (level >= 0) : reader + " not present in manifest";
        this.generations[level].remove(reader);
    }

    private void add(SSTableReader sstable, int level) {
        this.generations[level].add(sstable);
    }

    private static List<SSTableReader> overlapping(SSTableReader sstable, Iterable<SSTableReader> candidates) {
        ArrayList<SSTableReader> overlapped = new ArrayList<SSTableReader>();
        overlapped.add(sstable);
        Range promotedRange = new Range((Token)sstable.first.token, (Token)sstable.last.token);
        for (SSTableReader candidate : candidates) {
            Range candidateRange = new Range((Token)candidate.first.token, (Token)candidate.last.token);
            if (!candidateRange.intersects(promotedRange)) continue;
            overlapped.add(candidate);
        }
        return overlapped;
    }

    private Collection<SSTableReader> getCandidatesFor(int level) {
        assert (!this.generations[level].isEmpty());
        logger.debug("Choosing candidates for L{}", (Object)level);
        if (level == 0) {
            HashSet<SSTableReader> candidates = new HashSet<SSTableReader>();
            ArrayList<SSTableReader> ageSortedSSTables = new ArrayList<SSTableReader>(this.generations[0]);
            Collections.sort(ageSortedSSTables, SSTable.maxTimestampComparator);
            List<SSTableReader> L0 = LeveledManifest.overlapping((SSTableReader)ageSortedSSTables.get(0), this.generations[0]);
            L0 = L0.size() > MAX_COMPACTING_L0 ? L0.subList(0, MAX_COMPACTING_L0) : L0;
            for (SSTableReader sstable : L0) {
                candidates.addAll(LeveledManifest.overlapping(sstable, this.generations[1]));
            }
            return candidates;
        }
        Collections.sort(this.generations[level], SSTable.sstableComparator);
        for (SSTableReader sstable : this.generations[level]) {
            if (sstable.first.compareTo(this.lastCompactedKeys[level]) <= 0) continue;
            return LeveledManifest.overlapping(sstable, this.generations[level + 1]);
        }
        return LeveledManifest.overlapping(this.generations[level].get(0), this.generations[level + 1]);
    }

    public synchronized void serialize() {
        File manifestFile = LeveledManifest.tryGetManifest(this.cfs);
        if (manifestFile == null) {
            manifestFile = new File(new File(DatabaseDescriptor.getAllDataFileLocations()[0], this.cfs.table.name), this.cfs.columnFamily + ".json");
        }
        File oldFile = new File(manifestFile.getPath().replace(".json", "-old.json"));
        File tmpFile = new File(manifestFile.getPath().replace(".json", "-tmp.json"));
        JsonFactory f = new JsonFactory();
        try {
            JsonGenerator g = f.createJsonGenerator(tmpFile, JsonEncoding.UTF8);
            g.useDefaultPrettyPrinter();
            g.writeStartObject();
            g.writeArrayFieldStart("generations");
            for (int level = 0; level < this.generations.length; ++level) {
                g.writeStartObject();
                g.writeNumberField("generation", level);
                g.writeArrayFieldStart("members");
                for (SSTableReader ssTableReader : this.generations[level]) {
                    g.writeNumber(ssTableReader.descriptor.generation);
                }
                g.writeEndArray();
                g.writeEndObject();
            }
            g.writeEndArray();
            g.writeEndObject();
            g.close();
            if (oldFile.exists() && manifestFile.exists()) {
                FileUtils.deleteWithConfirm(oldFile);
            }
            if (manifestFile.exists()) {
                FileUtils.renameWithConfirm(manifestFile, oldFile);
            }
            assert (tmpFile.exists());
            FileUtils.renameWithConfirm(tmpFile, manifestFile);
            logger.debug("Saved manifest {}", (Object)manifestFile);
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public static File tryGetManifest(ColumnFamilyStore cfs) {
        for (String dir : DatabaseDescriptor.getAllDataFileLocations()) {
            File manifestFile = new File(new File(dir, cfs.table.name), cfs.columnFamily + ".json");
            if (!manifestFile.exists()) continue;
            logger.debug("Found manifest at {}", (Object)manifestFile);
            return manifestFile;
        }
        logger.debug("No level manifest found");
        return null;
    }

    public String toString() {
        return "Manifest@" + this.hashCode();
    }

    public int getLevelCount() {
        for (int i = this.generations.length - 1; i >= 0; --i) {
            if (this.generations[i].size() <= 0) continue;
            return i;
        }
        return 0;
    }

    public List<SSTableReader> getLevel(int i) {
        return this.generations[i];
    }
}

