/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.comet.shaded.arrow.vector.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.channels.Channels;

import org.apache.comet.shaded.arrow.memory.BufferAllocator;
import org.apache.comet.shaded.arrow.vector.ipc.ReadChannel;
import org.apache.comet.shaded.arrow.vector.ipc.WriteChannel;
import org.apache.comet.shaded.arrow.vector.ipc.message.MessageChannelReader;
import org.apache.comet.shaded.arrow.vector.ipc.message.MessageResult;
import org.apache.comet.shaded.arrow.vector.ipc.message.MessageSerializer;
import org.apache.comet.shaded.arrow.vector.types.pojo.Schema;

/**
 * Schema utility class including serialization and deserialization.
 */
public class SchemaUtility {
  private SchemaUtility() {
  }

  /**
   * Deserialize Arrow schema from byte array.
   */
  public static Schema deserialize(byte[] bytes, BufferAllocator allocator) throws IOException {
    try (MessageChannelReader schemaReader =
             new MessageChannelReader(
                 new ReadChannel(
                     new ByteArrayReadableSeekableByteChannel(bytes)), allocator)) {

      MessageResult result = schemaReader.readNext();
      if (result == null) {
        throw new IOException("Unexpected end of input. Missing schema.");
      }
      return MessageSerializer.deserializeSchema(result.getMessage());
    }
  }

  /**
   * Serialize Arrow schema into byte array.
   */
  public static byte[] serialize(Schema schema) throws IOException {
    final ByteArrayOutputStream out = new ByteArrayOutputStream();
    MessageSerializer.serialize(new WriteChannel(Channels.newChannel(out)), schema);
    return out.toByteArray();
  }
}
