/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.opentracing.Scope;
import java.io.IOException;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerNotOpenException;
import org.apache.hadoop.hdds.scm.container.common.helpers.InvalidContainerStateException;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.protocolPB.ContainerCommandResponseBuilders;
import org.apache.hadoop.hdds.security.token.TokenVerifier;
import org.apache.hadoop.hdds.tracing.TracingUtil;
import org.apache.hadoop.ozone.audit.AuditAction;
import org.apache.hadoop.ozone.audit.AuditEventStatus;
import org.apache.hadoop.ozone.audit.AuditLogger;
import org.apache.hadoop.ozone.audit.AuditLoggerType;
import org.apache.hadoop.ozone.audit.AuditMarker;
import org.apache.hadoop.ozone.audit.AuditMessage;
import org.apache.hadoop.ozone.audit.Auditor;
import org.apache.hadoop.ozone.audit.DNAction;
import org.apache.hadoop.ozone.container.common.helpers.ContainerCommandRequestPBHelper;
import org.apache.hadoop.ozone.container.common.helpers.ContainerMetrics;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.impl.ContainerSet;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.interfaces.ContainerDispatcher;
import org.apache.hadoop.ozone.container.common.interfaces.Handler;
import org.apache.hadoop.ozone.container.common.statemachine.StateContext;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.DispatcherContext;
import org.apache.hadoop.ozone.container.common.volume.VolumeSet;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HddsDispatcher
implements ContainerDispatcher,
Auditor {
    static final Logger LOG = LoggerFactory.getLogger(HddsDispatcher.class);
    private static final AuditLogger AUDIT = new AuditLogger(AuditLoggerType.DNLOGGER);
    private final Map<ContainerProtos.ContainerType, Handler> handlers;
    private final Configuration conf;
    private final ContainerSet containerSet;
    private final VolumeSet volumeSet;
    private final StateContext context;
    private final float containerCloseThreshold;
    private String scmID;
    private ContainerMetrics metrics;
    private final TokenVerifier tokenVerifier;
    private final boolean isBlockTokenEnabled;

    public HddsDispatcher(Configuration config, ContainerSet contSet, VolumeSet volumes, Map<ContainerProtos.ContainerType, Handler> handlers, StateContext context, ContainerMetrics metrics, TokenVerifier tokenVerifier) {
        this.conf = config;
        this.containerSet = contSet;
        this.volumeSet = volumes;
        this.context = context;
        this.handlers = handlers;
        this.metrics = metrics;
        this.containerCloseThreshold = this.conf.getFloat("hdds.container.close.threshold", 0.9f);
        this.isBlockTokenEnabled = this.conf.getBoolean("hdds.block.token.enabled", false);
        this.tokenVerifier = tokenVerifier;
    }

    @Override
    public void init() {
    }

    @Override
    public void shutdown() {
    }

    private boolean canIgnoreException(ContainerProtos.Result result) {
        switch (result) {
            case SUCCESS: 
            case CONTAINER_UNHEALTHY: 
            case CLOSED_CONTAINER_IO: 
            case DELETE_ON_OPEN_CONTAINER: {
                return true;
            }
        }
        return false;
    }

    @Override
    public void buildMissingContainerSetAndValidate(Map<Long, Long> container2BCSIDMap) {
        this.containerSet.buildMissingContainerSetAndValidate(container2BCSIDMap);
    }

    @Override
    public ContainerProtos.ContainerCommandResponseProto dispatch(ContainerProtos.ContainerCommandRequestProto msg, DispatcherContext dispatcherContext) {
        String spanName = "HddsDispatcher." + msg.getCmdType().name();
        try (Scope scope = TracingUtil.importAndCreateScope((String)spanName, (String)msg.getTraceID());){
            ContainerProtos.ContainerCommandResponseProto containerCommandResponseProto = this.dispatchRequest(msg, dispatcherContext);
            return containerCommandResponseProto;
        }
    }

    private ContainerProtos.ContainerCommandResponseProto dispatchRequest(ContainerProtos.ContainerCommandRequestProto msg, DispatcherContext dispatcherContext) {
        Handler handler;
        ContainerProtos.ContainerType containerType;
        Preconditions.checkNotNull((Object)msg);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Command {}, trace ID: {} ", (Object)msg.getCmdType().toString(), (Object)msg.getTraceID());
        }
        DNAction action = ContainerCommandRequestPBHelper.getAuditAction((ContainerProtos.Type)msg.getCmdType());
        EventType eventType = this.getEventType(msg);
        Map params = ContainerCommandRequestPBHelper.getAuditParams((ContainerProtos.ContainerCommandRequestProto)msg);
        ContainerProtos.ContainerCommandResponseProto responseProto = null;
        long startTime = System.nanoTime();
        ContainerProtos.Type cmdType = msg.getCmdType();
        long containerID = msg.getContainerID();
        this.metrics.incContainerOpsMetrics(cmdType);
        Container container = this.getContainer(containerID);
        boolean isWriteStage = cmdType == ContainerProtos.Type.WriteChunk && dispatcherContext != null && dispatcherContext.getStage() == DispatcherContext.WriteChunkStage.WRITE_DATA;
        boolean isWriteCommitStage = cmdType == ContainerProtos.Type.WriteChunk && dispatcherContext != null && dispatcherContext.getStage() == DispatcherContext.WriteChunkStage.COMMIT_DATA;
        try {
            this.validateBlockToken(msg);
        }
        catch (IOException ioe) {
            StorageContainerException sce = new StorageContainerException("Block token verification failed. " + ioe.getMessage(), (Throwable)ioe, ContainerProtos.Result.BLOCK_TOKEN_VERIFICATION_FAILED);
            return ContainerUtils.logAndReturnError(LOG, sce, msg);
        }
        boolean isCombinedStage = cmdType == ContainerProtos.Type.WriteChunk && (dispatcherContext == null || dispatcherContext.getStage() == DispatcherContext.WriteChunkStage.COMBINED);
        Map<Long, Long> container2BCSIDMap = null;
        if (dispatcherContext != null) {
            container2BCSIDMap = dispatcherContext.getContainer2BCSIDMap();
        }
        if (isWriteCommitStage) {
            Preconditions.checkNotNull(container2BCSIDMap);
            if (container2BCSIDMap.get(containerID) == null) {
                container2BCSIDMap.put(containerID, container.getBlockCommitSequenceId());
                this.containerSet.getMissingContainerSet().remove(containerID);
            }
        }
        if (this.getMissingContainerSet().contains(containerID)) {
            StorageContainerException sce = new StorageContainerException("ContainerID " + containerID + " has been lost and and cannot be recreated on this DataNode", ContainerProtos.Result.CONTAINER_MISSING);
            this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, sce);
            return ContainerUtils.logAndReturnError(LOG, sce, msg);
        }
        if (cmdType != ContainerProtos.Type.CreateContainer) {
            if (container == null && (isWriteStage || isCombinedStage || cmdType == ContainerProtos.Type.PutSmallFile)) {
                responseProto = this.createContainer(msg);
                if (responseProto.getResult() != ContainerProtos.Result.SUCCESS) {
                    StorageContainerException sce = new StorageContainerException("ContainerID " + containerID + " creation failed", responseProto.getResult());
                    this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, sce);
                    return ContainerUtils.logAndReturnError(LOG, sce, msg);
                }
                Preconditions.checkArgument((isWriteStage && container2BCSIDMap != null || dispatcherContext == null ? 1 : 0) != 0);
                if (container2BCSIDMap != null) {
                    container2BCSIDMap.putIfAbsent(containerID, 0L);
                }
                container = this.getContainer(containerID);
            }
            if (container == null) {
                StorageContainerException sce = new StorageContainerException("ContainerID " + containerID + " does not exist", ContainerProtos.Result.CONTAINER_NOT_FOUND);
                this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, sce);
                return ContainerUtils.logAndReturnError(LOG, sce, msg);
            }
            containerType = this.getContainerType(container);
        } else {
            if (!msg.hasCreateContainer()) {
                this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, new Exception("MALFORMED_REQUEST"));
                return ContainerCommandResponseBuilders.malformedRequest((ContainerProtos.ContainerCommandRequestProto)msg);
            }
            containerType = msg.getCreateContainer().getContainerType();
        }
        if (!HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProto)msg)) {
            this.sendCloseContainerActionIfNeeded(container);
        }
        if ((handler = this.getHandler(containerType)) == null) {
            StorageContainerException ex = new StorageContainerException("Invalid ContainerType " + containerType, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
            this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, ex);
            return ContainerUtils.logAndReturnError(LOG, ex, msg);
        }
        responseProto = handler.handle(msg, container, dispatcherContext);
        if (responseProto != null) {
            this.metrics.incContainerOpsLatencies(cmdType, System.nanoTime() - startTime);
            ContainerProtos.Result result = responseProto.getResult();
            if (cmdType == ContainerProtos.Type.CreateContainer && result == ContainerProtos.Result.SUCCESS && dispatcherContext != null) {
                Preconditions.checkNotNull(dispatcherContext.getContainer2BCSIDMap());
                container2BCSIDMap.putIfAbsent(containerID, 0L);
            }
            if (!HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProto)msg) && !this.canIgnoreException(result)) {
                if (container == null) {
                    throw new NullPointerException("Error on creating containers " + result + " " + responseProto.getMessage());
                }
                ContainerProtos.ContainerDataProto.State containerState = ((ContainerData)container.getContainerData()).getState();
                Preconditions.checkState((containerState == ContainerProtos.ContainerDataProto.State.OPEN || containerState == ContainerProtos.ContainerDataProto.State.CLOSING ? 1 : 0) != 0);
                try {
                    handler.markContainerUnhealthy(container);
                }
                catch (IOException ioe) {
                    LOG.error("Failed to mark container " + containerID + " UNHEALTHY. ", (Throwable)ioe);
                }
                Preconditions.checkArgument((((ContainerData)container.getContainerData()).getState() == ContainerProtos.ContainerDataProto.State.UNHEALTHY ? 1 : 0) != 0);
                this.sendCloseContainerActionIfNeeded(container);
            }
            if (result == ContainerProtos.Result.SUCCESS) {
                this.updateBCSID(container, dispatcherContext, cmdType);
                this.audit((AuditAction)action, eventType, params, AuditEventStatus.SUCCESS, null);
            } else {
                this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, new Exception(responseProto.getMessage()));
            }
            return responseProto;
        }
        this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, new Exception("UNSUPPORTED_REQUEST"));
        return ContainerCommandResponseBuilders.unsupportedRequest((ContainerProtos.ContainerCommandRequestProto)msg);
    }

    private void updateBCSID(Container container, DispatcherContext dispatcherContext, ContainerProtos.Type cmdType) {
        if (dispatcherContext != null && (cmdType == ContainerProtos.Type.PutBlock || cmdType == ContainerProtos.Type.PutSmallFile)) {
            Preconditions.checkNotNull((Object)container);
            long bcsID = container.getBlockCommitSequenceId();
            long containerId = ((ContainerData)container.getContainerData()).getContainerID();
            Map<Long, Long> container2BCSIDMap = dispatcherContext.getContainer2BCSIDMap();
            Preconditions.checkNotNull(container2BCSIDMap);
            Preconditions.checkArgument((boolean)container2BCSIDMap.containsKey(containerId));
            container2BCSIDMap.computeIfPresent(containerId, (u, v) -> {
                v = bcsID;
                return v;
            });
        }
    }

    @VisibleForTesting
    ContainerProtos.ContainerCommandResponseProto createContainer(ContainerProtos.ContainerCommandRequestProto containerRequest) {
        ContainerProtos.CreateContainerRequestProto.Builder createRequest = ContainerProtos.CreateContainerRequestProto.newBuilder();
        ContainerProtos.ContainerType containerType = ContainerProtos.ContainerType.KeyValueContainer;
        createRequest.setContainerType(containerType);
        ContainerProtos.ContainerCommandRequestProto.Builder requestBuilder = ContainerProtos.ContainerCommandRequestProto.newBuilder().setCmdType(ContainerProtos.Type.CreateContainer).setContainerID(containerRequest.getContainerID()).setCreateContainer(createRequest.build()).setPipelineID(containerRequest.getPipelineID()).setDatanodeUuid(containerRequest.getDatanodeUuid()).setTraceID(containerRequest.getTraceID());
        Handler handler = this.getHandler(containerType);
        return handler.handle(requestBuilder.build(), null, null);
    }

    private void validateBlockToken(ContainerProtos.ContainerCommandRequestProto msg) throws IOException {
        if (this.isBlockTokenEnabled && this.tokenVerifier != null && HddsUtils.requireBlockToken((ContainerProtos.Type)msg.getCmdType())) {
            this.tokenVerifier.verify(UserGroupInformation.getCurrentUser().getShortUserName(), msg.getEncodedToken(), msg.getCmdType(), HddsUtils.getBlockID((ContainerProtos.ContainerCommandRequestProto)msg).getContainerBlockID().toString());
        }
    }

    @Override
    public void validateContainerCommand(ContainerProtos.ContainerCommandRequestProto msg) throws StorageContainerException {
        long containerID = msg.getContainerID();
        Container container = this.getContainer(containerID);
        if (container == null) {
            return;
        }
        ContainerProtos.ContainerType containerType = container.getContainerType();
        ContainerProtos.Type cmdType = msg.getCmdType();
        DNAction action = ContainerCommandRequestPBHelper.getAuditAction((ContainerProtos.Type)cmdType);
        EventType eventType = this.getEventType(msg);
        Map params = ContainerCommandRequestPBHelper.getAuditParams((ContainerProtos.ContainerCommandRequestProto)msg);
        Handler handler = this.getHandler(containerType);
        if (handler == null) {
            StorageContainerException ex = new StorageContainerException("Invalid ContainerType " + containerType, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
            this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, ex);
            throw ex;
        }
        ContainerProtos.ContainerDataProto.State containerState = container.getContainerState();
        if (!HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProto)msg) && containerState != ContainerProtos.ContainerDataProto.State.OPEN) {
            switch (cmdType) {
                case CreateContainer: {
                    break;
                }
                case CloseContainer: {
                    break;
                }
                default: {
                    ContainerNotOpenException cex = new ContainerNotOpenException("Container " + containerID + " in " + containerState + " state");
                    this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, (Throwable)cex);
                    throw cex;
                }
            }
        } else if (HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProto)msg) && containerState == ContainerProtos.ContainerDataProto.State.INVALID) {
            InvalidContainerStateException iex = new InvalidContainerStateException("Container " + containerID + " in " + containerState + " state");
            this.audit((AuditAction)action, eventType, params, AuditEventStatus.FAILURE, (Throwable)iex);
            throw iex;
        }
        try {
            this.validateBlockToken(msg);
        }
        catch (IOException ioe) {
            StorageContainerException sce = new StorageContainerException("Block token verification failed. " + ioe.getMessage(), (Throwable)ioe, ContainerProtos.Result.BLOCK_TOKEN_VERIFICATION_FAILED);
            throw sce;
        }
    }

    private void sendCloseContainerActionIfNeeded(Container container) {
        boolean shouldClose;
        boolean isSpaceFull = this.isContainerFull(container);
        boolean bl = shouldClose = isSpaceFull || this.isContainerUnhealthy(container);
        if (shouldClose) {
            Object containerData = container.getContainerData();
            StorageContainerDatanodeProtocolProtos.ContainerAction.Reason reason = isSpaceFull ? StorageContainerDatanodeProtocolProtos.ContainerAction.Reason.CONTAINER_FULL : StorageContainerDatanodeProtocolProtos.ContainerAction.Reason.CONTAINER_UNHEALTHY;
            StorageContainerDatanodeProtocolProtos.ContainerAction action = StorageContainerDatanodeProtocolProtos.ContainerAction.newBuilder().setContainerID(((ContainerData)containerData).getContainerID()).setAction(StorageContainerDatanodeProtocolProtos.ContainerAction.Action.CLOSE).setReason(reason).build();
            this.context.addContainerActionIfAbsent(action);
        }
    }

    private boolean isContainerFull(Container container) {
        boolean isOpen = Optional.ofNullable(container).map(cont -> cont.getContainerState() == ContainerProtos.ContainerDataProto.State.OPEN).orElse(Boolean.FALSE);
        if (isOpen) {
            Object containerData = container.getContainerData();
            double containerUsedPercentage = 1.0f * (float)((ContainerData)containerData).getBytesUsed() / (float)((ContainerData)containerData).getMaxSize();
            return containerUsedPercentage >= (double)this.containerCloseThreshold;
        }
        return false;
    }

    private boolean isContainerUnhealthy(Container container) {
        return Optional.ofNullable(container).map(cont -> cont.getContainerState() == ContainerProtos.ContainerDataProto.State.UNHEALTHY).orElse(Boolean.FALSE);
    }

    @Override
    public Handler getHandler(ContainerProtos.ContainerType containerType) {
        return this.handlers.get(containerType);
    }

    @Override
    public void setScmId(String scmId) {
        Preconditions.checkNotNull((Object)scmId, (Object)"scmId Cannot be null");
        if (this.scmID == null) {
            this.scmID = scmId;
            for (Map.Entry<ContainerProtos.ContainerType, Handler> handlerMap : this.handlers.entrySet()) {
                handlerMap.getValue().setScmID(this.scmID);
            }
        }
    }

    @VisibleForTesting
    public Container getContainer(long containerID) {
        return this.containerSet.getContainer(containerID);
    }

    @VisibleForTesting
    public Set<Long> getMissingContainerSet() {
        return this.containerSet.getMissingContainerSet();
    }

    private ContainerProtos.ContainerType getContainerType(Container container) {
        return container.getContainerType();
    }

    @VisibleForTesting
    public void setMetricsForTesting(ContainerMetrics containerMetrics) {
        this.metrics = containerMetrics;
    }

    private EventType getEventType(ContainerProtos.ContainerCommandRequestProto msg) {
        return HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProto)msg) ? EventType.READ : EventType.WRITE;
    }

    private void audit(AuditAction action, EventType eventType, Map<String, String> params, AuditEventStatus result, Throwable exception) {
        switch (result) {
            case SUCCESS: {
                if (eventType == EventType.READ && AUDIT.getLogger().isInfoEnabled(AuditMarker.READ.getMarker())) {
                    AuditMessage amsg = this.buildAuditMessageForSuccess(action, params);
                    AUDIT.logReadSuccess(amsg);
                    break;
                }
                if (eventType != EventType.WRITE || !AUDIT.getLogger().isInfoEnabled(AuditMarker.WRITE.getMarker())) break;
                AuditMessage amsg = this.buildAuditMessageForSuccess(action, params);
                AUDIT.logWriteSuccess(amsg);
                break;
            }
            case FAILURE: {
                if (eventType == EventType.READ && AUDIT.getLogger().isErrorEnabled(AuditMarker.READ.getMarker())) {
                    AuditMessage amsg = this.buildAuditMessageForFailure(action, params, exception);
                    AUDIT.logReadFailure(amsg);
                    break;
                }
                if (eventType != EventType.WRITE || !AUDIT.getLogger().isErrorEnabled(AuditMarker.WRITE.getMarker())) break;
                AuditMessage amsg = this.buildAuditMessageForFailure(action, params, exception);
                AUDIT.logWriteFailure(amsg);
                break;
            }
            default: {
                if (!LOG.isDebugEnabled()) break;
                LOG.debug("Invalid audit event status - {}", (Object)result);
            }
        }
    }

    public AuditMessage buildAuditMessageForSuccess(AuditAction op, Map<String, String> auditMap) {
        return new AuditMessage.Builder().setUser(null).atIp(null).forOperation(op).withParams(auditMap).withResult(AuditEventStatus.SUCCESS).build();
    }

    public AuditMessage buildAuditMessageForFailure(AuditAction op, Map<String, String> auditMap, Throwable throwable) {
        return new AuditMessage.Builder().setUser(null).atIp(null).forOperation(op).withParams(auditMap).withResult(AuditEventStatus.FAILURE).withException(throwable).build();
    }

    static enum EventType {
        READ,
        WRITE;

    }
}

