/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdds.fs.SpaceUsageCheckFactory;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.ozone.common.InconsistentStorageStateException;
import org.apache.hadoop.ozone.container.common.impl.StorageLocationReport;
import org.apache.hadoop.ozone.container.common.utils.HddsVolumeUtil;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.HddsVolumeChecker;
import org.apache.hadoop.ozone.container.common.volume.VolumeInfo;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.ShutdownHookManager;
import org.apache.hadoop.util.Timer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VolumeSet {
    private static final Logger LOG = LoggerFactory.getLogger(VolumeSet.class);
    private Configuration conf;
    private Map<String, HddsVolume> volumeMap;
    private Map<String, HddsVolume> failedVolumeMap;
    private EnumMap<StorageType, List<HddsVolume>> volumeStateMap;
    private final ScheduledExecutorService diskCheckerservice;
    private final ScheduledFuture<?> periodicDiskChecker;
    private final SpaceUsageCheckFactory usageCheckFactory;
    private static final long DISK_CHECK_INTERVAL_MINUTES = 15L;
    private final ReentrantReadWriteLock volumeSetRWLock;
    private final String datanodeUuid;
    private String clusterID;
    private Runnable shutdownHook;
    private final HddsVolumeChecker volumeChecker;

    public VolumeSet(String dnUuid, Configuration conf) throws IOException {
        this(dnUuid, null, conf);
    }

    public VolumeSet(String dnUuid, String clusterID, Configuration conf) throws IOException {
        this.datanodeUuid = dnUuid;
        this.clusterID = clusterID;
        this.conf = conf;
        this.volumeSetRWLock = new ReentrantReadWriteLock();
        this.volumeChecker = this.getVolumeChecker(conf);
        this.diskCheckerservice = Executors.newScheduledThreadPool(1, r -> {
            Thread t = new Thread(r, "Periodic HDDS volume checker");
            t.setDaemon(true);
            return t;
        });
        this.periodicDiskChecker = this.diskCheckerservice.scheduleWithFixedDelay(() -> {
            try {
                this.checkAllVolumes();
            }
            catch (IOException e) {
                LOG.warn("Exception while checking disks", (Throwable)e);
            }
        }, 15L, 15L, TimeUnit.MINUTES);
        this.usageCheckFactory = SpaceUsageCheckFactory.create((Configuration)conf);
        this.initializeVolumeSet();
    }

    @VisibleForTesting
    HddsVolumeChecker getVolumeChecker(Configuration configuration) throws DiskChecker.DiskErrorException {
        return new HddsVolumeChecker(configuration, new Timer());
    }

    private void initializeVolumeSet() throws IOException {
        this.volumeMap = new ConcurrentHashMap<String, HddsVolume>();
        this.failedVolumeMap = new ConcurrentHashMap<String, HddsVolume>();
        this.volumeStateMap = new EnumMap(StorageType.class);
        Collection rawLocations = this.conf.getTrimmedStringCollection("hdds.datanode.dir");
        if (rawLocations.isEmpty()) {
            rawLocations = this.conf.getTrimmedStringCollection("dfs.datanode.data.dir");
        }
        if (rawLocations.isEmpty()) {
            throw new IllegalArgumentException("No location configured in either hdds.datanode.dir or dfs.datanode.data.dir");
        }
        for (StorageType storageType : StorageType.values()) {
            this.volumeStateMap.put(storageType, new ArrayList());
        }
        for (String locationString : rawLocations) {
            try {
                StorageLocation location = StorageLocation.parse((String)locationString);
                HddsVolume hddsVolume = this.createVolume(location.getUri().getPath(), location.getStorageType());
                this.checkAndSetClusterID(hddsVolume.getClusterID());
                LOG.info("Added Volume : {} to VolumeSet", (Object)hddsVolume.getHddsRootDir().getPath());
                if (!hddsVolume.getHddsRootDir().mkdirs() && !hddsVolume.getHddsRootDir().exists()) {
                    throw new IOException("Failed to create HDDS storage dir " + hddsVolume.getHddsRootDir());
                }
                this.volumeMap.put(hddsVolume.getHddsRootDir().getPath(), hddsVolume);
                this.volumeStateMap.get(hddsVolume.getStorageType()).add(hddsVolume);
            }
            catch (IOException e) {
                HddsVolume volume = new HddsVolume.Builder(locationString).failedVolume(true).build();
                this.failedVolumeMap.put(locationString, volume);
                LOG.error("Failed to parse the storage location: " + locationString, (Throwable)e);
            }
        }
        if (this.volumeMap.size() == 0) {
            throw new DiskChecker.DiskOutOfSpaceException("No storage locations configured");
        }
        this.checkAllVolumes();
        this.shutdownHook = () -> this.saveVolumeSetUsed();
        ShutdownHookManager.get().addShutdownHook(this.shutdownHook, 10);
    }

    private void checkAllVolumes() throws IOException {
        Set<HddsVolume> failedVolumes;
        List<HddsVolume> allVolumes = this.getVolumesList();
        try {
            failedVolumes = this.volumeChecker.checkAllVolumes(allVolumes);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted while running disk check", e);
        }
        if (failedVolumes.size() > 0) {
            LOG.warn("checkAllVolumes got {} failed volumes - {}", (Object)failedVolumes.size(), failedVolumes);
            this.handleVolumeFailures(failedVolumes);
        } else {
            LOG.debug("checkAllVolumes encountered no failures");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleVolumeFailures(Set<HddsVolume> failedVolumes) {
        for (HddsVolume v : failedVolumes) {
            this.writeLock();
            try {
                this.volumeMap.remove(v.getHddsRootDir().getPath());
                this.failedVolumeMap.putIfAbsent(v.getHddsRootDir().getPath(), v);
            }
            finally {
                this.writeUnlock();
            }
        }
    }

    private void checkAndSetClusterID(String idFromVersionFile) throws InconsistentStorageStateException {
        if (this.clusterID == null) {
            this.clusterID = idFromVersionFile;
            return;
        }
        if (!idFromVersionFile.equals(this.clusterID)) {
            throw new InconsistentStorageStateException("Mismatched ClusterIDs. VolumeSet has: " + this.clusterID + ", and version file has: " + idFromVersionFile);
        }
    }

    public void readLock() {
        this.volumeSetRWLock.readLock().lock();
    }

    public void readUnlock() {
        this.volumeSetRWLock.readLock().unlock();
    }

    public void writeLock() {
        this.volumeSetRWLock.writeLock().lock();
    }

    public void writeUnlock() {
        this.volumeSetRWLock.writeLock().unlock();
    }

    private HddsVolume createVolume(String locationString, StorageType storageType) throws IOException {
        HddsVolume.Builder volumeBuilder = new HddsVolume.Builder(locationString).conf(this.conf).datanodeUuid(this.datanodeUuid).clusterID(this.clusterID).usageCheckFactory(this.usageCheckFactory).storageType(storageType);
        return volumeBuilder.build();
    }

    boolean addVolume(String dataDir) {
        return this.addVolume(dataDir, StorageType.DEFAULT);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean addVolume(String volumeRoot, StorageType storageType) {
        boolean success;
        String hddsRoot = HddsVolumeUtil.getHddsRoot(volumeRoot);
        this.writeLock();
        try {
            if (this.volumeMap.containsKey(hddsRoot)) {
                LOG.warn("Volume : {} already exists in VolumeMap", (Object)hddsRoot);
                success = false;
            } else {
                if (this.failedVolumeMap.containsKey(hddsRoot)) {
                    this.failedVolumeMap.remove(hddsRoot);
                }
                HddsVolume hddsVolume = this.createVolume(volumeRoot, storageType);
                this.volumeMap.put(hddsVolume.getHddsRootDir().getPath(), hddsVolume);
                this.volumeStateMap.get(hddsVolume.getStorageType()).add(hddsVolume);
                LOG.info("Added Volume : {} to VolumeSet", (Object)hddsVolume.getHddsRootDir().getPath());
                success = true;
            }
        }
        catch (IOException ex) {
            LOG.error("Failed to add volume " + volumeRoot + " to VolumeSet", (Throwable)ex);
            success = false;
        }
        finally {
            this.writeUnlock();
        }
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void failVolume(String dataDir) {
        String hddsRoot = HddsVolumeUtil.getHddsRoot(dataDir);
        this.writeLock();
        try {
            if (this.volumeMap.containsKey(hddsRoot)) {
                HddsVolume hddsVolume = this.volumeMap.get(hddsRoot);
                hddsVolume.failVolume();
                this.volumeMap.remove(hddsRoot);
                this.volumeStateMap.get(hddsVolume.getStorageType()).remove(hddsVolume);
                this.failedVolumeMap.put(hddsRoot, hddsVolume);
                LOG.info("Moving Volume : {} to failed Volumes", (Object)hddsRoot);
            } else if (this.failedVolumeMap.containsKey(hddsRoot)) {
                LOG.info("Volume : {} is not active", (Object)hddsRoot);
            } else {
                LOG.warn("Volume : {} does not exist in VolumeSet", (Object)hddsRoot);
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeVolume(String dataDir) throws IOException {
        String hddsRoot = HddsVolumeUtil.getHddsRoot(dataDir);
        this.writeLock();
        try {
            if (this.volumeMap.containsKey(hddsRoot)) {
                HddsVolume hddsVolume = this.volumeMap.get(hddsRoot);
                hddsVolume.shutdown();
                this.volumeMap.remove(hddsRoot);
                this.volumeStateMap.get(hddsVolume.getStorageType()).remove(hddsVolume);
                LOG.info("Removed Volume : {} from VolumeSet", (Object)hddsRoot);
            } else if (this.failedVolumeMap.containsKey(hddsRoot)) {
                HddsVolume hddsVolume = this.failedVolumeMap.get(hddsRoot);
                hddsVolume.setState(HddsVolume.VolumeState.NON_EXISTENT);
                this.failedVolumeMap.remove(hddsRoot);
                LOG.info("Removed Volume : {} from failed VolumeSet", (Object)hddsRoot);
            } else {
                LOG.warn("Volume : {} does not exist in VolumeSet", (Object)hddsRoot);
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    private void saveVolumeSetUsed() {
        for (HddsVolume hddsVolume : this.volumeMap.values()) {
            try {
                hddsVolume.shutdown();
            }
            catch (Exception ex) {
                LOG.error("Failed to shutdown volume : " + hddsVolume.getHddsRootDir(), (Throwable)ex);
            }
        }
    }

    public void shutdown() {
        this.saveVolumeSetUsed();
        this.stopDiskChecker();
        if (this.shutdownHook != null) {
            ShutdownHookManager.get().removeShutdownHook(this.shutdownHook);
        }
    }

    private void stopDiskChecker() {
        this.periodicDiskChecker.cancel(true);
        this.volumeChecker.shutdownAndWait(0, TimeUnit.SECONDS);
        this.diskCheckerservice.shutdownNow();
    }

    @VisibleForTesting
    public List<HddsVolume> getVolumesList() {
        return ImmutableList.copyOf(this.volumeMap.values());
    }

    @VisibleForTesting
    public List<HddsVolume> getFailedVolumesList() {
        return ImmutableList.copyOf(this.failedVolumeMap.values());
    }

    @VisibleForTesting
    public Map<String, HddsVolume> getVolumeMap() {
        return ImmutableMap.copyOf(this.volumeMap);
    }

    @VisibleForTesting
    public Map<StorageType, List<HddsVolume>> getVolumeStateMap() {
        return ImmutableMap.copyOf(this.volumeStateMap);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StorageContainerDatanodeProtocolProtos.NodeReportProto getNodeReport() throws IOException {
        this.readLock();
        try {
            HddsVolume hddsVolume;
            StorageLocationReport[] reports = new StorageLocationReport[this.volumeMap.size() + this.failedVolumeMap.size()];
            int counter = 0;
            for (Map.Entry<String, HddsVolume> entry : this.volumeMap.entrySet()) {
                long capacity;
                long remaining;
                long scmUsed;
                hddsVolume = entry.getValue();
                VolumeInfo volumeInfo = hddsVolume.getVolumeInfo();
                boolean failed = false;
                try {
                    scmUsed = volumeInfo.getScmUsed();
                    remaining = volumeInfo.getAvailable();
                    capacity = volumeInfo.getCapacity();
                }
                catch (UncheckedIOException ex) {
                    LOG.warn("Failed to get scmUsed and remaining for container storage location {}", (Object)volumeInfo.getRootDir(), (Object)ex);
                    scmUsed = 0L;
                    remaining = 0L;
                    capacity = 0L;
                    failed = true;
                }
                StorageLocationReport.Builder builder = StorageLocationReport.newBuilder();
                builder.setStorageLocation(volumeInfo.getRootDir()).setId(hddsVolume.getStorageID()).setFailed(failed).setCapacity(capacity).setRemaining(remaining).setScmUsed(scmUsed).setStorageType(hddsVolume.getStorageType());
                StorageLocationReport r = builder.build();
                reports[counter++] = r;
            }
            for (Map.Entry<String, HddsVolume> entry : this.failedVolumeMap.entrySet()) {
                hddsVolume = entry.getValue();
                StorageLocationReport.Builder builder = StorageLocationReport.newBuilder();
                builder.setStorageLocation(hddsVolume.getHddsRootDir().getAbsolutePath()).setId(hddsVolume.getStorageID()).setFailed(true).setCapacity(0L).setRemaining(0L).setScmUsed(0L).setStorageType(hddsVolume.getStorageType());
                StorageLocationReport r = builder.build();
                reports[counter++] = r;
            }
            StorageContainerDatanodeProtocolProtos.NodeReportProto.Builder nrb = StorageContainerDatanodeProtocolProtos.NodeReportProto.newBuilder();
            for (int i = 0; i < reports.length; ++i) {
                nrb.addStorageReport(reports[i].getProtoBufMessage());
            }
            StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReportProto = nrb.build();
            return nodeReportProto;
        }
        finally {
            this.readUnlock();
        }
    }
}

