/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.time.Instant;
import java.util.Map;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdfs.util.Canceler;
import org.apache.hadoop.hdfs.util.DataTransferThrottler;
import org.apache.hadoop.io.nativeio.NativeIO;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerDataYaml;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.interfaces.ContainerPacker;
import org.apache.hadoop.ozone.container.common.interfaces.VolumeChoosingPolicy;
import org.apache.hadoop.ozone.container.common.utils.ReferenceCountedDB;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.VolumeSet;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueBlockIterator;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerCheck;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerLocationUtil;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerUtil;
import org.apache.hadoop.util.DiskChecker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyValueContainer
implements Container<KeyValueContainerData> {
    private static final Logger LOG = LoggerFactory.getLogger(Container.class);
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private final KeyValueContainerData containerData;
    private Configuration config;

    public KeyValueContainer(KeyValueContainerData containerData, Configuration ozoneConfig) {
        Preconditions.checkNotNull((Object)containerData, (Object)"KeyValueContainerData cannot be null");
        Preconditions.checkNotNull((Object)ozoneConfig, (Object)"Ozone configuration cannot be null");
        this.config = ozoneConfig;
        this.containerData = containerData;
    }

    @Override
    public void create(VolumeSet volumeSet, VolumeChoosingPolicy volumeChoosingPolicy, String scmId) throws StorageContainerException {
        Preconditions.checkNotNull((Object)volumeChoosingPolicy, (Object)"VolumeChoosingPolicy cannot be null");
        Preconditions.checkNotNull((Object)volumeSet, (Object)"VolumeSet cannot be null");
        Preconditions.checkNotNull((Object)scmId, (Object)"scmId cannot be null");
        File containerMetaDataPath = null;
        long maxSize = this.containerData.getMaxSize();
        volumeSet.readLock();
        try {
            HddsVolume containerVolume = volumeChoosingPolicy.chooseVolume(volumeSet.getVolumesList(), maxSize);
            String hddsVolumeDir = containerVolume.getHddsRootDir().toString();
            long containerID = this.containerData.getContainerID();
            containerMetaDataPath = KeyValueContainerLocationUtil.getContainerMetaDataPath(hddsVolumeDir, scmId, containerID);
            this.containerData.setMetadataPath(containerMetaDataPath.getPath());
            File chunksPath = KeyValueContainerLocationUtil.getChunksLocationPath(hddsVolumeDir, scmId, containerID);
            ContainerUtils.verifyIsNewContainer(containerMetaDataPath);
            File dbFile = this.getContainerDBFile();
            KeyValueContainerUtil.createContainerMetaData(containerMetaDataPath, chunksPath, dbFile, this.config);
            String impl = this.config.getTrimmed("ozone.metastore.impl", "RocksDB");
            this.containerData.setChunksPath(chunksPath.getPath());
            this.containerData.setContainerDBType(impl);
            this.containerData.setDbFile(dbFile);
            this.containerData.setVolume(containerVolume);
            File containerFile = this.getContainerFile();
            this.createContainerFile(containerFile);
        }
        catch (StorageContainerException ex) {
            if (containerMetaDataPath != null && containerMetaDataPath.getParentFile().exists()) {
                FileUtil.fullyDelete((File)containerMetaDataPath.getParentFile());
            }
            throw ex;
        }
        catch (DiskChecker.DiskOutOfSpaceException ex) {
            throw new StorageContainerException("Container creation failed, due to disk out of space", (Throwable)ex, ContainerProtos.Result.DISK_OUT_OF_SPACE);
        }
        catch (FileAlreadyExistsException ex) {
            throw new StorageContainerException("Container creation failed because ContainerFile already exists", (Throwable)ex, ContainerProtos.Result.CONTAINER_ALREADY_EXISTS);
        }
        catch (IOException ex) {
            if (containerMetaDataPath != null && containerMetaDataPath.getParentFile().exists()) {
                FileUtil.fullyDelete((File)containerMetaDataPath.getParentFile());
            }
            throw new StorageContainerException("Container creation failed. " + ex.getMessage(), (Throwable)ex, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
        }
        finally {
            volumeSet.readUnlock();
        }
    }

    public void populatePathFields(String scmId, HddsVolume containerVolume, String hddsVolumeDir) {
        long containerId = this.containerData.getContainerID();
        File containerMetaDataPath = KeyValueContainerLocationUtil.getContainerMetaDataPath(hddsVolumeDir, scmId, containerId);
        File chunksPath = KeyValueContainerLocationUtil.getChunksLocationPath(hddsVolumeDir, scmId, containerId);
        File dbFile = KeyValueContainerLocationUtil.getContainerDBFile(containerMetaDataPath, containerId);
        this.containerData.setMetadataPath(containerMetaDataPath.getPath());
        this.containerData.setChunksPath(chunksPath.getPath());
        this.containerData.setDbFile(dbFile);
        this.containerData.setVolume(containerVolume);
    }

    private void writeToContainerFile(File containerFile, boolean isCreate) throws StorageContainerException {
        File tempContainerFile = null;
        long containerId = this.containerData.getContainerID();
        try {
            tempContainerFile = this.createTempFile(containerFile);
            ContainerDataYaml.createContainerFile(ContainerProtos.ContainerType.KeyValueContainer, this.containerData, tempContainerFile);
            if (isCreate) {
                NativeIO.renameTo((File)tempContainerFile, (File)containerFile);
            } else {
                Files.move(tempContainerFile.toPath(), containerFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
        }
        catch (IOException ex) {
            throw new StorageContainerException("Error while creating/ updating .container file. ContainerID: " + containerId, (Throwable)ex, ContainerProtos.Result.CONTAINER_FILES_CREATE_ERROR);
        }
        finally {
            if (tempContainerFile != null && tempContainerFile.exists() && !tempContainerFile.delete()) {
                LOG.warn("Unable to delete container temporary file: {}.", (Object)tempContainerFile.getAbsolutePath());
            }
        }
    }

    private void createContainerFile(File containerFile) throws StorageContainerException {
        this.writeToContainerFile(containerFile, true);
    }

    private void updateContainerFile(File containerFile) throws StorageContainerException {
        this.writeToContainerFile(containerFile, false);
    }

    @Override
    public void delete() throws StorageContainerException {
        long containerId = this.containerData.getContainerID();
        try {
            KeyValueContainerUtil.removeContainer(this.containerData, this.config);
        }
        catch (StorageContainerException ex) {
            throw ex;
        }
        catch (IOException ex) {
            String errMsg = String.format("Failed to cleanup container. ID: %d", containerId);
            LOG.error(errMsg, (Throwable)ex);
            throw new StorageContainerException(errMsg, (Throwable)ex, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
        }
    }

    @Override
    public void markContainerForClose() throws StorageContainerException {
        this.writeLock();
        try {
            if (this.getContainerState() != ContainerProtos.ContainerDataProto.State.OPEN) {
                throw new StorageContainerException("Attempting to close a " + this.getContainerState() + " container.", ContainerProtos.Result.CONTAINER_NOT_OPEN);
            }
            this.updateContainerData(() -> this.containerData.setState(ContainerProtos.ContainerDataProto.State.CLOSING));
        }
        finally {
            this.writeUnlock();
        }
    }

    @Override
    public void markContainerUnhealthy() throws StorageContainerException {
        this.writeLock();
        try {
            this.updateContainerData(() -> this.containerData.setState(ContainerProtos.ContainerDataProto.State.UNHEALTHY));
        }
        finally {
            this.writeUnlock();
        }
    }

    @Override
    public void quasiClose() throws StorageContainerException {
        this.flushAndSyncDB();
        this.writeLock();
        try {
            this.flushAndSyncDB();
            this.updateContainerData(this.containerData::quasiCloseContainer);
        }
        finally {
            this.writeUnlock();
        }
    }

    @Override
    public void close() throws StorageContainerException {
        this.flushAndSyncDB();
        this.writeLock();
        try {
            this.flushAndSyncDB();
            this.updateContainerData(this.containerData::closeContainer);
        }
        finally {
            this.writeUnlock();
        }
        LOG.info("Container {} is closed with bcsId {}.", (Object)this.containerData.getContainerID(), (Object)this.containerData.getBlockCommitSequenceId());
    }

    @Override
    public void updateDataScanTimestamp(Instant time) throws StorageContainerException {
        this.writeLock();
        try {
            this.updateContainerData(() -> this.containerData.updateDataScanTime(time));
        }
        finally {
            this.writeUnlock();
        }
    }

    private void updateContainerData(Runnable update) throws StorageContainerException {
        Preconditions.checkState((boolean)this.hasWriteLock());
        ContainerProtos.ContainerDataProto.State oldState = null;
        try {
            oldState = this.containerData.getState();
            update.run();
            File containerFile = this.getContainerFile();
            this.updateContainerFile(containerFile);
        }
        catch (StorageContainerException ex) {
            if (oldState != null && this.containerData.getState() != ContainerProtos.ContainerDataProto.State.UNHEALTHY) {
                this.containerData.setState(oldState);
            }
            throw ex;
        }
    }

    private void compactDB() throws StorageContainerException {
        try (ReferenceCountedDB db = BlockUtils.getDB(this.containerData, this.config);){
            db.getStore().compactDB();
        }
        catch (StorageContainerException ex) {
            throw ex;
        }
        catch (IOException ex) {
            LOG.error("Error in DB compaction while closing container", (Throwable)ex);
            throw new StorageContainerException((Throwable)ex, ContainerProtos.Result.ERROR_IN_COMPACT_DB);
        }
    }

    private void flushAndSyncDB() throws StorageContainerException {
        try (ReferenceCountedDB db = BlockUtils.getDB(this.containerData, this.config);){
            db.getStore().flushDB(true);
            LOG.info("Container {} is synced with bcsId {}.", (Object)this.containerData.getContainerID(), (Object)this.containerData.getBlockCommitSequenceId());
        }
        catch (StorageContainerException ex) {
            throw ex;
        }
        catch (IOException ex) {
            LOG.error("Error in DB sync while closing container", (Throwable)ex);
            throw new StorageContainerException((Throwable)ex, ContainerProtos.Result.ERROR_IN_DB_SYNC);
        }
    }

    @Override
    public KeyValueContainerData getContainerData() {
        return this.containerData;
    }

    @Override
    public ContainerProtos.ContainerDataProto.State getContainerState() {
        return this.containerData.getState();
    }

    @Override
    public ContainerProtos.ContainerType getContainerType() {
        return ContainerProtos.ContainerType.KeyValueContainer;
    }

    @Override
    public void update(Map<String, String> metadata, boolean forceUpdate) throws StorageContainerException {
        long containerId = this.containerData.getContainerID();
        if (!this.containerData.isValid()) {
            LOG.debug("Invalid container data. ContainerID: {}", (Object)containerId);
            throw new StorageContainerException("Invalid container data. ContainerID: " + containerId, ContainerProtos.Result.INVALID_CONTAINER_STATE);
        }
        if (!forceUpdate && !this.containerData.isOpen()) {
            throw new StorageContainerException("Updating a closed container without force option is not allowed. ContainerID: " + containerId, ContainerProtos.Result.UNSUPPORTED_REQUEST);
        }
        Map<String, String> oldMetadata = this.containerData.getMetadata();
        try {
            this.writeLock();
            for (Map.Entry<String, String> entry : metadata.entrySet()) {
                this.containerData.addMetadata(entry.getKey(), entry.getValue());
            }
            File containerFile = this.getContainerFile();
            this.updateContainerFile(containerFile);
        }
        catch (StorageContainerException ex) {
            this.containerData.setMetadata(oldMetadata);
            throw ex;
        }
        finally {
            this.writeUnlock();
        }
    }

    @Override
    public void updateDeleteTransactionId(long deleteTransactionId) {
        this.containerData.updateDeleteTransactionId(deleteTransactionId);
    }

    @Override
    public KeyValueBlockIterator blockIterator() throws IOException {
        return new KeyValueBlockIterator(this.containerData.getContainerID(), new File(this.containerData.getContainerPath()));
    }

    @Override
    public void importContainerData(InputStream input, ContainerPacker<KeyValueContainerData> packer) throws IOException {
        this.writeLock();
        try {
            if (this.getContainerFile().exists()) {
                String errorMessage = String.format("Can't import container (cid=%d) data to a specific location as the container descriptor (%s) has already been exist.", this.getContainerData().getContainerID(), this.getContainerFile().getAbsolutePath());
                throw new IOException(errorMessage);
            }
            byte[] descriptorContent = packer.unpackContainerData(this, input);
            Preconditions.checkNotNull((Object)descriptorContent, (Object)("Container descriptor is missing from the container archive: " + this.getContainerData().getContainerID()));
            KeyValueContainerData originalContainerData = (KeyValueContainerData)ContainerDataYaml.readContainer(descriptorContent);
            this.containerData.setState(originalContainerData.getState());
            this.containerData.setContainerDBType(originalContainerData.getContainerDBType());
            this.containerData.setBytesUsed(originalContainerData.getBytesUsed());
            this.update(originalContainerData.getMetadata(), true);
            KeyValueContainerUtil.parseKVContainerData(this.containerData, this.config);
        }
        catch (Exception ex) {
            try {
                FileUtils.deleteDirectory((File)new File(this.containerData.getMetadataPath()));
                FileUtils.deleteDirectory((File)new File(this.containerData.getChunksPath()));
                FileUtils.deleteDirectory((File)this.getContainerFile());
            }
            catch (Exception deleteex) {
                LOG.error("Can not cleanup destination directories after a container import error (cid" + this.containerData.getContainerID() + ")", (Throwable)deleteex);
            }
            throw ex;
        }
        finally {
            this.writeUnlock();
        }
    }

    @Override
    public void exportContainerData(OutputStream destination, ContainerPacker<KeyValueContainerData> packer) throws IOException {
        ContainerProtos.ContainerDataProto.State state = this.getContainerData().getState();
        if (state != ContainerProtos.ContainerDataProto.State.CLOSED && state != ContainerProtos.ContainerDataProto.State.QUASI_CLOSED) {
            throw new IllegalStateException("Only closed/quasi closed containers could be exported: Where as ContainerId=" + this.getContainerData().getContainerID() + " is in state " + state);
        }
        this.compactDB();
        packer.pack(this, destination);
    }

    public void readLock() {
        this.lock.readLock().lock();
    }

    public void readUnlock() {
        this.lock.readLock().unlock();
    }

    public boolean hasReadLock() {
        return this.lock.readLock().tryLock();
    }

    public void writeLock() {
        this.lock.writeLock().lock();
    }

    public void writeUnlock() {
        this.lock.writeLock().unlock();
    }

    public boolean hasWriteLock() {
        return this.lock.writeLock().isHeldByCurrentThread();
    }

    public void readLockInterruptibly() throws InterruptedException {
        this.lock.readLock().lockInterruptibly();
    }

    public void writeLockInterruptibly() throws InterruptedException {
        this.lock.writeLock().lockInterruptibly();
    }

    @Override
    public File getContainerFile() {
        return KeyValueContainer.getContainerFile(this.containerData.getMetadataPath(), this.containerData.getContainerID());
    }

    static File getContainerFile(String metadataPath, long containerId) {
        return new File(metadataPath, containerId + ".container");
    }

    @Override
    public void updateBlockCommitSequenceId(long blockCommitSequenceId) {
        this.containerData.updateBlockCommitSequenceId(blockCommitSequenceId);
    }

    @Override
    public long getBlockCommitSequenceId() {
        return this.containerData.getBlockCommitSequenceId();
    }

    @Override
    public StorageContainerDatanodeProtocolProtos.ContainerReplicaProto getContainerReport() throws StorageContainerException {
        StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.Builder ciBuilder = StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.newBuilder();
        ciBuilder.setContainerID(this.containerData.getContainerID()).setReadCount(this.containerData.getReadCount()).setWriteCount(this.containerData.getWriteCount()).setReadBytes(this.containerData.getReadBytes()).setWriteBytes(this.containerData.getWriteBytes()).setKeyCount(this.containerData.getKeyCount()).setUsed(this.containerData.getBytesUsed()).setState(this.getHddsState()).setDeleteTransactionId(this.containerData.getDeleteTransactionId()).setBlockCommitSequenceId(this.containerData.getBlockCommitSequenceId()).setOriginNodeId(this.containerData.getOriginNodeId());
        return ciBuilder.build();
    }

    private StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State getHddsState() throws StorageContainerException {
        StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State state;
        switch (this.containerData.getState()) {
            case OPEN: {
                state = StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.OPEN;
                break;
            }
            case CLOSING: {
                state = StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSING;
                break;
            }
            case QUASI_CLOSED: {
                state = StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED;
                break;
            }
            case CLOSED: {
                state = StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED;
                break;
            }
            case UNHEALTHY: {
                state = StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.UNHEALTHY;
                break;
            }
            default: {
                throw new StorageContainerException("Invalid Container state found: " + this.containerData.getContainerID(), ContainerProtos.Result.INVALID_CONTAINER_STATE);
            }
        }
        return state;
    }

    public File getContainerDBFile() {
        return new File(this.containerData.getMetadataPath(), this.containerData.getContainerID() + "-dn-container.db");
    }

    @Override
    public boolean scanMetaData() {
        long containerId = this.containerData.getContainerID();
        KeyValueContainerCheck checker = new KeyValueContainerCheck(this.containerData.getMetadataPath(), this.config, containerId);
        return checker.fastCheck();
    }

    @Override
    public boolean shouldScanData() {
        return this.containerData.getState() == ContainerProtos.ContainerDataProto.State.CLOSED || this.containerData.getState() == ContainerProtos.ContainerDataProto.State.QUASI_CLOSED;
    }

    @Override
    public boolean scanData(DataTransferThrottler throttler, Canceler canceler) {
        if (!this.shouldScanData()) {
            throw new IllegalStateException("The checksum verification can not be done for container in state " + this.containerData.getState());
        }
        long containerId = this.containerData.getContainerID();
        KeyValueContainerCheck checker = new KeyValueContainerCheck(this.containerData.getMetadataPath(), this.config, containerId);
        return checker.fullCheck(throttler, canceler);
    }

    private File createTempFile(File file) throws IOException {
        return File.createTempFile("tmp_" + System.currentTimeMillis() + "_", file.getName(), file.getParentFile());
    }

    private static enum ContainerCheckLevel {
        NO_CHECK,
        FAST_CHECK,
        FULL_CHECK;

    }
}

