/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.helpers;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.GatheringByteChannel;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.security.NoSuchAlgorithmException;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfo;
import org.apache.hadoop.ozone.container.common.volume.VolumeIOStats;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.util.Time;
import org.apache.ratis.util.function.CheckedSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ChunkUtils {
    private static final Set<Path> LOCKS = ConcurrentHashMap.newKeySet();
    private static final Logger LOG = LoggerFactory.getLogger(ChunkUtils.class);
    private static final Set<? extends OpenOption> WRITE_OPTIONS = Collections.unmodifiableSet(EnumSet.of(StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.SPARSE));
    private static final Set<? extends OpenOption> READ_OPTIONS = Collections.unmodifiableSet(EnumSet.of(StandardOpenOption.READ));
    private static final FileAttribute<?>[] NO_ATTRIBUTES = new FileAttribute[0];

    private ChunkUtils() {
    }

    public static void writeData(File chunkFile, ChunkInfo chunkInfo, ChunkBuffer data, VolumeIOStats volumeIOStats, boolean sync) throws StorageContainerException, ExecutionException, InterruptedException, NoSuchAlgorithmException {
        int bufferSize = ChunkUtils.validateBufferSize(chunkInfo, data);
        Path path = chunkFile.toPath();
        long startTime = Time.monotonicNow();
        ChunkUtils.processFileExclusively(path, () -> {
            FileChannel file = null;
            try {
                long size;
                file = FileChannel.open(path, WRITE_OPTIONS, NO_ATTRIBUTES);
                try (FileLock ignored = file.lock();){
                    file.position(chunkInfo.getOffset());
                    size = data.writeTo((GatheringByteChannel)file);
                }
                volumeIOStats.incWriteTime(Time.monotonicNow() - startTime);
                volumeIOStats.incWriteOpCount();
                volumeIOStats.incWriteBytes(size);
                if (size != (long)bufferSize) {
                    LOG.error("Invalid write size found. Size:{}  Expected: {} ", (Object)size, (Object)bufferSize);
                    throw new StorageContainerException("Invalid write size found. Size: " + size + " Expected: " + bufferSize, ContainerProtos.Result.INVALID_WRITE_SIZE);
                }
            }
            catch (StorageContainerException ex) {
                throw ex;
            }
            catch (IOException e) {
                throw new StorageContainerException((Throwable)e, ContainerProtos.Result.IO_EXCEPTION);
            }
            finally {
                ChunkUtils.closeFile(file, sync);
            }
            return null;
        });
        if (LOG.isDebugEnabled()) {
            LOG.debug("Write Chunk completed for chunkFile: {}, size {}", (Object)chunkFile, (Object)bufferSize);
        }
    }

    public static int validateBufferSize(ChunkInfo chunkInfo, ChunkBuffer data) throws StorageContainerException {
        int bufferSize = data.remaining();
        if ((long)bufferSize != chunkInfo.getLen()) {
            String err = String.format("data array does not match the length specified. DataLen: %d Byte Array: %d", chunkInfo.getLen(), bufferSize);
            LOG.error(err);
            throw new StorageContainerException(err, ContainerProtos.Result.INVALID_WRITE_SIZE);
        }
        return bufferSize;
    }

    public static ByteBuffer readData(File chunkFile, ChunkInfo data, VolumeIOStats volumeIOStats) throws StorageContainerException {
        long offset = data.getOffset();
        long len = data.getLen();
        ByteBuffer buf = ByteBuffer.allocate((int)len);
        Path path = chunkFile.toPath();
        long startTime = Time.monotonicNow();
        return (ByteBuffer)ChunkUtils.processFileExclusively(path, () -> {
            FileChannel file = null;
            try {
                file = FileChannel.open(path, READ_OPTIONS, NO_ATTRIBUTES);
                try (Object ignored = file.lock(offset, len, true);){
                    file.read(buf, offset);
                    buf.flip();
                }
                volumeIOStats.incReadTime(Time.monotonicNow() - startTime);
                volumeIOStats.incReadOpCount();
                volumeIOStats.incReadBytes(len);
                ignored = buf;
                return ignored;
            }
            catch (NoSuchFileException e) {
                throw new StorageContainerException((Throwable)e, ContainerProtos.Result.UNABLE_TO_FIND_CHUNK);
            }
            catch (IOException e) {
                throw new StorageContainerException((Throwable)e, ContainerProtos.Result.IO_EXCEPTION);
            }
            finally {
                if (file != null) {
                    IOUtils.closeStream((Closeable)file);
                }
            }
        });
    }

    public static boolean validateChunkForOverwrite(File chunkFile, ChunkInfo info) {
        if (ChunkUtils.isOverWriteRequested(chunkFile, info)) {
            if (!ChunkUtils.isOverWritePermitted(info)) {
                LOG.warn("Duplicate write chunk request. Chunk overwrite without explicit request. {}", (Object)info.toString());
            }
            return true;
        }
        return false;
    }

    public static File getChunkFile(KeyValueContainerData containerData, ChunkInfo info) throws StorageContainerException {
        Preconditions.checkNotNull((Object)containerData, (Object)"Container data can't be null");
        String chunksPath = containerData.getChunksPath();
        if (chunksPath == null) {
            LOG.error("Chunks path is null in the container data");
            throw new StorageContainerException("Unable to get Chunks directory.", ContainerProtos.Result.UNABLE_TO_FIND_DATA_DIR);
        }
        File chunksLoc = new File(chunksPath);
        if (!chunksLoc.exists()) {
            LOG.error("Chunks path does not exist");
            throw new StorageContainerException("Unable to get Chunks directory.", ContainerProtos.Result.UNABLE_TO_FIND_DATA_DIR);
        }
        return chunksLoc.toPath().resolve(info.getChunkName()).toFile();
    }

    public static boolean isOverWriteRequested(File chunkFile, ChunkInfo chunkInfo) {
        if (!chunkFile.exists()) {
            return false;
        }
        long offset = chunkInfo.getOffset();
        return offset < chunkFile.length();
    }

    public static boolean isOverWritePermitted(ChunkInfo chunkInfo) {
        String overWrite = (String)chunkInfo.getMetadata().get("OverWriteRequested");
        return overWrite != null && !overWrite.isEmpty() && Boolean.valueOf(overWrite) != false;
    }

    @VisibleForTesting
    static <T, E extends Exception> T processFileExclusively(Path path, CheckedSupplier<T, E> op) throws E {
        while (!LOCKS.add(path)) {
        }
        try {
            Object object = op.get();
            return (T)object;
        }
        finally {
            LOCKS.remove(path);
        }
    }

    private static void closeFile(FileChannel file, boolean sync) throws StorageContainerException {
        if (file != null) {
            try {
                if (sync) {
                    file.force(true);
                }
                file.close();
            }
            catch (IOException e) {
                throw new StorageContainerException("Error closing chunk file", (Throwable)e, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
            }
        }
    }
}

