/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.impl;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.concurrent.ExecutionException;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfo;
import org.apache.hadoop.ozone.container.common.impl.ChunkLayOutVersion;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.DispatcherContext;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.VolumeIOStats;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.ChunkUtils;
import org.apache.hadoop.ozone.container.keyvalue.interfaces.ChunkManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkManagerImpl
implements ChunkManager {
    private static final Logger LOG = LoggerFactory.getLogger(ChunkManagerImpl.class);
    private final boolean doSyncWrite;

    public ChunkManagerImpl(boolean sync) {
        this.doSyncWrite = sync;
    }

    @Override
    public void writeChunk(Container container, BlockID blockID, ChunkInfo info, ChunkBuffer data, DispatcherContext dispatcherContext) throws StorageContainerException {
        Preconditions.checkNotNull((Object)dispatcherContext);
        DispatcherContext.WriteChunkStage stage = dispatcherContext.getStage();
        try {
            KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
            HddsVolume volume = containerData.getVolume();
            VolumeIOStats volumeIOStats = volume.getVolumeIOStats();
            File chunkFile = ChunkUtils.getChunkFile(containerData, info);
            boolean isOverwrite = ChunkUtils.validateChunkForOverwrite(chunkFile, info);
            File tmpChunkFile = this.getTmpChunkFile(chunkFile, dispatcherContext);
            if (LOG.isDebugEnabled()) {
                LOG.debug("writing chunk:{} chunk stage:{} chunk file:{} tmp chunk file:{}", new Object[]{info.getChunkName(), stage, chunkFile, tmpChunkFile});
            }
            switch (stage) {
                case WRITE_DATA: {
                    if (isOverwrite) {
                        LOG.warn("ChunkFile already exists {}. Deleting it.", (Object)chunkFile);
                        FileUtil.fullyDelete((File)chunkFile);
                    }
                    if (tmpChunkFile.exists()) {
                        LOG.warn("tmpChunkFile already exists {}. Overwriting it.", (Object)tmpChunkFile);
                    }
                    ChunkUtils.writeData(tmpChunkFile, info, data, volumeIOStats, this.doSyncWrite);
                    break;
                }
                case COMMIT_DATA: {
                    if (isOverwrite) {
                        LOG.warn("ChunkFile already exists {}", (Object)chunkFile);
                        return;
                    }
                    this.commitChunk(tmpChunkFile, chunkFile);
                    this.updateContainerWriteStats(container, info, isOverwrite);
                    break;
                }
                case COMBINED: {
                    ChunkUtils.writeData(chunkFile, info, data, volumeIOStats, this.doSyncWrite);
                    this.updateContainerWriteStats(container, info, isOverwrite);
                    break;
                }
                default: {
                    throw new IOException("Can not identify write operation.");
                }
            }
        }
        catch (StorageContainerException ex) {
            throw ex;
        }
        catch (NoSuchAlgorithmException ex) {
            LOG.error("write data failed.", (Throwable)ex);
            throw new StorageContainerException("Internal error: ", (Throwable)ex, ContainerProtos.Result.NO_SUCH_ALGORITHM);
        }
        catch (IOException | ExecutionException ex) {
            LOG.error("write data failed.", (Throwable)ex);
            throw new StorageContainerException("Internal error: ", (Throwable)ex, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.error("write data failed.", (Throwable)e);
            throw new StorageContainerException("Internal error: ", (Throwable)e, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
        }
    }

    protected void updateContainerWriteStats(Container container, ChunkInfo info, boolean isOverwrite) {
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        if (!isOverwrite) {
            containerData.incrBytesUsed(info.getLen());
        }
        containerData.incrWriteCount();
        containerData.incrWriteBytes(info.getLen());
    }

    @Override
    public ChunkBuffer readChunk(Container container, BlockID blockID, ChunkInfo info, DispatcherContext dispatcherContext) throws StorageContainerException {
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        HddsVolume volume = containerData.getVolume();
        VolumeIOStats volumeIOStats = volume.getVolumeIOStats();
        if (containerData.getLayOutVersion() == ChunkLayOutVersion.getLatestVersion().getVersion()) {
            File finalChunkFile = ChunkUtils.getChunkFile(containerData, info);
            ArrayList<File> possibleFiles = new ArrayList<File>();
            possibleFiles.add(finalChunkFile);
            if (dispatcherContext != null && dispatcherContext.isReadFromTmpFile()) {
                possibleFiles.add(this.getTmpChunkFile(finalChunkFile, dispatcherContext));
                possibleFiles.add(finalChunkFile);
            }
            for (File chunkFile : possibleFiles) {
                try {
                    ByteBuffer data = ChunkUtils.readData(chunkFile, info, volumeIOStats);
                    containerData.incrReadCount();
                    long length = info.getLen();
                    containerData.incrReadBytes(length);
                    return ChunkBuffer.wrap((ByteBuffer)data);
                }
                catch (StorageContainerException ex) {
                    if (ex.getResult() == ContainerProtos.Result.UNABLE_TO_FIND_CHUNK) continue;
                    throw ex;
                }
            }
            throw new StorageContainerException("Chunk file can't be found " + ((Object)possibleFiles).toString(), ContainerProtos.Result.UNABLE_TO_FIND_CHUNK);
        }
        return null;
    }

    @Override
    public void deleteChunk(Container container, BlockID blockID, ChunkInfo info) throws StorageContainerException {
        Preconditions.checkNotNull((Object)blockID, (Object)"Block ID cannot be null.");
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        if (containerData.getLayOutVersion() == ChunkLayOutVersion.getLatestVersion().getVersion()) {
            File chunkFile = ChunkUtils.getChunkFile(containerData, info);
            if (!chunkFile.exists()) {
                LOG.warn("Chunk file doe not exist. chunk info :{}", (Object)info.toString());
                return;
            }
            if (info.getOffset() == 0L && info.getLen() == chunkFile.length()) {
                FileUtil.fullyDelete((File)chunkFile);
                containerData.decrBytesUsed(chunkFile.length());
            } else {
                LOG.error("Not Supported Operation. Trying to delete a chunk that is in shared file. chunk info : {}", (Object)info.toString());
                throw new StorageContainerException("Not Supported Operation. Trying to delete a chunk that is in shared file. chunk info : " + info.toString(), ContainerProtos.Result.UNSUPPORTED_REQUEST);
            }
        }
    }

    @Override
    public void shutdown() {
    }

    private File getTmpChunkFile(File chunkFile, DispatcherContext dispatcherContext) {
        return new File(chunkFile.getParent(), chunkFile.getName() + "." + "tmp" + "." + dispatcherContext.getTerm() + "." + dispatcherContext.getLogIndex());
    }

    private void commitChunk(File tmpChunkFile, File chunkFile) throws IOException {
        Files.move(tmpChunkFile.toPath(), chunkFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }
}

