/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.engine.flush.CloseFileListener;
import org.apache.iotdb.db.engine.flush.FlushListener;
import org.apache.iotdb.db.engine.flush.TsFileFlushPolicy;
import org.apache.iotdb.db.engine.querycontext.QueryDataSource;
import org.apache.iotdb.db.engine.storagegroup.StorageGroupProcessor;
import org.apache.iotdb.db.engine.storagegroup.TsFileProcessor;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.exception.BatchInsertionException;
import org.apache.iotdb.db.exception.LoadFileException;
import org.apache.iotdb.db.exception.ShutdownException;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.exception.StorageGroupProcessorException;
import org.apache.iotdb.db.exception.TsFileProcessorException;
import org.apache.iotdb.db.exception.WriteProcessException;
import org.apache.iotdb.db.exception.WriteProcessRejectException;
import org.apache.iotdb.db.exception.metadata.IllegalPathException;
import org.apache.iotdb.db.exception.metadata.MetadataException;
import org.apache.iotdb.db.exception.metadata.StorageGroupNotSetException;
import org.apache.iotdb.db.exception.query.QueryProcessException;
import org.apache.iotdb.db.exception.runtime.StorageEngineFailureException;
import org.apache.iotdb.db.metadata.PartialPath;
import org.apache.iotdb.db.metadata.mnode.StorageGroupMNode;
import org.apache.iotdb.db.qp.physical.crud.InsertRowPlan;
import org.apache.iotdb.db.qp.physical.crud.InsertRowsOfOneDevicePlan;
import org.apache.iotdb.db.qp.physical.crud.InsertTabletPlan;
import org.apache.iotdb.db.query.context.QueryContext;
import org.apache.iotdb.db.query.control.QueryFileManager;
import org.apache.iotdb.db.rescon.SystemInfo;
import org.apache.iotdb.db.service.IService;
import org.apache.iotdb.db.service.IoTDB;
import org.apache.iotdb.db.service.ServiceType;
import org.apache.iotdb.db.utils.FilePathUtils;
import org.apache.iotdb.db.utils.UpgradeUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.read.expression.impl.SingleSeriesExpression;
import org.apache.iotdb.tsfile.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageEngine
implements IService {
    private final Logger logger;
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private static final long TTL_CHECK_INTERVAL = 60000L;
    private final String systemDir;
    private final ConcurrentHashMap<PartialPath, StorageGroupProcessor> processorMap = new ConcurrentHashMap();
    private static final ExecutorService recoveryThreadPool = IoTDBThreadPoolFactory.newFixedThreadPool(Runtime.getRuntime().availableProcessors(), "Recovery-Thread-Pool");
    private AtomicBoolean isAllSgReady = new AtomicBoolean(false);
    private ExecutorService recoverAllSgThreadPool;
    private ScheduledExecutorService ttlCheckThread;
    private TsFileFlushPolicy fileFlushPolicy = new TsFileFlushPolicy.DirectFlushPolicy();
    private List<CloseFileListener> customCloseFileListeners = new ArrayList<CloseFileListener>();
    private List<FlushListener> customFlushListeners = new ArrayList<FlushListener>();
    private static long timePartitionInterval = -1L;
    private static boolean enablePartition = IoTDBDescriptor.getInstance().getConfig().isEnablePartition();

    public boolean isAllSgReady() {
        return this.isAllSgReady.get();
    }

    public void setAllSgReady(boolean allSgReady) {
        this.isAllSgReady.set(allSgReady);
    }

    public static StorageEngine getInstance() {
        return InstanceHolder.INSTANCE;
    }

    private StorageEngine() {
        this.logger = LoggerFactory.getLogger(StorageEngine.class);
        this.systemDir = FilePathUtils.regularizePath(config.getSystemDir()) + "storage_groups";
        if (!enablePartition) {
            timePartitionInterval = Long.MAX_VALUE;
        } else {
            StorageEngine.initTimePartition();
        }
        try {
            FileUtils.forceMkdir((File)SystemFileFactory.INSTANCE.getFile(this.systemDir));
        }
        catch (IOException e) {
            throw new StorageEngineFailureException(e);
        }
        UpgradeUtils.recoverUpgrade();
        this.recover();
    }

    public void recover() {
        this.recoverAllSgThreadPool = IoTDBThreadPoolFactory.newSingleThreadExecutor("Begin-Recovery-Pool");
        this.recoverAllSgThreadPool.submit(this::recoverAllSgs);
    }

    private void recoverAllSgs() {
        List<StorageGroupMNode> sgNodes = IoTDB.metaManager.getAllStorageGroupNodes();
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        for (StorageGroupMNode storageGroupMNode : sgNodes) {
            futures.add(recoveryThreadPool.submit(() -> {
                try {
                    StorageGroupProcessor processor = new StorageGroupProcessor(this.systemDir, storageGroup.getFullPath(), this.fileFlushPolicy);
                    processor.setDataTTL(storageGroup.getDataTTL());
                    processor.setCustomCloseFileListeners(this.customCloseFileListeners);
                    processor.setCustomFlushListeners(this.customFlushListeners);
                    this.processorMap.put(storageGroup.getPartialPath(), processor);
                    this.logger.info("Storage Group Processor {} is recovered successfully", (Object)storageGroup.getFullPath());
                }
                catch (Exception e) {
                    this.logger.error("meet error when recovering storage group: {}", (Object)storageGroup.getFullPath(), (Object)e);
                }
                return null;
            }));
        }
        for (Future future : futures) {
            try {
                future.get();
            }
            catch (ExecutionException e) {
                throw new StorageEngineFailureException("StorageEngine failed to recover.", e);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new StorageEngineFailureException("StorageEngine failed to recover.", e);
            }
        }
        recoveryThreadPool.shutdown();
        this.setAllSgReady(true);
    }

    private static void initTimePartition() {
        timePartitionInterval = StorageEngine.convertMilliWithPrecision(IoTDBDescriptor.getInstance().getConfig().getPartitionInterval() * 1000L);
    }

    public static long convertMilliWithPrecision(long milliTime) {
        String timePrecision;
        long result = milliTime;
        switch (timePrecision = IoTDBDescriptor.getInstance().getConfig().getTimestampPrecision()) {
            case "ns": {
                result = milliTime * 1000000L;
                break;
            }
            case "us": {
                result = milliTime * 1000L;
                break;
            }
        }
        return result;
    }

    @Override
    public void start() {
        this.ttlCheckThread = Executors.newSingleThreadScheduledExecutor();
        this.ttlCheckThread.scheduleAtFixedRate(this::checkTTL, 60000L, 60000L, TimeUnit.MILLISECONDS);
    }

    private void checkTTL() {
        try {
            for (StorageGroupProcessor processor : this.processorMap.values()) {
                processor.checkFilesTTL();
            }
        }
        catch (ConcurrentModificationException concurrentModificationException) {
        }
        catch (Exception e) {
            this.logger.error("An error occurred when checking TTL", (Throwable)e);
        }
        if (this.isAllSgReady.get() && !this.recoverAllSgThreadPool.isShutdown()) {
            this.recoverAllSgThreadPool.shutdownNow();
        }
    }

    @Override
    public void stop() {
        this.syncCloseAllProcessor();
        if (this.ttlCheckThread != null) {
            this.ttlCheckThread.shutdownNow();
            try {
                this.ttlCheckThread.awaitTermination(60L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                this.logger.warn("TTL check thread still doesn't exit after 60s");
                Thread.currentThread().interrupt();
                throw new StorageEngineFailureException("StorageEngine failed to stop because of ttlCheckThread.", e);
            }
        }
        recoveryThreadPool.shutdownNow();
        if (!this.recoverAllSgThreadPool.isShutdown()) {
            this.recoverAllSgThreadPool.shutdownNow();
            try {
                this.recoverAllSgThreadPool.awaitTermination(60L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                this.logger.warn("recoverAllSgThreadPool thread still doesn't exit after 60s");
                Thread.currentThread().interrupt();
                throw new StorageEngineFailureException("StorageEngine failed to stop because of recoverAllSgThreadPool.", e);
            }
        }
        this.reset();
    }

    @Override
    public void shutdown(long millseconds) throws ShutdownException {
        try {
            this.forceCloseAllProcessor();
        }
        catch (TsFileProcessorException e) {
            throw new ShutdownException(e);
        }
        if (this.ttlCheckThread != null) {
            this.ttlCheckThread.shutdownNow();
            try {
                this.ttlCheckThread.awaitTermination(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                this.logger.warn("TTL check thread still doesn't exit after 30s");
                Thread.currentThread().interrupt();
            }
        }
        recoveryThreadPool.shutdownNow();
        this.reset();
    }

    @Override
    public ServiceType getID() {
        return ServiceType.STORAGE_ENGINE_SERVICE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StorageGroupProcessor getProcessor(PartialPath path) throws StorageEngineException {
        try {
            StorageGroupMNode storageGroupMNode = IoTDB.metaManager.getStorageGroupNodeByPath(path);
            PartialPath storageGroupPath = storageGroupMNode.getPartialPath();
            StorageGroupProcessor processor = this.processorMap.get(storageGroupPath);
            if (processor == null) {
                if (this.isAllSgReady.get()) {
                    StorageGroupMNode storageGroupMNode2 = storageGroupMNode;
                    synchronized (storageGroupMNode2) {
                        processor = this.processorMap.get(storageGroupPath);
                        if (processor == null) {
                            this.logger.info("construct a processor instance, the storage group is {}, Thread is {}", (Object)storageGroupPath, (Object)Thread.currentThread().getId());
                            processor = new StorageGroupProcessor(this.systemDir, storageGroupPath.getFullPath(), this.fileFlushPolicy);
                            processor.setDataTTL(storageGroupMNode.getDataTTL());
                            processor.setCustomFlushListeners(this.customFlushListeners);
                            processor.setCustomCloseFileListeners(this.customCloseFileListeners);
                            this.processorMap.put(storageGroupPath, processor);
                        }
                    }
                } else {
                    throw new StorageEngineException("the sg " + storageGroupPath + " may not ready now, please wait and retry later", TSStatusCode.STORAGE_GROUP_NOT_READY.getStatusCode());
                }
            }
            return processor;
        }
        catch (StorageGroupProcessorException | MetadataException e) {
            throw new StorageEngineException(e);
        }
    }

    public synchronized void reset() {
        this.processorMap.clear();
    }

    public void insert(InsertRowPlan insertRowPlan) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(insertRowPlan.getDeviceId());
        try {
            storageGroupProcessor.insert(insertRowPlan);
        }
        catch (WriteProcessException e) {
            throw new StorageEngineException(e);
        }
    }

    public void insert(InsertRowsOfOneDevicePlan insertRowsOfOneDevicePlan) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(insertRowsOfOneDevicePlan.getDeviceId());
        try {
            storageGroupProcessor.insert(insertRowsOfOneDevicePlan);
        }
        catch (WriteProcessException e) {
            throw new StorageEngineException(e);
        }
    }

    public void insertTablet(InsertTabletPlan insertTabletPlan) throws StorageEngineException, BatchInsertionException {
        StorageGroupProcessor storageGroupProcessor;
        try {
            storageGroupProcessor = this.getProcessor(insertTabletPlan.getDeviceId());
        }
        catch (StorageEngineException e) {
            throw new StorageEngineException(String.format("Get StorageGroupProcessor of device %s failed", insertTabletPlan.getDeviceId()), e);
        }
        storageGroupProcessor.insertTablet(insertTabletPlan);
    }

    public void syncCloseAllProcessor() {
        this.logger.info("Start closing all storage group processor");
        for (StorageGroupProcessor processor : this.processorMap.values()) {
            processor.syncCloseAllWorkingTsFileProcessors();
        }
    }

    public void forceCloseAllProcessor() throws TsFileProcessorException {
        this.logger.info("Start force closing all storage group processor");
        for (StorageGroupProcessor processor : this.processorMap.values()) {
            processor.forceCloseAllWorkingTsFileProcessors();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeStorageGroupProcessor(PartialPath storageGroupPath, boolean isSeq, boolean isSync) {
        StorageGroupProcessor processor = this.processorMap.get(storageGroupPath);
        if (processor == null) {
            return;
        }
        if (this.logger.isInfoEnabled()) {
            this.logger.info("{} closing sg processor is called for closing {}, seq = {}", new Object[]{isSync ? "sync" : "async", storageGroupPath, isSeq});
        }
        processor.writeLock();
        try {
            if (isSeq) {
                for (TsFileProcessor tsfileProcessor : new ArrayList<TsFileProcessor>(processor.getWorkSequenceTsFileProcessors())) {
                    if (isSync) {
                        processor.syncCloseOneTsFileProcessor(true, tsfileProcessor);
                        continue;
                    }
                    processor.asyncCloseOneTsFileProcessor(true, tsfileProcessor);
                }
            } else {
                for (TsFileProcessor tsfileProcessor : new ArrayList<TsFileProcessor>(processor.getWorkUnsequenceTsFileProcessors())) {
                    if (isSync) {
                        processor.syncCloseOneTsFileProcessor(false, tsfileProcessor);
                        continue;
                    }
                    processor.asyncCloseOneTsFileProcessor(false, tsfileProcessor);
                }
            }
        }
        finally {
            processor.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeStorageGroupProcessor(PartialPath storageGroupPath, long partitionId, boolean isSeq, boolean isSync) throws StorageGroupNotSetException {
        StorageGroupProcessor processor = this.processorMap.get(storageGroupPath);
        if (processor != null) {
            this.logger.info("async closing sg processor is called for closing {}, seq = {}, partitionId = {}", new Object[]{storageGroupPath, isSeq, partitionId});
            processor.writeLock();
            ArrayList<TsFileProcessor> processors = isSeq ? new ArrayList<TsFileProcessor>(processor.getWorkSequenceTsFileProcessors()) : new ArrayList<TsFileProcessor>(processor.getWorkUnsequenceTsFileProcessors());
            try {
                for (TsFileProcessor tsfileProcessor : processors) {
                    if (tsfileProcessor.getTimeRangeId() != partitionId) continue;
                    if (isSync) {
                        processor.syncCloseOneTsFileProcessor(isSeq, tsfileProcessor);
                    }
                    processor.asyncCloseOneTsFileProcessor(isSeq, tsfileProcessor);
                }
            }
            finally {
                processor.writeUnlock();
            }
        } else {
            throw new StorageGroupNotSetException(storageGroupPath.getFullPath());
        }
    }

    public void update(String deviceId, String measurementId, long startTime, long endTime, TSDataType type, String v) {
    }

    public void delete(PartialPath path, long startTime, long endTime, long planIndex) throws StorageEngineException {
        try {
            List<PartialPath> sgPaths = IoTDB.metaManager.searchAllRelatedStorageGroups(path);
            for (PartialPath storageGroupPath : sgPaths) {
                StorageGroupProcessor storageGroupProcessor = this.getProcessor(storageGroupPath);
                PartialPath newPath = path.alterPrefixPath(storageGroupPath);
                storageGroupProcessor.delete(newPath, startTime, endTime, planIndex);
            }
        }
        catch (IOException | MetadataException e) {
            throw new StorageEngineException(e.getMessage());
        }
    }

    public void deleteTimeseries(PartialPath path, long planIndex) throws StorageEngineException {
        try {
            for (PartialPath storageGroupPath : IoTDB.metaManager.searchAllRelatedStorageGroups(path)) {
                StorageGroupProcessor storageGroupProcessor = this.getProcessor(storageGroupPath);
                PartialPath newPath = path.alterPrefixPath(storageGroupPath);
                storageGroupProcessor.delete(newPath, Long.MIN_VALUE, Long.MAX_VALUE, planIndex);
            }
        }
        catch (IOException | MetadataException e) {
            throw new StorageEngineException(e.getMessage());
        }
    }

    public QueryDataSource query(SingleSeriesExpression seriesExpression, QueryContext context, QueryFileManager filePathsManager) throws StorageEngineException, QueryProcessException {
        PartialPath fullPath = (PartialPath)seriesExpression.getSeriesPath();
        PartialPath deviceId = fullPath.getDevicePath();
        String measurementId = seriesExpression.getSeriesPath().getMeasurement();
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(deviceId);
        return storageGroupProcessor.query(deviceId, measurementId, context, filePathsManager, seriesExpression.getFilter());
    }

    public int countUpgradeFiles() {
        int totalUpgradeFileNum = 0;
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            totalUpgradeFileNum += storageGroupProcessor.countUpgradeFiles();
        }
        return totalUpgradeFileNum;
    }

    public void upgradeAll() throws StorageEngineException {
        if (IoTDBDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support file upgrade");
        }
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            storageGroupProcessor.upgrade();
        }
    }

    public void mergeAll(boolean fullMerge) throws StorageEngineException {
        if (IoTDBDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support merge");
        }
        for (StorageGroupProcessor storageGroupProcessor : this.processorMap.values()) {
            storageGroupProcessor.merge(fullMerge);
        }
    }

    public void deleteAllDataFilesInOneStorageGroup(PartialPath storageGroupPath) {
        if (this.processorMap.containsKey(storageGroupPath)) {
            this.syncDeleteDataFiles(storageGroupPath);
        }
    }

    private void syncDeleteDataFiles(PartialPath storageGroupPath) {
        this.logger.info("Force to delete the data in storage group processor {}", (Object)storageGroupPath);
        StorageGroupProcessor processor = this.processorMap.get(storageGroupPath);
        processor.syncDeleteDataFiles();
    }

    public synchronized boolean deleteAll() {
        this.logger.info("Start deleting all storage groups' timeseries");
        this.syncCloseAllProcessor();
        for (PartialPath storageGroup : IoTDB.metaManager.getAllStorageGroupPaths()) {
            this.deleteAllDataFilesInOneStorageGroup(storageGroup);
        }
        return true;
    }

    public void setTTL(PartialPath storageGroup, long dataTTL) throws StorageEngineException {
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(storageGroup);
        storageGroupProcessor.setDataTTL(dataTTL);
    }

    public void deleteStorageGroup(PartialPath storageGroupPath) {
        this.deleteAllDataFilesInOneStorageGroup(storageGroupPath);
        StorageGroupProcessor processor = this.processorMap.remove(storageGroupPath);
        if (processor != null) {
            processor.deleteFolder(this.systemDir);
        }
    }

    public void loadNewTsFileForSync(TsFileResource newTsFileResource) throws StorageEngineException, LoadFileException, IllegalPathException {
        this.getProcessor(new PartialPath(newTsFileResource.getTsFile().getParentFile().getName())).loadNewTsFileForSync(newTsFileResource);
    }

    public void loadNewTsFile(TsFileResource newTsFileResource) throws LoadFileException, StorageEngineException, MetadataException {
        Map<String, Integer> deviceMap = newTsFileResource.getDeviceToIndexMap();
        if (deviceMap == null || deviceMap.isEmpty()) {
            throw new StorageEngineException("Can not get the corresponding storage group.");
        }
        String device = deviceMap.keySet().iterator().next();
        PartialPath devicePath = new PartialPath(device);
        PartialPath storageGroupPath = IoTDB.metaManager.getStorageGroupPath(devicePath);
        this.getProcessor(storageGroupPath).loadNewTsFile(newTsFileResource);
    }

    public boolean deleteTsfileForSync(File deletedTsfile) throws StorageEngineException, IllegalPathException {
        return this.getProcessor(new PartialPath(deletedTsfile.getParentFile().getName())).deleteTsfile(deletedTsfile);
    }

    public boolean deleteTsfile(File deletedTsfile) throws StorageEngineException, IllegalPathException {
        return this.getProcessor(new PartialPath(this.getSgByEngineFile(deletedTsfile))).deleteTsfile(deletedTsfile);
    }

    public boolean moveTsfile(File tsfileToBeMoved, File targetDir) throws StorageEngineException, IllegalPathException {
        return this.getProcessor(new PartialPath(this.getSgByEngineFile(tsfileToBeMoved))).moveTsfile(tsfileToBeMoved, targetDir);
    }

    private String getSgByEngineFile(File file) {
        return file.getParentFile().getParentFile().getName();
    }

    public Map<PartialPath, Map<Long, List<TsFileResource>>> getAllClosedStorageGroupTsFile() {
        HashMap<PartialPath, Map<Long, List<TsFileResource>>> ret = new HashMap<PartialPath, Map<Long, List<TsFileResource>>>();
        for (Map.Entry<PartialPath, StorageGroupProcessor> entry : this.processorMap.entrySet()) {
            List<TsFileResource> allResources = entry.getValue().getSequenceFileTreeSet();
            allResources.addAll(entry.getValue().getUnSequenceFileList());
            for (TsFileResource sequenceFile : allResources) {
                if (!sequenceFile.isClosed()) continue;
                long partitionNum = sequenceFile.getTimePartition();
                Map storageGroupFiles = ret.computeIfAbsent(entry.getKey(), n -> new HashMap());
                storageGroupFiles.computeIfAbsent(partitionNum, n -> new ArrayList()).add(sequenceFile);
            }
        }
        return ret;
    }

    public void setFileFlushPolicy(TsFileFlushPolicy fileFlushPolicy) {
        this.fileFlushPolicy = fileFlushPolicy;
    }

    public boolean isFileAlreadyExist(TsFileResource tsFileResource, PartialPath storageGroup, long partitionNum) {
        StorageGroupProcessor processor = this.processorMap.get(storageGroup);
        return processor != null && processor.isFileAlreadyExist(tsFileResource, partitionNum);
    }

    public static long getTimePartitionInterval() {
        if (timePartitionInterval == -1L) {
            StorageEngine.initTimePartition();
        }
        return timePartitionInterval;
    }

    public static void setTimePartitionInterval(long timePartitionInterval) {
        StorageEngine.timePartitionInterval = timePartitionInterval;
    }

    public static long getTimePartition(long time) {
        return enablePartition ? time / timePartitionInterval : 0L;
    }

    public void setPartitionVersionToMax(PartialPath storageGroup, long partitionId, long newMaxVersion) throws StorageEngineException {
        this.getProcessor(storageGroup).setPartitionFileVersionToMax(partitionId, newMaxVersion);
    }

    public void removePartitions(PartialPath storageGroupPath, StorageGroupProcessor.TimePartitionFilter filter) throws StorageEngineException {
        this.getProcessor(storageGroupPath).removePartitions(filter);
    }

    public Map<PartialPath, StorageGroupProcessor> getProcessorMap() {
        return this.processorMap;
    }

    public Map<String, List<Pair<Long, Boolean>>> getWorkingStorageGroupPartitions() {
        ConcurrentHashMap<String, List<Pair<Long, Boolean>>> res = new ConcurrentHashMap<String, List<Pair<Long, Boolean>>>();
        for (Map.Entry<PartialPath, StorageGroupProcessor> entry : this.processorMap.entrySet()) {
            Pair tmpPair;
            ArrayList<Pair> partitionIdList = new ArrayList<Pair>();
            StorageGroupProcessor processor = entry.getValue();
            for (TsFileProcessor tsFileProcessor : processor.getWorkSequenceTsFileProcessors()) {
                tmpPair = new Pair((Object)tsFileProcessor.getTimeRangeId(), (Object)true);
                partitionIdList.add(tmpPair);
            }
            for (TsFileProcessor tsFileProcessor : processor.getWorkUnsequenceTsFileProcessors()) {
                tmpPair = new Pair((Object)tsFileProcessor.getTimeRangeId(), (Object)false);
                partitionIdList.add(tmpPair);
            }
            res.put(entry.getKey().getFullPath(), partitionIdList);
        }
        return res;
    }

    public static void setEnablePartition(boolean enablePartition) {
        StorageEngine.enablePartition = enablePartition;
    }

    public static boolean isEnablePartition() {
        return enablePartition;
    }

    public void registerFlushListener(FlushListener listener) {
        this.customFlushListeners.add(listener);
    }

    public void registerCloseFileListener(CloseFileListener listener) {
        this.customCloseFileListeners.add(listener);
    }

    public List<StorageGroupProcessor> mergeLock(List<PartialPath> pathList) throws StorageEngineException {
        HashSet<StorageGroupProcessor> set = new HashSet<StorageGroupProcessor>();
        for (PartialPath path : pathList) {
            set.add(this.getProcessor(path));
        }
        List<StorageGroupProcessor> list = set.stream().sorted(Comparator.comparing(StorageGroupProcessor::getStorageGroupName)).collect(Collectors.toList());
        list.forEach(storageGroupProcessor -> storageGroupProcessor.getTsFileManagement().readLock());
        return list;
    }

    public void mergeUnLock(List<StorageGroupProcessor> list) {
        list.forEach(storageGroupProcessor -> storageGroupProcessor.getTsFileManagement().readUnLock());
    }

    public static void blockInsertionIfReject() throws WriteProcessRejectException {
        long startTime = System.currentTimeMillis();
        while (SystemInfo.getInstance().isRejected()) {
            try {
                TimeUnit.MILLISECONDS.sleep(config.getCheckPeriodWhenInsertBlocked());
                if (System.currentTimeMillis() - startTime <= (long)config.getMaxWaitingTimeWhenInsertBlocked()) continue;
                throw new WriteProcessRejectException("System rejected over " + config.getMaxWaitingTimeWhenInsertBlocked() + "ms");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    static class InstanceHolder {
        private static final StorageEngine INSTANCE = new StorageEngine();

        private InstanceHolder() {
        }
    }
}

