/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.user.ldap;

import org.apache.commons.configuration.ConversionException;
import org.apache.commons.configuration.HierarchicalConfiguration;
import org.apache.commons.configuration.plist.PropertyListConfiguration;
import org.apache.james.core.MailAddress;
import org.apache.james.domainlist.api.DomainList;
import org.apache.james.user.ldap.DockerLdapSingleton;
import org.apache.james.user.ldap.ReadOnlyUsersLDAPRepository;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReadOnlyUsersLDAPRepositoryTest {
    private static final Logger LOGGER = LoggerFactory.getLogger(ReadOnlyUsersLDAPRepositoryTest.class);
    private static final String JAMES_USER_MAIL = "james-user@james.org";
    private static final String UNKNOWN = "unknown";
    private static final String BAD_PASSWORD = "badpassword";
    private DomainList domainList;

    @BeforeEach
    void setUp() {
        this.domainList = (DomainList)Mockito.mock(DomainList.class);
    }

    private static HierarchicalConfiguration ldapRepositoryConfiguration() {
        PropertyListConfiguration configuration = new PropertyListConfiguration();
        configuration.addProperty("[@ldapHost]", (Object)DockerLdapSingleton.ldapContainer.getLdapHost());
        configuration.addProperty("[@principal]", (Object)"cn=admin\\,dc=james\\,dc=org");
        configuration.addProperty("[@credentials]", (Object)"mysecretpassword");
        configuration.addProperty("[@userBase]", (Object)"ou=People\\,dc=james\\,dc=org");
        configuration.addProperty("[@userIdAttribute]", (Object)"uid");
        configuration.addProperty("[@userObjectClass]", (Object)"inetOrgPerson");
        configuration.addProperty("[@maxRetries]", (Object)"4");
        configuration.addProperty("[@retryStartInterval]", (Object)"0");
        configuration.addProperty("[@retryMaxInterval]", (Object)"8");
        configuration.addProperty("[@retryIntervalScale]", (Object)"1000");
        return configuration;
    }

    private static HierarchicalConfiguration ldapRepositoryConfigurationWithVirtualHosting() {
        PropertyListConfiguration configuration = new PropertyListConfiguration();
        configuration.addProperty("[@ldapHost]", (Object)DockerLdapSingleton.ldapContainer.getLdapHost());
        configuration.addProperty("[@principal]", (Object)"cn=admin\\,dc=james\\,dc=org");
        configuration.addProperty("[@credentials]", (Object)"mysecretpassword");
        configuration.addProperty("[@userBase]", (Object)"ou=People\\,dc=james\\,dc=org");
        configuration.addProperty("[@userIdAttribute]", (Object)"mail");
        configuration.addProperty("[@userObjectClass]", (Object)"inetOrgPerson");
        configuration.addProperty("[@maxRetries]", (Object)"4");
        configuration.addProperty("[@retryStartInterval]", (Object)"0");
        configuration.addProperty("[@retryMaxInterval]", (Object)"8");
        configuration.addProperty("[@retryIntervalScale]", (Object)"1000");
        configuration.addProperty("supportsVirtualHosting", (Object)true);
        return configuration;
    }

    @Nested
    class TestUser {
        TestUser() {
        }

        @Test
        public void knownUserShouldBeAbleToLogInWhenPasswordIsCorrect() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration());
            Assertions.assertThat((boolean)ldapRepository.test("james-user", "secret")).isTrue();
        }

        @Test
        public void knownUserShouldNotBeAbleToLogInWhenPasswordIsNotCorrect() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration());
            Assertions.assertThat((boolean)ldapRepository.test("james-user", ReadOnlyUsersLDAPRepositoryTest.BAD_PASSWORD)).isFalse();
        }

        @Test
        public void unknownUserShouldNotBeAbleToLogIn() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration());
            Assertions.assertThat((boolean)ldapRepository.test(ReadOnlyUsersLDAPRepositoryTest.UNKNOWN, ReadOnlyUsersLDAPRepositoryTest.BAD_PASSWORD)).isFalse();
        }

        @Test
        public void unknownUserShouldNotBeAbleToLogInWhenPasswordIsCorrect() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration());
            Assertions.assertThat((boolean)ldapRepository.test(ReadOnlyUsersLDAPRepositoryTest.UNKNOWN, "secret")).isFalse();
        }

        @Test
        public void knownUserShouldBeAbleToLogInWhenPasswordIsCorrectWithVirtualHosting() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting());
            Assertions.assertThat((boolean)ldapRepository.test(ReadOnlyUsersLDAPRepositoryTest.JAMES_USER_MAIL, "secret")).isTrue();
        }

        @Test
        public void testShouldStillWorksAfterRestartingLDAP() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting());
            ldapRepository.test(ReadOnlyUsersLDAPRepositoryTest.JAMES_USER_MAIL, "secret");
            DockerLdapSingleton.ldapContainer.pause();
            try {
                ldapRepository.test(ReadOnlyUsersLDAPRepositoryTest.JAMES_USER_MAIL, "secret");
            }
            catch (Exception e) {
                LOGGER.info("This exception is expected as we shut down the LDAP and forced its use", (Throwable)e);
            }
            DockerLdapSingleton.ldapContainer.unpause();
            Assertions.assertThat((boolean)ldapRepository.test(ReadOnlyUsersLDAPRepositoryTest.JAMES_USER_MAIL, "secret")).isTrue();
        }

        @Test
        public void knownUserShouldNotBeAbleToLogInWhenPasswordIsNotCorrectWithVirtualHosting() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting());
            Assertions.assertThat((boolean)ldapRepository.test("james-user", ReadOnlyUsersLDAPRepositoryTest.BAD_PASSWORD)).isFalse();
        }

        @Test
        public void unknownUserShouldNotBeAbleToLogInWithVirtualHosting() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting());
            Assertions.assertThat((boolean)ldapRepository.test(ReadOnlyUsersLDAPRepositoryTest.UNKNOWN, ReadOnlyUsersLDAPRepositoryTest.BAD_PASSWORD)).isFalse();
        }

        @Test
        public void unknownUserShouldNotBeAbleToLogInWhenPasswordIsCorrectWithVirtualHosting() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting());
            Assertions.assertThat((boolean)ldapRepository.test(ReadOnlyUsersLDAPRepositoryTest.UNKNOWN, "secret")).isFalse();
        }

        @Test
        public void specialCharacterInUserInputShouldBeSanitized() throws Exception {
            String patternMatchingMultipleUsers = "j*";
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting());
            Assertions.assertThat((boolean)ldapRepository.test(patternMatchingMultipleUsers, "secret")).isFalse();
        }

        @Test
        public void containsWithGetUserShouldBeTrue() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration());
            Assertions.assertThat((boolean)ldapRepository.contains(ldapRepository.getUser(new MailAddress(ReadOnlyUsersLDAPRepositoryTest.JAMES_USER_MAIL)))).isTrue();
        }

        @Test
        public void containsWithGetUserShouldBeTrueWithVirtualHosting() throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = this.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting());
            Assertions.assertThat((boolean)ldapRepository.contains(ldapRepository.getUser(new MailAddress(ReadOnlyUsersLDAPRepositoryTest.JAMES_USER_MAIL)))).isTrue();
        }

        private ReadOnlyUsersLDAPRepository startUsersRepository(HierarchicalConfiguration ldapRepositoryConfiguration) throws Exception {
            ReadOnlyUsersLDAPRepository ldapRepository = new ReadOnlyUsersLDAPRepository(ReadOnlyUsersLDAPRepositoryTest.this.domainList);
            ldapRepository.configure(ldapRepositoryConfiguration);
            ldapRepository.init();
            return ldapRepository;
        }
    }

    @Nested
    class SupportVirtualHosting {
        SupportVirtualHosting() {
        }

        @Test
        public void supportVirtualHostingShouldReturnFalseByDefault() throws Exception {
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository(ReadOnlyUsersLDAPRepositoryTest.this.domainList);
            usersLDAPRepository.configure(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration());
            Assertions.assertThat((boolean)usersLDAPRepository.supportVirtualHosting()).isFalse();
        }

        @Test
        public void supportVirtualHostingShouldReturnTrueWhenReportedInConfig() throws Exception {
            HierarchicalConfiguration configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration();
            configuration.addProperty("supportsVirtualHosting", (Object)"true");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository(ReadOnlyUsersLDAPRepositoryTest.this.domainList);
            usersLDAPRepository.configure(configuration);
            Assertions.assertThat((boolean)usersLDAPRepository.supportVirtualHosting()).isTrue();
        }

        @Test
        public void supportVirtualHostingShouldReturnFalseWhenReportedInConfig() throws Exception {
            HierarchicalConfiguration configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration();
            configuration.addProperty("supportsVirtualHosting", (Object)"false");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository(ReadOnlyUsersLDAPRepositoryTest.this.domainList);
            usersLDAPRepository.configure(configuration);
            Assertions.assertThat((boolean)usersLDAPRepository.supportVirtualHosting()).isFalse();
        }

        @Test
        public void configureShouldThrowOnNonBooleanValueForSupportsVirtualHosting() throws Exception {
            HierarchicalConfiguration configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration();
            configuration.addProperty("supportsVirtualHosting", (Object)"bad");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository(ReadOnlyUsersLDAPRepositoryTest.this.domainList);
            Assertions.assertThatThrownBy(() -> usersLDAPRepository.configure(configuration)).isInstanceOf(ConversionException.class);
        }
    }
}

