/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.user.ldap;

import com.google.common.collect.ImmutableList;
import com.unboundid.ldap.sdk.LDAPException;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import org.apache.commons.configuration2.HierarchicalConfiguration;
import org.apache.commons.configuration2.ex.ConversionException;
import org.apache.commons.configuration2.plist.PropertyListConfiguration;
import org.apache.commons.configuration2.tree.ImmutableNode;
import org.apache.james.core.Username;
import org.apache.james.domainlist.api.DomainList;
import org.apache.james.domainlist.api.mock.SimpleDomainList;
import org.apache.james.user.api.UsersRepository;
import org.apache.james.user.ldap.DockerLdapSingleton;
import org.apache.james.user.ldap.LdapGenericContainer;
import org.apache.james.user.ldap.ReadOnlyLDAPUsersDAO;
import org.apache.james.user.ldap.ReadOnlyUsersLDAPRepository;
import org.apache.james.user.lib.UsersRepositoryContract;
import org.apache.james.user.lib.UsersRepositoryImpl;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testcontainers.shaded.org.awaitility.Awaitility;

class ReadOnlyUsersLDAPRepositoryTest {
    static final Logger LOGGER = LoggerFactory.getLogger(ReadOnlyUsersLDAPRepositoryTest.class);
    static final Username JAMES_USER_MAIL = Username.of((String)(DockerLdapSingleton.JAMES_USER.getLocalPart() + "@james.org"));
    static final Username UNKNOWN = Username.of((String)"unknown");
    static final String BAD_PASSWORD = "badpassword";
    public static final String SUPPORTS_VIRTUAL_HOSTING = "supportsVirtualHosting";
    static LdapGenericContainer ldapContainer = LdapGenericContainer.builder().domain("james.org").password("mysecretpassword").build();

    ReadOnlyUsersLDAPRepositoryTest() {
    }

    @BeforeAll
    static void setUpAll() {
        ldapContainer.start();
    }

    @AfterAll
    static void afterAll() {
        ldapContainer.stop();
    }

    @Test
    void shouldNotStartWithInvalidFilter() throws Exception {
        HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
        configuration.addProperty("[@filter]", (Object)"INVALID!!!");
        ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
        usersLDAPRepository.configure(configuration);
        Assertions.assertThatThrownBy(() -> ((ReadOnlyUsersLDAPRepository)usersLDAPRepository).init()).isInstanceOf(LDAPException.class);
    }

    @Test
    void shouldThrowWhenLdapsOnLdapPort() throws Exception {
        HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
        configuration.addProperty("[@ldapHost]", (Object)ldapContainer.getLdapsBadHost());
        ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
        usersLDAPRepository.configure(configuration);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ReadOnlyUsersLDAPRepository)usersLDAPRepository).init()).isInstanceOf(LDAPException.class)).hasMessageContaining("SSLHandshakeException");
    }

    @Test
    void shouldThrowWhenLdapsOnLdapPortWithoutCertificateValidation() throws Exception {
        HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
        configuration.addProperty("[@ldapHost]", (Object)ldapContainer.getLdapsBadHost());
        configuration.addProperty("[@trustAllCerts]", (Object)"true");
        Awaitility.await().untilAsserted(() -> {
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            usersLDAPRepository.configure(configuration);
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ReadOnlyUsersLDAPRepository)usersLDAPRepository).init()).isInstanceOf(LDAPException.class)).hasMessageContaining("SSLHandshakeException");
        });
    }

    private static ReadOnlyUsersLDAPRepository startUsersRepository(HierarchicalConfiguration<ImmutableNode> ldapRepositoryConfiguration, DomainList domainList) throws Exception {
        ReadOnlyUsersLDAPRepository ldapRepository = new ReadOnlyUsersLDAPRepository(domainList);
        ldapRepository.configure(ldapRepositoryConfiguration);
        ldapRepository.init();
        return ldapRepository;
    }

    static HierarchicalConfiguration<ImmutableNode> ldapRepositoryConfiguration(LdapGenericContainer ldapContainer) {
        return ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer, Optional.of(Username.of((String)"admin")));
    }

    static HierarchicalConfiguration<ImmutableNode> ldapRepositoryConfiguration(LdapGenericContainer ldapContainer, Optional<Username> administrator) {
        PropertyListConfiguration configuration = ReadOnlyUsersLDAPRepositoryTest.baseConfiguration(ldapContainer);
        configuration.addProperty("[@userIdAttribute]", (Object)"uid");
        administrator.ifPresent(username -> configuration.addProperty("[@administratorId]", (Object)username.asString()));
        return configuration;
    }

    static HierarchicalConfiguration<ImmutableNode> ldapRepositoryConfigurationWithVirtualHosting(LdapGenericContainer ldapContainer) {
        return ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting(ldapContainer, Optional.of(DockerLdapSingleton.ADMIN));
    }

    static HierarchicalConfiguration<ImmutableNode> ldapRepositoryConfigurationWithVirtualHosting(LdapGenericContainer ldapContainer, Optional<Username> administrator) {
        PropertyListConfiguration configuration = ReadOnlyUsersLDAPRepositoryTest.baseConfiguration(ldapContainer);
        configuration.addProperty("[@userIdAttribute]", (Object)"mail");
        configuration.addProperty(SUPPORTS_VIRTUAL_HOSTING, (Object)true);
        administrator.ifPresent(username -> configuration.addProperty("[@administratorId]", (Object)username.asString()));
        return configuration;
    }

    private static PropertyListConfiguration baseConfiguration(LdapGenericContainer ldapContainer) {
        PropertyListConfiguration configuration = new PropertyListConfiguration();
        configuration.addProperty("[@ldapHost]", (Object)ldapContainer.getLdapHost());
        configuration.addProperty("[@principal]", (Object)"cn=admin,dc=james,dc=org");
        configuration.addProperty("[@credentials]", (Object)"mysecretpassword");
        configuration.addProperty("[@userBase]", (Object)"ou=people,dc=james,dc=org");
        configuration.addProperty("[@userObjectClass]", (Object)"inetOrgPerson");
        configuration.addProperty("[@connectionTimeout]", (Object)"100");
        configuration.addProperty("[@readTimeout]", (Object)"100");
        return configuration;
    }

    @Nested
    class SupportVirtualHosting {
        SupportVirtualHosting() {
        }

        @Test
        void supportVirtualHostingShouldReturnFalseByDefault() throws Exception {
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            usersLDAPRepository.configure(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer));
            Assertions.assertThat((boolean)usersLDAPRepository.supportVirtualHosting()).isFalse();
        }

        @Test
        void supportVirtualHostingShouldReturnTrueWhenReportedInConfig() throws Exception {
            HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
            configuration.addProperty(ReadOnlyUsersLDAPRepositoryTest.SUPPORTS_VIRTUAL_HOSTING, (Object)"true");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            usersLDAPRepository.configure(configuration);
            Assertions.assertThat((boolean)usersLDAPRepository.supportVirtualHosting()).isTrue();
        }

        @Test
        void supportVirtualHostingShouldReturnFalseWhenReportedInConfig() throws Exception {
            HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
            configuration.addProperty(ReadOnlyUsersLDAPRepositoryTest.SUPPORTS_VIRTUAL_HOSTING, (Object)"false");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            usersLDAPRepository.configure(configuration);
            Assertions.assertThat((boolean)usersLDAPRepository.supportVirtualHosting()).isFalse();
        }

        @Test
        void configureShouldThrowOnNonBooleanValueForSupportsVirtualHosting() {
            HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
            configuration.addProperty(ReadOnlyUsersLDAPRepositoryTest.SUPPORTS_VIRTUAL_HOSTING, (Object)"bad");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            Assertions.assertThatThrownBy(() -> usersLDAPRepository.configure(configuration)).isInstanceOf(ConversionException.class);
        }
    }

    @Nested
    class WhenDisableVirtualHosting
    implements UsersRepositoryContract.WithOutVirtualHostingReadOnlyContract {
        @RegisterExtension
        UsersRepositoryContract.UserRepositoryExtension extension = UsersRepositoryContract.UserRepositoryExtension.withoutVirtualHosting();
        private ReadOnlyUsersLDAPRepository usersRepository;
        private UsersRepositoryContract.TestSystem testSystem;

        WhenDisableVirtualHosting() {
        }

        @BeforeEach
        void setUp(UsersRepositoryContract.TestSystem testSystem) throws Exception {
            this.usersRepository = ReadOnlyUsersLDAPRepositoryTest.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer), (DomainList)testSystem.getDomainList());
            this.testSystem = testSystem;
        }

        public UsersRepositoryImpl<ReadOnlyLDAPUsersDAO> testee() {
            return this.usersRepository;
        }

        public UsersRepository testee(Optional<Username> administrator) throws Exception {
            return ReadOnlyUsersLDAPRepositoryTest.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer, administrator), (DomainList)this.testSystem.getDomainList());
        }

        @Test
        void knownUserShouldBeAbleToLogInWhenPasswordIsCorrect() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.test(DockerLdapSingleton.JAMES_USER, "secret")).isTrue();
        }

        @Test
        void listShouldReturnExistingUsers() throws Exception {
            Assertions.assertThat((List)ImmutableList.copyOf((Iterator)this.usersRepository.list())).containsOnly((Object[])new Username[]{DockerLdapSingleton.JAMES_USER});
        }

        @Test
        void countUsersShouldReturnOne() throws Exception {
            Assertions.assertThat((int)this.usersRepository.countUsers()).isEqualTo(1);
        }

        @Test
        void containsShouldReturnTrueWhenUserExists() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.contains(DockerLdapSingleton.JAMES_USER)).isTrue();
        }

        @Test
        void containsShouldReturnFalseWhenUserDoesNotExists() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.contains(Username.of((String)"unknown"))).isFalse();
        }

        @Test
        void knownUserShouldNotBeAbleToLogInWhenPasswordIsNotCorrect() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.test(DockerLdapSingleton.JAMES_USER, ReadOnlyUsersLDAPRepositoryTest.BAD_PASSWORD)).isFalse();
        }

        @Test
        void unknownUserShouldNotBeAbleToLogIn() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.test(UNKNOWN, ReadOnlyUsersLDAPRepositoryTest.BAD_PASSWORD)).isFalse();
        }

        @Test
        void unknownUserShouldNotBeAbleToLogInWhenPasswordIsCorrect() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.test(UNKNOWN, "secret")).isFalse();
        }

        @Test
        void containsWithGetUserShouldBeTrue() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.contains(this.usersRepository.getUsername(JAMES_USER_MAIL.asMailAddress()))).isTrue();
        }

        @Test
        void isAdministratorShouldReturnTrueWhenConfiguredAndUserIsAdmin(UsersRepositoryContract.TestSystem testSystem) throws Exception {
            Assertions.assertThat((boolean)this.testee().isAdministrator(testSystem.getAdmin())).isTrue();
        }

        @Disabled(value="JAMES-3088 Users are provisioned by default from Dockerfile, cannot setup this test case,See @link{ReadOnlyUsersLDAPRepositoryEmptyListTest}")
        @Test
        public void listShouldReturnEmptyIteratorWhenEmptyRepository(UsersRepositoryContract.TestSystem testSystem) {
        }

        @Disabled(value="JAMES-3088 Users are provisioned by default from Dockerfile, cannot setup this test case,See @link{ReadOnlyUsersLDAPRepositoryEmptyListTest}")
        @Test
        public void countUsersShouldReturnZeroWhenEmptyRepository() {
        }
    }

    @Nested
    class WhenEnableVirtualHosting
    implements UsersRepositoryContract.WithVirtualHostingReadOnlyContract {
        @RegisterExtension
        UsersRepositoryContract.UserRepositoryExtension extension = UsersRepositoryContract.UserRepositoryExtension.withVirtualHost();
        private ReadOnlyUsersLDAPRepository usersRepository;
        private UsersRepositoryContract.TestSystem testSystem;

        WhenEnableVirtualHosting() {
        }

        @BeforeEach
        void setUp(UsersRepositoryContract.TestSystem testSystem) throws Exception {
            this.usersRepository = ReadOnlyUsersLDAPRepositoryTest.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting(ldapContainer), (DomainList)testSystem.getDomainList());
            this.testSystem = testSystem;
        }

        public UsersRepositoryImpl<ReadOnlyLDAPUsersDAO> testee() {
            return this.usersRepository;
        }

        public UsersRepository testee(Optional<Username> administrator) throws Exception {
            return ReadOnlyUsersLDAPRepositoryTest.startUsersRepository(ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting(ldapContainer, administrator), (DomainList)this.testSystem.getDomainList());
        }

        @Test
        void isAdministratorShouldReturnTrueWhenConfiguredAndUserIsAdmin(UsersRepositoryContract.TestSystem testSystem) throws Exception {
            Assertions.assertThat((boolean)this.testee().isAdministrator(testSystem.getAdmin())).isTrue();
        }

        @Test
        void knownUserShouldBeAbleToLogInWhenPasswordIsCorrectWithVirtualHosting() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.test(JAMES_USER_MAIL, "secret")).isTrue();
        }

        @Test
        void listShouldReturnExistingUsers() throws Exception {
            Assertions.assertThat((List)ImmutableList.copyOf((Iterator)this.usersRepository.list())).containsOnly((Object[])new Username[]{JAMES_USER_MAIL});
        }

        @Test
        void countUsersShouldReturnOne() throws Exception {
            Assertions.assertThat((int)this.usersRepository.countUsers()).isEqualTo(1);
        }

        @Test
        void containsShouldReturnTrueWhenUserExists() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.contains(JAMES_USER_MAIL)).isTrue();
        }

        @Test
        void containsShouldReturnFalseWhenUserDoesNotExists() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.contains(Username.of((String)"unknown@james.org"))).isFalse();
        }

        @Test
        void testShouldStillWorkAfterRestartingLDAP() throws Exception {
            this.usersRepository.test(JAMES_USER_MAIL, "secret");
            DockerLdapSingleton.ldapContainer.pause();
            try {
                this.usersRepository.test(JAMES_USER_MAIL, "secret");
            }
            catch (Exception e) {
                LOGGER.info("This exception is expected as we shut down the LDAP and forced its use", (Throwable)e);
            }
            DockerLdapSingleton.ldapContainer.unpause();
            Assertions.assertThat((boolean)this.usersRepository.test(JAMES_USER_MAIL, "secret")).isTrue();
        }

        @Test
        void knownUserShouldNotBeAbleToLogInWhenPasswordIsNotCorrectWithVirtualHosting() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.test(DockerLdapSingleton.JAMES_USER, ReadOnlyUsersLDAPRepositoryTest.BAD_PASSWORD)).isFalse();
        }

        @Test
        void unknownUserShouldNotBeAbleToLogInWhenPasswordIsCorrectWithVirtualHosting() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.test(UNKNOWN, "secret")).isFalse();
        }

        @Test
        void specialCharacterInUserInputShouldBeSanitized() throws Exception {
            Username patternMatchingMultipleUsers = Username.of((String)"j*");
            Assertions.assertThat((boolean)this.usersRepository.test(patternMatchingMultipleUsers, "secret")).isFalse();
        }

        @Test
        void containsWithGetUserShouldBeTrueWithVirtualHosting() throws Exception {
            Assertions.assertThat((boolean)this.usersRepository.contains(this.usersRepository.getUsername(JAMES_USER_MAIL.asMailAddress()))).isTrue();
        }

        @Disabled(value="JAMES-3088 Users are provisioned by default from Dockerfile, cannot setup this test case,See @link{ReadOnlyUsersLDAPRepositoryEmptyListTest}")
        @Test
        public void listShouldReturnEmptyIteratorWhenEmptyRepository(UsersRepositoryContract.TestSystem testSystem) {
        }

        @Disabled(value="JAMES-3088 Users are provisioned by default from Dockerfile, cannot setup this test case,See @link{ReadOnlyUsersLDAPRepositoryEmptyListTest}")
        @Test
        public void countUsersShouldReturnZeroWhenEmptyRepository() {
        }
    }

    @Nested
    class FilterTests {
        FilterTests() {
        }

        @Test
        void filterShouldKeepMatchingEntries() throws Exception {
            HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
            configuration.addProperty("[@filter]", (Object)"(sn=james-user)");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            usersLDAPRepository.configure(configuration);
            usersLDAPRepository.init();
            Assertions.assertThat((boolean)usersLDAPRepository.contains(DockerLdapSingleton.JAMES_USER)).isTrue();
        }

        @Test
        void filterShouldFilterOutNonMatchingEntries() throws Exception {
            HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
            configuration.addProperty("[@filter]", (Object)"(sn=nomatch)");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            usersLDAPRepository.configure(configuration);
            usersLDAPRepository.init();
            Assertions.assertThat((boolean)usersLDAPRepository.contains(DockerLdapSingleton.JAMES_USER)).isFalse();
        }

        @Test
        void countShouldTakeFilterIntoAccount() throws Exception {
            HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
            configuration.addProperty("[@filter]", (Object)"(sn=nomatch)");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            usersLDAPRepository.configure(configuration);
            usersLDAPRepository.init();
            Assertions.assertThat((int)usersLDAPRepository.countUsers()).isEqualTo(0);
        }

        @Test
        void listShouldTakeFilterIntoAccount() throws Exception {
            HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfiguration(ldapContainer);
            configuration.addProperty("[@filter]", (Object)"(sn=nomatch)");
            ReadOnlyUsersLDAPRepository usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            usersLDAPRepository.configure(configuration);
            usersLDAPRepository.init();
            Assertions.assertThat((List)ImmutableList.copyOf((Iterator)usersLDAPRepository.list())).isEmpty();
        }
    }

    @Nested
    class ExtraDNTests {
        private ReadOnlyUsersLDAPRepository usersLDAPRepository;

        ExtraDNTests() {
        }

        @BeforeEach
        void setUp() throws Exception {
            HierarchicalConfiguration<ImmutableNode> configuration = ReadOnlyUsersLDAPRepositoryTest.ldapRepositoryConfigurationWithVirtualHosting(ldapContainer);
            configuration.addProperty("domains.extra.org", (Object)"ou=whatever,dc=james,dc=org");
            this.usersLDAPRepository = new ReadOnlyUsersLDAPRepository((DomainList)new SimpleDomainList());
            this.usersLDAPRepository.configure(configuration);
            this.usersLDAPRepository.init();
        }

        @Test
        void shouldContainMasterDomain() throws Exception {
            Assertions.assertThat((boolean)this.usersLDAPRepository.contains(JAMES_USER_MAIL)).isTrue();
        }

        @Test
        void shouldRejectUnhandledDomain() throws Exception {
            Assertions.assertThat((boolean)this.usersLDAPRepository.contains(Username.of((String)"bob@nonexistant.org"))).isFalse();
        }

        @Test
        void shouldContainEntriesInExtraDN() throws Exception {
            Assertions.assertThat((boolean)this.usersLDAPRepository.contains(Username.of((String)"bob@extra.org"))).isTrue();
            Assertions.assertThat((int)this.usersLDAPRepository.countUsers()).isEqualTo(2);
            Assertions.assertThat((List)ImmutableList.copyOf((Iterator)this.usersLDAPRepository.list())).containsOnly((Object[])new Username[]{JAMES_USER_MAIL, Username.of((String)"bob@extra.org")});
        }

        @Test
        void shouldCountUsersInBothOrgs() throws Exception {
            Assertions.assertThat((int)this.usersLDAPRepository.countUsers()).isEqualTo(2);
        }

        @Test
        void shouldListUsersInBothOrgs() throws Exception {
            Assertions.assertThat((List)ImmutableList.copyOf((Iterator)this.usersLDAPRepository.list())).containsOnly((Object[])new Username[]{JAMES_USER_MAIL, Username.of((String)"bob@extra.org")});
        }
    }
}

