package org.apache.maven.plugin.pmd;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

/**
 * Base class for mojos that check if there were any PMD violations.
 *
 * @author <a href="mailto:brett@apache.org">Brett Porter</a>
 * @version $Id: AbstractPmdViolationCheckMojo.java 1233066 2012-01-18 20:59:15Z rfscholte $
 */
public abstract class AbstractPmdViolationCheckMojo<D>
    extends AbstractMojo
{
    /**
     * The location of the XML report to check, as generated by the PMD report.
     *
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private File targetDirectory;

    /**
     * Whether to fail the build if the validation check fails.
     *
     * @parameter expression="${pmd.failOnViolation}" default-value="true"
     * @required
     */
    private boolean failOnViolation;

    /**
     * The project language, for determining whether to run the report.
     *
     * @parameter expression="${project.artifact.artifactHandler.language}"
     * @required
     * @readonly
     */
    private String language;

    /**
     * Whether to build an aggregated report at the root, or build individual reports.
     *
     * @parameter expression="${aggregate}" default-value="false"
     * @since 2.2
     */
    protected boolean aggregate;

    /**
     * Print details of check failures to build output.
     *
     * @parameter expression="${pmd.verbose}" default-value="false"
     */
    private boolean verbose;

    /**
     * The project to analyze.
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    protected MavenProject project;

    protected void executeCheck( String filename, String tagName, String key, int failurePriority )
        throws MojoFailureException, MojoExecutionException
    {
        if ( aggregate && !project.isExecutionRoot() )
        {
            return;
        }

        if ( "java".equals( language ) || aggregate )
        {
            File outputFile = new File( targetDirectory, filename );

            if ( outputFile.exists() )
            {
                Reader reader = null;
                try
                {
                    ViolationDetails<D> violations = getViolations( outputFile, failurePriority );

                    List<D> failures = violations.getFailureDetails();
                    List<D> warnings = violations.getWarningDetails();

                    if ( verbose )
                    {
                        printErrors( failures, warnings );
                    }

                    int failureCount = failures.size();
                    int warningCount = warnings.size();

                    String message = getMessage( failureCount, warningCount, key, outputFile );

                    if ( failureCount > 0 && failOnViolation )
                    {
                        throw new MojoFailureException( message );
                    }

                    this.getLog().info( message );
                }
                catch ( IOException e )
                {
                    throw new MojoExecutionException(
                                                      "Unable to read PMD results xml: " + outputFile.getAbsolutePath(),
                                                      e );
                }
                catch ( XmlPullParserException e )
                {
                    throw new MojoExecutionException(
                                                      "Unable to read PMD results xml: " + outputFile.getAbsolutePath(),
                                                      e );
                }
                finally
                {
                    IOUtil.close( reader );
                }
            }
            else
            {
                throw new MojoFailureException( "Unable to perform check, " + "unable to find " + outputFile );
            }
        }
    }

    /**
     * Method for collecting the violations found by the PMD tool
     *
     * @param xpp
     *            the xml parser object
     * @param tagName
     *            the element that will be checked
     * @return an int that specifies the number of violations found
     * @throws XmlPullParserException
     * @throws IOException
     */
    private ViolationDetails<D> getViolations( File analysisFile, int failurePriority )
        throws XmlPullParserException, IOException
    {
        List<D> failures = new ArrayList<D>();
        List<D> warnings = new ArrayList<D>();

        List<D> violations = getErrorDetails( analysisFile );
        
        for( D violation : violations )
        {
            int priority = getPriority( violation );
            if ( priority <= failurePriority )
            {
                failures.add( violation );
            }
            else
            {
                warnings.add( violation );
            }
        }
        
        ViolationDetails<D> details = newViolationDetailsInstance();
        details.setFailureDetails( failures );
        details.setWarningDetails( warnings );
        return details;
    }
    
    protected abstract int getPriority( D errorDetail );
    
    protected abstract ViolationDetails<D> newViolationDetailsInstance();

    /**
     * Prints the warnings and failures
     *
     * @param failures
     *            list of failures
     * @param warnings
     *            list of warnings
     */
    protected void printErrors( List<D> failures, List<D> warnings )
    {
        for ( D warning :  warnings )
        {
            printError( warning, "Warning" );
        }

        for ( D failure : failures )
        {
            printError( failure, "Failure" );
        }
    }

    /**
     * Gets the output message
     *
     * @param failureCount
     * @param warningCount
     * @param key
     * @param outputFile
     * @return
     */
    private String getMessage( int failureCount, int warningCount, String key, File outputFile )
    {
        StringBuffer message = new StringBuffer();
        if ( failureCount > 0 || warningCount > 0 )
        {
            if ( failureCount > 0 )
            {
                message.append( "You have " + failureCount + " " + key + ( failureCount > 1 ? "s" : "" ) );
            }

            if ( warningCount > 0 )
            {
                if ( failureCount > 0 )
                {
                    message.append( " and " );
                }
                else
                {
                    message.append( "You have " );
                }
                message.append( warningCount + " warning" + ( warningCount > 1 ? "s" : "" ) );
            }

            message.append( ". For more details see:" ).append( outputFile.getAbsolutePath() );
        }
        return message.toString();
    }

    /**
     * Formats the failure details and prints them as an INFO message
     *
     * @param item
     */
    protected abstract void printError( D item, String severity );

    /**
     * Gets the attributes and text for the violation tag and puts them in a
     * HashMap
     *
     * @param xpp
     * @throws XmlPullParserException
     * @throws IOException
     */
    protected abstract List<D> getErrorDetails( File analisysFile )
        throws XmlPullParserException, IOException;
}