/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.skywalking.oap.meter.analyzer.dsl;

import com.google.common.collect.ImmutableMap;
import java.util.Arrays;
import java.util.Collection;
import lombok.extern.slf4j.Slf4j;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import static com.google.common.collect.ImmutableMap.of;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.fail;

@Slf4j
@RunWith(Parameterized.class)
public class ArithmeticTest {

    @Parameterized.Parameter
    public String name;

    @Parameterized.Parameter(1)
    public ImmutableMap<String, SampleFamily> input;

    @Parameterized.Parameter(2)
    public String expression;

    @Parameterized.Parameter(3)
    public Result want;

    @Parameterized.Parameter(4)
    public boolean isThrow;

    @Parameterized.Parameters(name = "{index}: {0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][] {
            {
                "plus-scalar-1",
                of("instance_cpu_percentage", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592418480.0).build(),
                    Sample.builder().labels(of("idc", "t2")).value(1600592418481.0).build()
                )),
                "1000 + instance_cpu_percentage.tagEqual('idc','t1')",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592419480.0).build()
                )),
                false,
            },
            {
                "plus-scalar",
                of("instance_cpu_percentage", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592418480.0).build(),
                    Sample.builder().labels(of("idc", "t2")).value(1600592418481.0).build()
                )),
                "instance_cpu_percentage.tagEqual('idc','t1') + 1000",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592419480.0).build()
                )),
                false,
            },
            {
                "minus-scalar",
                of("instance_cpu_percentage", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592418480.0).build(),
                    Sample.builder().labels(of("idc", "t2")).value(1600592418481.0).build()
                )),
                "instance_cpu_percentage.tagEqual('idc','t1') - 1000",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592417480.0).build()
                )),
                false,
            },
            {
                "multiply-scalar",
                of("instance_cpu_percentage", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592418480.0).build(),
                    Sample.builder().labels(of("idc", "t2")).value(1600592418481.0).build()
                )),
                "instance_cpu_percentage.tagEqual('idc','t1') * 1000",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592418480000.0).build()
                )),
                false,
            },
            {
                "divide-scalar",
                of("instance_cpu_percentage", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592418480.0).build(),
                    Sample.builder().labels(of("idc", "t2")).value(1600592418481.0).build()
                )),
                "instance_cpu_percentage.tagEqual('idc','t1') / 10",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(160059241848.0).build()
                )),
                false,
            },
            {
                "divide-zero",
                of("instance_cpu_percentage", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(1600592418480.0).build(),
                    Sample.builder().labels(of("idc", "t2")).value(1600592418481.0).build()
                )),
                "instance_cpu_percentage.tagEqual('idc','t1') / 0",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(Double.POSITIVE_INFINITY).build()
                )),
                false,
            },
            {
                "empty-plus-empty",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.EMPTY),
                "http_success_request + http_error_request",
                Result.fail(),
                false,
            },
            {
                "empty-plus-sampleFamily",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.build(
                        Sample.builder().labels(of("idc", "t1")).value(50).build(),
                        Sample.builder().labels(of("idc", "t2")).value(3).build()
                    )),
                "http_success_request + http_error_request",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(3).build()
                )),
                false,
            },
            {
                "sampleFamily-plus-empty",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.build(
                        Sample.builder().labels(of("idc", "t1")).value(50).build(),
                        Sample.builder().labels(of("idc", "t2")).value(3).build()
                    )),
                "http_error_request + http_success_request ",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(3).build()
                )),
                false,
            },
            {
                "sampleFamily-plus-sampleFamily",
                of("http_success_request", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(100).build(),
                    Sample.builder().labels(of("idc", "t2")).value(30).build(),
                    Sample.builder().labels(of("idc", "t3")).value(40).build(),
                    Sample.builder().labels(of("region", "us")).value(80).build()
                ), "http_error_request", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(3).build(),
                    Sample.builder().labels(of("idc", "t5")).value(3).build(),
                    Sample.builder().labels(of("tz", "en-US")).value(3).build()
                )),
                "http_success_request + http_error_request",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(150).build(),
                    Sample.builder().labels(of("idc", "t2")).value(33).build()
                )),
                false,
            },
            {
                "empty-minus-empty",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.EMPTY),
                "http_success_request - http_error_request",
                Result.fail(),
                false,
            },
            {
                "empty-minus-sampleFamily",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.build(
                        Sample.builder().labels(of("idc", "t1")).value(50).build(),
                        Sample.builder().labels(of("idc", "t2")).value(3).build()
                    )),
                "http_success_request - http_error_request",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(-50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(-3).build()
                )),
                false,
            },
            {
                "sampleFamily-minus-empty",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.build(
                        Sample.builder().labels(of("idc", "t1")).value(50).build(),
                        Sample.builder().labels(of("idc", "t2")).value(3).build()
                    )),
                "http_error_request - http_success_request ",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(3).build()
                )),
                false,
            },
            {
                "sampleFamily-minus-sampleFamily",
                of("http_success_request", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(100).build(),
                    Sample.builder().labels(of("idc", "t2")).value(30).build(),
                    Sample.builder().labels(of("idc", "t3")).value(40).build(),
                    Sample.builder().labels(of("region", "us")).value(80).build()
                ), "http_error_request", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(3).build(),
                    Sample.builder().labels(of("idc", "t5")).value(3).build(),
                    Sample.builder().labels(of("tz", "en-US")).value(3).build()
                )),
                "http_success_request - http_error_request",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(27).build()
                )),
                false,
            },
            {
                "empty-multiple-empty",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.EMPTY),
                "http_success_request * http_error_request",
                Result.fail(),
                false,
            },
            {
                "empty-multiple-sampleFamily",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.build(
                        Sample.builder().labels(of("idc", "t1")).value(50).build(),
                        Sample.builder().labels(of("idc", "t2")).value(3).build()
                    )),
                "http_success_request * http_error_request",
                Result.fail(),
                false,
            },
            {
                "sampleFamily-multiple-empty",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.build(
                        Sample.builder().labels(of("idc", "t1")).value(50).build(),
                        Sample.builder().labels(of("idc", "t2")).value(3).build()
                    )),
                "http_error_request * http_success_request ",
                Result.fail(),
                false,
            },
            {
                "sampleFamily-multiple-sampleFamily",
                of("http_success_request", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(100).build(),
                    Sample.builder().labels(of("idc", "t2")).value(30).build(),
                    Sample.builder().labels(of("idc", "t3")).value(40).build(),
                    Sample.builder().labels(of("region", "us")).value(80).build()
                ), "http_error_request", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(3).build(),
                    Sample.builder().labels(of("idc", "t5")).value(3).build(),
                    Sample.builder().labels(of("tz", "en-US")).value(3).build()
                )),
                "http_success_request * http_error_request",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(5000).build(),
                    Sample.builder().labels(of("idc", "t2")).value(90).build()
                )),
                false,
            },
            {
                "empty-divide-empty",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.EMPTY),
                "http_success_request / http_error_request",
                Result.fail(),
                false,
            },
            {
                "empty-divide-sampleFamily",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.build(
                        Sample.builder().labels(of("idc", "t1")).value(50).build(),
                        Sample.builder().labels(of("idc", "t2")).value(3).build()
                    )),
                "http_success_request / http_error_request",
                Result.fail(),
                false,
            },
            {
                "sampleFamily-divide-empty",
                of("http_success_request", SampleFamily.EMPTY,
                    "http_error_request", SampleFamily.build(
                        Sample.builder().labels(of("idc", "t1")).value(50).build(),
                        Sample.builder().labels(of("idc", "t2")).value(3).build()
                    )),
                "http_error_request / http_success_request ",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(Double.POSITIVE_INFINITY).build(),
                    Sample.builder().labels(of("idc", "t2")).value(Double.POSITIVE_INFINITY).build()
                )),
                false,
            },
            {
                "sampleFamily-divide-sampleFamily",
                of("http_success_request", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(100).build(),
                    Sample.builder().labels(of("idc", "t2")).value(30).build(),
                    Sample.builder().labels(of("idc", "t3")).value(40).build(),
                    Sample.builder().labels(of("region", "us")).value(80).build()
                ), "http_error_request", SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(50).build(),
                    Sample.builder().labels(of("idc", "t2")).value(3).build(),
                    Sample.builder().labels(of("idc", "t5")).value(3).build(),
                    Sample.builder().labels(of("tz", "en-US")).value(3).build()
                )),
                "http_success_request / http_error_request",
                Result.success(SampleFamily.build(
                    Sample.builder().labels(of("idc", "t1")).value(2).build(),
                    Sample.builder().labels(of("idc", "t2")).value(10).build()
                )),
                false,
            },
        });
    }

    @Test
    public void test() {
        Expression e = DSL.parse(expression);
        Result r = null;
        try {
            r = e.run(input);
        } catch (Throwable t) {
            if (isThrow) {
                return;
            }
            log.error("Test failed", t);
            fail("Should not throw anything");
        }
        if (isThrow) {
            fail("Should throw something");
        }
        assertThat(r, is(want));
    }
}