package org.appenders.log4j2.elasticsearch.hc;

/*-
 * #%L
 * log4j2-elasticsearch
 * %%
 * Copyright (C) 2019 Rafal Foltynski
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.fasterxml.jackson.databind.ObjectReader;

import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CountDownLatch;
import java.util.function.Function;

/**
 * Allows to execute blocking calls in asynchronous framework.
 * {@link #getResult()} will wait until the result is ready. Uses {@link java.util.concurrent.CountDownLatch} internally
 *
 * @param <T> response type
 */
public class BlockingResponseHandler<T extends Response> implements ResponseHandler<T> {

    protected CountDownLatch countDownLatch = new CountDownLatch(1);
    protected final ObjectReader objectReader;
    protected final Function<Exception, T> errorResponseTemplate;
    protected T result;

    /**
     * @param objectReader used to deserialize response
     * @param errorResponseTemplate used to generate response in case of any failures
     */
    public BlockingResponseHandler(ObjectReader objectReader, Function<Exception, T> errorResponseTemplate) {
        this.objectReader = objectReader;
        this.errorResponseTemplate = errorResponseTemplate;
    }

    /**
     * Sets the result. Releases threads waiting at {@link #getResult()}
     *
     * @param result result
     */
    @Override
    public final void completed(T result) {
        this.result = result;
        countDownLatch.countDown();
    }

    /**
     * Handles exceptions. Sets response generated by {@link #errorResponseTemplate}.
     * Releases threads waiting at {@link #getResult()}
     *
     * @param exception exception to handle
     */
    @Override
    public final void failed(Exception exception) {
        this.result = errorResponseTemplate.apply(exception);
        countDownLatch.countDown();
    }

    @Override
    public T deserializeResponse(InputStream inputStream) throws IOException {
        return objectReader.readValue(inputStream);
    }

    /**
     * @return result when available, waits otherwise
     */
    public T getResult() {
        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return result;
    }

}
