/*
 * Decompiled with CFR 0.152.
 */
package org.biojava.nbio.structure;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.biojava.nbio.structure.align.util.HTTPConnectionTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

public class PDBStatus {
    private static final Logger logger = LoggerFactory.getLogger(PDBStatus.class);
    public static final String DEFAULT_PDB_SERVER = "www.rcsb.org";
    public static final String PDB_SERVER_PROPERTY = "PDB.SERVER";
    private static Map<String, Map<String, String>> recordsCache = new Hashtable<String, Map<String, String>>();

    public static Status getStatus(String pdbId) {
        Status[] statuses = PDBStatus.getStatus(new String[]{pdbId});
        if (statuses != null) {
            assert (statuses.length == 1);
            return statuses[0];
        }
        return null;
    }

    public static Status[] getStatus(String[] pdbIds) {
        Status[] statuses = new Status[pdbIds.length];
        List<Map<String, String>> attrList = PDBStatus.getStatusIdRecords(pdbIds);
        if (attrList == null || attrList.size() != pdbIds.length) {
            logger.error("Error getting Status for {} from the PDB website.", (Object)Arrays.toString(pdbIds));
            return null;
        }
        for (int pdbNum = 0; pdbNum < pdbIds.length; ++pdbNum) {
            boolean foundAttr = false;
            for (Map<String, String> attrs : attrList) {
                String id = attrs.get("structureId");
                if (id == null || !id.equalsIgnoreCase(pdbIds[pdbNum])) continue;
                String statusStr = attrs.get("status");
                Status status = null;
                if (statusStr == null) {
                    logger.error("No status returned for {}", (Object)pdbIds[pdbNum]);
                    statuses[pdbNum] = null;
                } else {
                    status = Status.fromString(statusStr);
                }
                if (status == null) {
                    logger.error("Unknown status '{}'", (Object)statusStr);
                    statuses[pdbNum] = null;
                }
                statuses[pdbNum] = status;
                foundAttr = true;
            }
            if (foundAttr) continue;
            logger.error("No result found for {}", (Object)pdbIds[pdbNum]);
            statuses[pdbNum] = null;
        }
        return statuses;
    }

    public static String getCurrent(String oldPdbId) {
        List<String> replacements = PDBStatus.getReplacement(oldPdbId, true, false);
        if (replacements != null && !replacements.isEmpty()) {
            return replacements.get(0);
        }
        return null;
    }

    public static List<String> getReplacement(String oldPdbId, boolean recurse, boolean includeObsolete) {
        List<Map<String, String>> attrList = PDBStatus.getStatusIdRecords(new String[]{oldPdbId});
        if (attrList == null || attrList.size() != 1) {
            logger.error("Error getting Status for {} from the PDB website.", (Object)oldPdbId);
            return null;
        }
        Map<String, String> attrs = attrList.get(0);
        String id = attrs.get("structureId");
        if (id == null || !id.equalsIgnoreCase(oldPdbId)) {
            logger.error("Results returned from the query don't match {}", (Object)oldPdbId);
            return null;
        }
        String statusStr = attrs.get("status");
        if (statusStr == null) {
            logger.error("No status returned for {}", (Object)oldPdbId);
            return null;
        }
        Status status = Status.fromString(statusStr);
        if (status == null) {
            logger.error("Unknown status '{}'", (Object)statusStr);
            return null;
        }
        LinkedList<String> results = new LinkedList<String>();
        switch (status) {
            case CURRENT: {
                results.add(oldPdbId);
                return results;
            }
            case OBSOLETE: {
                String replacementStr = attrs.get("replacedBy");
                if (replacementStr == null) {
                    logger.error("{} is OBSOLETE but lacks a replacedBy attribute.", (Object)oldPdbId);
                    return null;
                }
                replacementStr = replacementStr.toUpperCase();
                if (includeObsolete) {
                    results.add(oldPdbId);
                }
                if (replacementStr.equals("NONE")) {
                    return results;
                }
                String[] replacements = replacementStr.split(" ");
                Arrays.sort(replacements, new Comparator<String>(){

                    @Override
                    public int compare(String o1, String o2) {
                        return o2.compareToIgnoreCase(o1);
                    }
                });
                block8: for (String replacement : replacements) {
                    if (recurse) {
                        List<String> others = PDBStatus.getReplacement(replacement, recurse, includeObsolete);
                        PDBStatus.mergeReversed(results, others);
                        continue;
                    }
                    if (includeObsolete) {
                        PDBStatus.mergeReversed(results, Arrays.asList(replacement));
                        continue;
                    }
                    Status replacementStatus = PDBStatus.getStatus(replacement);
                    switch (replacementStatus) {
                        case OBSOLETE: {
                            continue block8;
                        }
                        default: {
                            PDBStatus.mergeReversed(results, Arrays.asList(replacement));
                        }
                    }
                }
                return results;
            }
            case UNKNOWN: {
                return null;
            }
        }
        String replacementStr = attrs.get("replacedBy");
        if (replacementStr == null) {
            results.add(oldPdbId);
            return results;
        }
        if ((replacementStr = replacementStr.toUpperCase()).equals("NONE")) {
            return null;
        }
        results.add(oldPdbId);
        String[] replacements = replacementStr.split(" ");
        Arrays.sort(replacements, new Comparator<String>(){

            @Override
            public int compare(String o1, String o2) {
                return o2.compareToIgnoreCase(o1);
            }
        });
        for (String replacement : replacements) {
            if (recurse) {
                List<String> others = PDBStatus.getReplacement(replacement, recurse, includeObsolete);
                PDBStatus.mergeReversed(results, others);
                continue;
            }
            PDBStatus.mergeReversed(results, Arrays.asList(replacement));
        }
        return results;
    }

    private static void mergeReversed(List<String> merged, List<String> other) {
        if (other.isEmpty()) {
            return;
        }
        if (merged.isEmpty()) {
            merged.addAll(other);
            return;
        }
        ListIterator<String> m = merged.listIterator();
        ListIterator<String> o = other.listIterator();
        String prevO = o.next();
        while (m.hasNext()) {
            String nextM = m.next();
            m.previous();
            while (prevO.compareTo(nextM) > 0) {
                m.add(prevO);
                if (!o.hasNext()) {
                    return;
                }
                prevO = o.next();
            }
            if (prevO.equals(nextM)) {
                if (!o.hasNext()) {
                    return;
                }
                prevO = o.next();
            }
            m.next();
        }
        m.add(prevO);
        while (o.hasNext()) {
            m.add(o.next());
        }
    }

    public static List<String> getReplaces(String newPdbId, boolean recurse) {
        List<Map<String, String>> attrList = PDBStatus.getStatusIdRecords(new String[]{newPdbId});
        if (attrList == null || attrList.size() != 1) {
            logger.error("Error getting Status for {} from the PDB website.", (Object)newPdbId);
            return null;
        }
        Map<String, String> attrs = attrList.get(0);
        String id = attrs.get("structureId");
        if (id == null || !id.equals(newPdbId)) {
            logger.error("Results returned from the query don't match {}", (Object)newPdbId);
            return null;
        }
        String replacedList = attrs.get("replaces");
        if (replacedList == null) {
            return new ArrayList<String>();
        }
        String[] directDescendents = replacedList.split("\\s");
        if (recurse) {
            LinkedList<String> allDescendents = new LinkedList<String>();
            for (String replaced : directDescendents) {
                List<String> roots = PDBStatus.getReplaces(replaced, recurse);
                PDBStatus.mergeReversed(allDescendents, roots);
            }
            PDBStatus.mergeReversed(allDescendents, Arrays.asList(directDescendents));
            return allDescendents;
        }
        return Arrays.asList(directDescendents);
    }

    public static void clearCache() {
        recordsCache.clear();
    }

    private static List<Map<String, String>> getStatusIdRecords(String[] pdbIDs) {
        ArrayList<Map<String, String>> result = new ArrayList<Map<String, String>>(pdbIDs.length);
        String serverName = System.getProperty(PDB_SERVER_PROPERTY);
        if (serverName == null) {
            serverName = DEFAULT_PDB_SERVER;
        } else {
            logger.info(String.format("Got System property %s=%s", PDB_SERVER_PROPERTY, serverName));
        }
        if (pdbIDs.length < 1) {
            throw new IllegalArgumentException("No pdbIDs specified");
        }
        String urlStr = String.format("http://%s/pdb/rest/idStatus?structureId=", serverName);
        for (String pdbId : pdbIDs) {
            if (recordsCache.containsKey(pdbId = pdbId.toUpperCase())) {
                result.add(recordsCache.get(pdbId));
                continue;
            }
            urlStr = urlStr + pdbId + ",";
        }
        if (urlStr.charAt(urlStr.length() - 1) == '=') {
            return result;
        }
        try {
            logger.info("Fetching {}", (Object)urlStr);
            URL url = new URL(urlStr);
            InputStream uStream = url.openStream();
            InputSource source = new InputSource(uStream);
            SAXParserFactory parserFactory = SAXParserFactory.newInstance();
            SAXParser parser = parserFactory.newSAXParser();
            XMLReader reader = parser.getXMLReader();
            PDBStatusXMLHandler handler = new PDBStatusXMLHandler();
            reader.setContentHandler(handler);
            reader.parse(source);
            List<Map<String, String>> records = handler.getRecords();
            for (Map<String, String> record : records) {
                String pdbId = record.get("structureId").toUpperCase();
                if (pdbId == null) continue;
                recordsCache.put(pdbId, record);
            }
            result.addAll(handler.getRecords());
        }
        catch (IOException e) {
            logger.error("Problem getting status for {} from PDB server. Error: {}", (Object)Arrays.toString(pdbIDs), (Object)e.getMessage());
            return null;
        }
        catch (SAXException e) {
            logger.error("Problem getting status for {} from PDB server. Error: {}", (Object)Arrays.toString(pdbIDs), (Object)e.getMessage());
            return null;
        }
        catch (ParserConfigurationException e) {
            logger.error("Problem getting status for {} from PDB server. Error: {}", (Object)Arrays.toString(pdbIDs), (Object)e.getMessage());
            return null;
        }
        return result;
    }

    public static SortedSet<String> getCurrentPDBIds() throws IOException {
        TreeSet<String> allPDBs = new TreeSet<String>();
        String serverName = System.getProperty(PDB_SERVER_PROPERTY);
        if (serverName == null) {
            serverName = DEFAULT_PDB_SERVER;
        } else {
            logger.info(String.format("Got System property %s=%s", PDB_SERVER_PROPERTY, serverName));
        }
        String urlStr = String.format("http://%s/pdb/rest/getCurrent", serverName);
        URL u = new URL(urlStr);
        InputStream stream = HTTPConnectionTools.getInputStream(u, 60000);
        if (stream != null) {
            BufferedReader reader = new BufferedReader(new InputStreamReader(stream));
            String line = null;
            while ((line = reader.readLine()) != null) {
                int index = line.lastIndexOf("structureId=");
                if (index <= 0) continue;
                allPDBs.add(line.substring(index + 13, index + 17));
            }
        }
        return allPDBs;
    }

    private static class PDBStatusXMLHandler
    extends DefaultHandler {
        private List<Map<String, String>> records = new ArrayList<Map<String, String>>();

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
            if (qName.equals("record")) {
                HashMap<String, String> attrMap = new HashMap<String, String>(attributes.getLength() * 2);
                for (int i = 0; i < attributes.getLength(); ++i) {
                    attrMap.put(attributes.getQName(i), attributes.getValue(i));
                }
                this.records.add(attrMap);
            }
        }

        @Override
        public void error(SAXParseException e) throws SAXException {
            logger.error(e.getMessage());
            super.error(e);
        }

        public List<Map<String, String>> getRecords() {
            return this.records;
        }
    }

    public static enum Status {
        OBSOLETE,
        CURRENT,
        AUTH,
        HOLD,
        HPUB,
        POLC,
        PROC,
        REFI,
        REPL,
        WAIT,
        WDRN,
        MODEL,
        UNKNOWN;


        public static Status fromString(String statusStr) {
            Status status;
            String statusStrUpper = statusStr.toUpperCase();
            if (statusStrUpper.equalsIgnoreCase("OBSOLETE")) {
                status = OBSOLETE;
            } else if (statusStrUpper.equalsIgnoreCase("CURRENT")) {
                status = CURRENT;
            } else if (statusStrUpper.equalsIgnoreCase("AUTH")) {
                status = AUTH;
            } else if (statusStrUpper.equalsIgnoreCase("HOLD")) {
                status = HOLD;
            } else if (statusStrUpper.equalsIgnoreCase("HPUB")) {
                status = HPUB;
            } else if (statusStrUpper.equalsIgnoreCase("POLC")) {
                status = POLC;
            } else if (statusStrUpper.equalsIgnoreCase("PROC")) {
                status = PROC;
            } else if (statusStrUpper.equalsIgnoreCase("REFI")) {
                status = REFI;
            } else if (statusStrUpper.equalsIgnoreCase("REPL")) {
                status = REPL;
            } else if (statusStrUpper.equalsIgnoreCase("WAIT")) {
                status = WAIT;
            } else if (statusStrUpper.equalsIgnoreCase("WDRN")) {
                status = WDRN;
            } else if (statusStrUpper.equalsIgnoreCase("MODEL")) {
                status = MODEL;
            } else if (statusStrUpper.equalsIgnoreCase("UNKNOWN")) {
                status = UNKNOWN;
            } else {
                throw new IllegalArgumentException("Unable to parse status '" + statusStrUpper + "'.");
            }
            return status;
        }
    }
}

