package org.bitbucket.mcmichailidis.sqlitedbhandler;

import org.bitbucket.mcmichailidis.sqlitedbhandler.exceptions.ConnectionNotFoundException;
import org.bitbucket.mcmichailidis.sqlitedbhandler.exceptions.WrongNameFormatException;
import org.bitbucket.mcmichailidis.sqlitedbhandler.models.BasicRule;
import org.bitbucket.mcmichailidis.sqlitedbhandler.models.ColumnInfo;
import org.bitbucket.mcmichailidis.sqlitedbhandler.models.Row;
import org.bitbucket.mcmichailidis.sqlitedbhandler.annotations.ApiEntry;
import org.bitbucket.mcmichailidis.sqlitedbhandler.tools.InsertDataModule;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.net.URL;
import java.sql.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import static org.bitbucket.mcmichailidis.sqlitedbhandler.tools.NameValidator.validateName;

/**
 * @author KuroiKage.
 */
public class DBCore {
    private static DBCore myInstance = null;

    @ApiEntry
    public static DBCore getInstance(URL filePath)
            throws WrongNameFormatException, ConnectionNotFoundException {

        return getInstance(filePath.getPath());
    }

    @ApiEntry
    public static DBCore getInstance(String sqliteDatabaseName)
            throws WrongNameFormatException, ConnectionNotFoundException {

        if (myInstance != null) {
            return myInstance;
        }

        if (!validateName(sqliteDatabaseName)) {
            throw new WrongNameFormatException();
        }

        if (!sqliteDatabaseName.endsWith(".db")) {
            sqliteDatabaseName = sqliteDatabaseName + ".db";
        }

        myInstance = new DBCore(sqliteDatabaseName);

        return myInstance;
    }

    /**
     * The instance of DBCore
     *
     * @return Return the initialized instance of DBCore OR null if the core was never initialized
     */
    @ApiEntry
    public static DBCore getInstance() {
        return myInstance;
    }

    private final Connection CONNECTION;

    private DBCore(String sqliteDatabaseName) throws ConnectionNotFoundException {
        Connection c;

        try {
            c = DriverManager.getConnection("jdbc:sqlite:" + sqliteDatabaseName);
        } catch (SQLException ex) {
            Logger.getLogger(DBCore.class.getName()).log(Level.SEVERE, null, ex);
            throw new ConnectionNotFoundException(sqliteDatabaseName);
        }

        CONNECTION = c;
    }

    public void DBTerminate() throws SQLException {
        CONNECTION.close();
    }

    public void DBInit(String initQuery) {
        try {
            Statement stmt = CONNECTION.createStatement();

            stmt.executeUpdate(initQuery);
        } catch (SQLException ex) {
            Logger.getLogger(DBCore.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    public List<String> getTables() {
        List<String> toReturn = new ArrayList<>();
        try {
            String query = "SELECT name FROM sqlite_master WHERE type=\"table\" AND name!=\"sqlite_sequence\"";
            Statement stmt = CONNECTION.createStatement();
            ResultSet queryResults = stmt.executeQuery(query);
            while (queryResults.next()) {
                toReturn.add(queryResults.getString("name"));
            }
        } catch (SQLException ex) {
            Logger.getLogger(DBCore.class.getName()).log(Level.SEVERE, null, ex);
        }

        return toReturn;
    }

    public List<ColumnInfo> getTableColumns(String tableName) {
        List<ColumnInfo> toReturn = new ArrayList<>();
        try {
            String query = "PRAGMA TABLE_INFO (" + tableName + ")";
            Statement stmt = CONNECTION.createStatement();
            ResultSet queryResults = stmt.executeQuery(query);
            while (queryResults.next()) {
                ColumnInfo r = new ColumnInfo(queryResults.getString("name"), queryResults.getString("type"), tableName);
                toReturn.add(r);
            }
        } catch (SQLException ex) {
            Logger.getLogger(DBCore.class.getName()).log(Level.SEVERE, null, ex);
        }

        return toReturn;
    }

    public List<Row> getAllRows(String tableName) {
        List<ColumnInfo> columnInfo = getTableColumns(tableName);


        return queryBuilder(tableName, columnInfo);
    }

    public List<Row> queryBuilder(String tableName, List<ColumnInfo> columnInfo, BasicRule... basicRule) {
        List<Row> toReturn = new ArrayList<>();

        StringBuilder columns = new StringBuilder("*");

        if (columnInfo != null && !columnInfo.isEmpty()) {
            columns = new StringBuilder();
            for (ColumnInfo vL : columnInfo) {
                columns.append(vL.getName()).append(',');
            }
            columns.deleteCharAt(columns.length() - 1);
        }

        StringBuilder rules = new StringBuilder();

        rules = basicRuleMerge(rules, basicRule);

        String query = "SELECT " + columns.toString() + " FROM " + tableName + " " + rules.toString();

        try {
            Statement stmt = CONNECTION.createStatement();
            ResultSet queryResults = stmt.executeQuery(query);

            if (columnInfo == null || columnInfo.isEmpty()) {
                columnInfo = getTableColumns(tableName);
            }

            while (queryResults.next()) {
                Row r = new Row();
                for (ColumnInfo aColumnInfo : columnInfo) {
                    if (tableName.equals(aColumnInfo.getTableName())) {
                        r.insertData(aColumnInfo.getName(), queryResults.getString(aColumnInfo.getName()));
                    }
                }
                toReturn.add(r);
            }
        } catch (SQLException ex) {
            Logger.getLogger(DBCore.class.getName()).log(Level.SEVERE, "The query was : " + query, ex);
        }

        return toReturn;
    }

    public ResultSet rawQuery(String query) throws SQLException {
        Statement stmt = CONNECTION.createStatement();

        return stmt.executeQuery(query);
    }

    private int rawUpdate(String query) throws SQLException {
        Statement stmt = CONNECTION.createStatement();

        return stmt.executeUpdate(query);
    }

    public int insertData(String table, Map<String, String> data) throws SQLException {
        InsertDataModule idm = new InsertDataModule();
        String query = idm.work(table,data);

        return rawUpdate(query);
    }

    public int insertData(String table, Object o) throws SQLException {
        Map<Field, Annotation[]> data = new HashMap<>();

        for (Field field : o.getClass().getDeclaredFields()) {
            Annotation[] annotations = field.getAnnotations();
            field.setAccessible(true);

            data.put(field, annotations);
        }

        InsertDataModule idm = new InsertDataModule();
        String query = idm.work(table, data, o);

        return rawUpdate(query);
    }

    public int deleteData(String tableName, BasicRule... basicRule) throws SQLException {
        StringBuilder query =  new StringBuilder("DELETE FROM ");
        query.append(tableName);

        query = basicRuleMerge(query, basicRule);

        return rawUpdate(query.append(';').toString());
    }

    private StringBuilder basicRuleMerge(StringBuilder query, BasicRule[] basicRule) {
        if (basicRule.length > 0) {
            query.append(" WHERE ");
            for (BasicRule vL : basicRule) {
                query.append(vL.toRule()).append(',');
            }
            query.deleteCharAt(query.length() - 1);
        }
        return query;
    }
}
