/*
 * Copyright 2017 Swatantra Agrawal. All rights reserved.
 */

package org.swat.json.utils;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.CollectionType;
import com.fasterxml.jackson.databind.type.MapType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import org.swat.core.utils.CoreInterceptor;
import org.swat.core.utils.CoreRtException;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by Swatantra Agrawal
 * on 27-AUG-2017
 */
public class JsonUtil {
  public static final JsonUtil JSON_UTIL = new JsonUtil();
  public static final JsonUtil JSON_SILENT = CoreInterceptor.silent(JSON_UTIL);
  public static final JsonUtil JSON_LOGGED = CoreInterceptor.logged(JSON_UTIL);

  private final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

  public JsonUtil() {
    OBJECT_MAPPER.setSerializationInclusion(JsonInclude.Include.NON_NULL);
  }

  /**
   * Parses the json to desired type
   *
   * @param jsonStream The input stream
   * @param clazz      The Object type
   * @param <T>        Generics
   * @return The parsed Object.
   * @throws CoreRtException in case of any Exception
   */
  public <T> T readObject(InputStream jsonStream, Class<T> clazz) throws CoreRtException {
    try {
      return OBJECT_MAPPER.readValue(jsonStream, clazz);
    } catch (IOException e) {
      throw new CoreRtException(e);
    }
  }

  /**
   * Parses the json to List of desired type
   *
   * @param stream The JSON stream
   * @param clazz  The Object type
   * @param <T>    Generics
   * @return The parsed List.
   * @throws CoreRtException in case of any Exception
   */
  public <T> List<T> readList(InputStream stream, Class<T> clazz) throws CoreRtException {
    TypeFactory typeFactory = TypeFactory.defaultInstance();
    CollectionType collectionType = typeFactory.constructCollectionType(ArrayList.class, clazz);
    try {
      return OBJECT_MAPPER.readValue(stream, collectionType);
    } catch (IOException e) {
      throw new CoreRtException(e);
    }
  }

  /**
   * Parses the json to List of desired type
   *
   * @param json  The JSON
   * @param clazz The Object type
   * @param <T>   Generics
   * @return The parsed List.
   * @throws CoreRtException in case of any Exception
   */
  public <T> List<T> readList(String json, Class<T> clazz) throws CoreRtException {
    TypeFactory typeFactory = TypeFactory.defaultInstance();
    CollectionType collectionType = typeFactory.constructCollectionType(ArrayList.class, clazz);
    try {
      return OBJECT_MAPPER.readValue(json, collectionType);
    } catch (IOException e) {
      throw new CoreRtException(e);
    }
  }

  /**
   * Parses the json to Map of desired type
   *
   * @param json       The JSON
   * @param keyClass   The Key Type
   * @param valueClass The Value Type
   * @param <K>        Generics
   * @param <V>        Generics
   * @return The parsed Map.
   * @throws CoreRtException in case of any Exception
   */
  public <K, V> Map<K, V> readMap(String json, Class<K> keyClass, Class<V> valueClass) throws CoreRtException {
    TypeFactory typeFactory = TypeFactory.defaultInstance();
    MapType mapType = typeFactory.constructMapType(HashMap.class, keyClass, valueClass);
    try {
      return OBJECT_MAPPER.readValue(json, mapType);
    } catch (IOException e) {
      throw new CoreRtException(e);
    }
  }

  /**
   * Method convert to json string for the input object
   *
   * @param object {@link Object} - The input object
   * @return {@link String} - The output json string
   * @throws CoreRtException in case of any Exception
   */
  public String toJsonString(Object object) throws CoreRtException {
    try {
      return OBJECT_MAPPER.writeValueAsString(object);
    } catch (JsonProcessingException e) {
      throw new CoreRtException(e);
    }
  }

  /**
   * Reads json as Object
   *
   * @param json  The json
   * @param clazz The class of Object
   * @param <T>   The type of Object
   * @return The object
   * @throws CoreRtException in case of any Exception
   */
  public <T> T readObject(String json, Class<T> clazz) throws CoreRtException {
    try {
      return OBJECT_MAPPER.readValue(json, clazz);
    } catch (IOException e) {
      throw new CoreRtException(e);
    }
  }

  /**
   * Returns prettified json of the object
   *
   * @param object The object
   * @return The prettified json
   * @throws CoreRtException in case of any Exception
   */
  public String pretty(Object object) throws CoreRtException {
    try {
      return OBJECT_MAPPER.writerWithDefaultPrettyPrinter().writeValueAsString(object);
    } catch (JsonProcessingException e) {
      throw new CoreRtException(e);
    }
  }
}
