/*
 * Copyright © 2018 Swatantra Agrawal. All rights reserved.
 */

package org.swat.json.utils;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * The interface Custom field aware.
 */
public interface CustomFieldAware {
  /**
   * Any getters map.
   *
   * @return the map
   */
  @JsonAnyGetter
  @org.codehaus.jackson.annotate.JsonAnyGetter
  default Map<String, Object> anyGetters() {
    return customFields();
  }

  /**
   * Custom fields map.
   *
   * @return the map
   */
  Map<String, Object> customFields();

  /**
   * Add to set boolean.
   *
   * @param name  the name
   * @param value the value
   * @return the boolean
   */
  default boolean addToSet(String name, Object value) {
    if (value == null) {
      return false;
    }
    Object obj = customFields().get(name);
    if (!(obj instanceof Collection)) {
      obj = new LinkedHashSet<>();
      customFields().put(name, obj);
    }
    Collection<Object> collection = (Collection) obj;
    return collection.add(value);
  }

  /**
   * Add to list.
   *
   * @param name  the name
   * @param value the value
   */
  default void addToList(String name, Object value) {
    if (value == null) {
      return;
    }
    Object obj = customFields().get(name);
    if (!(obj instanceof Collection)) {
      obj = new ArrayList<>();
      customFields().put(name, obj);
    }
    Collection<Object> collection = (Collection) obj;
    collection.add(value);
  }

  /**
   * Add and get int.
   *
   * @param key   the key
   * @param delta the delta
   * @return the int
   */
  default int addAndGet(String key, int delta) {
    Object value = customField(key);
    if (!(value instanceof AtomicInteger)) {
      value = new AtomicInteger(0);
      putCustomField(key, value);
    }
    return ((AtomicInteger) value).addAndGet(delta);
  }

  /**
   * Custom field t.
   *
   * @param <T> the type parameter
   * @param key the key
   * @return the t
   */
  default <T> T customField(String key) {
    return (T) customFields().get(key);
  }

  /**
   * Put custom field.
   *
   * @param name  the name
   * @param value the value
   */
  @JsonAnySetter
  @org.codehaus.jackson.annotate.JsonAnySetter
  default void putCustomField(String name, Object value) {
    customFields().put(name, value);
  }
}
