/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces;

import java.io.StringWriter;

import org.apache.xerces.dom.DocumentImpl;
import org.bluestemsoftware.open.eoa.commons.util.QNameUtils;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.Constants;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.DocumentContext;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.component.ComponentContext;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier;
import org.bluestemsoftware.specification.eoa.component.ComponentDeployment.CurrentReference;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier.Scheme;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentName;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentType;
import org.bluestemsoftware.specification.eoa.component.policy.Policy;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyDocument;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyExpression;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchema;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class WSPolicyDocumentImpl extends DocumentImpl implements WSPolicyDocument, WSPolicyDocument.Provider {

    private static final long serialVersionUID = 1L;

    private String targetNamespace;
    private FragmentIdentifier fragmentIdentifier;
    private ComponentName componentName;
    private ComponentContext componentContext;
    private DocumentContext documentContext;
    private Boolean isDeployed = Boolean.FALSE;

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.RootExtension#getExtensionType()
     */
    public final String getExtensionType() {
        return WSPolicyDocument.TYPE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension#getExtensionFactory()
     */
    public final ExtensionFactory getExtensionFactory() {
        return SystemContext.getContext().getSystem().getSchemaFactory(XMLSchema.TYPE);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getName()
     */
    public ComponentName getName() {
        if (componentName == null && targetNamespace != null) {
            String id = ((WSPolicyExpression)getDocumentElement()).getExpressionID();
            componentName = new ComponentName(targetNamespace, id);
        }
        return componentName;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getComponentType()
     */
    public ComponentType getComponentType() {
        return ComponentType.POLICY;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getComponentContext()
     */
    public ComponentContext getComponentContext() {
        return componentContext;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getRuntimeProvidable()
     */
    public Policy getRuntimeProvidable() {
        ComponentName temp = getName();
        if (temp != null) {
            return (Policy)getComponentContext().getDeployment().getComponent(ComponentType.POLICY, componentName);
        } else {
            return null;
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension#getExtensionProvider()
     */
    public org.bluestemsoftware.specification.eoa.ext.Extension.Provider getExtensionProvider() {
        return this;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#deploy(org.bluestemsoftware.specification.eoa.component.ComponentContext)
     */
    public void deploy(ComponentContext componentContext) throws DeploymentException {
        if (isDeployed == null) {
            throw new DeploymentException("Runtime component "
                    + getFragmentIdentifier()
                    + " is circularly referenced by "
                    + CurrentReference.getCurrentReference());
        }
        if (isDeployed) {
            return;
        }
        isDeployed = null;
        this.componentContext = componentContext;
        isDeployed = Boolean.TRUE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#isDeployed()
     */
    public boolean isDeployed() {
        return isDeployed;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.ComponentRT#getFragmentIdentifier()
     */
    public FragmentIdentifier getFragmentIdentifier() {
        if (fragmentIdentifier == null && targetNamespace != null) {
            String id = ((WSPolicyExpression)getDocumentElement()).getExpressionID();
            fragmentIdentifier = new FragmentIdentifier(targetNamespace, Scheme.EOA, "policyDocument", id);
        }
        return fragmentIdentifier;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyDocument#getPolicyExpression()
     */
    public PolicyExpression getPolicyExpression() {
        return (PolicyExpression)getDocumentElement();
    }

    /*
     * (non-Javadoc)
     * 
     * Overrides ancestor such that we can intercept invocations by PolicyExpressionParser and
     * return our assertion impl element
     */
    @Override
    public Element createElementNS(String namespaceURI, String qualifiedName, String localpart) throws DOMException {
        Element result = null;
        if (namespaceURI.equals(WSPolicyDocument.TYPE)) {
            if (localpart.equals("Policy")) {
                result = new WSPolicyExpressionImpl(this);
            } else {
                result = super.createElementNS(namespaceURI, qualifiedName, localpart);
            }
        } else {
            result = new WSPolicyAssertionImpl(this, namespaceURI, qualifiedName);
        }
        
        // declare element namespace if parent element is defined within a
        // different namespace. nested expression and/or assertion may
        // be serialized standalone and we want to avoid serializer 
        // error. see comments on fixUpNamespace
        
        if (documentContext != null && documentContext.getParent() != null) {
            String parentNamespace = documentContext.getParent().getNamespaceURI();
            if (parentNamespace != null) {
                if (!parentNamespace.equals(result.getNamespaceURI())) {
                    fixUpNamespace(result);
                }
            }            
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * @see org.apache.xerces.dom.CoreDocumentImpl#createElementNS(java.lang.String,
     *      java.lang.String)
     */
    @Override
    public Element createElementNS(String namespaceURI, String qualifiedName) throws DOMException {
        String localPart = QNameUtils.getLocalPart(qualifiedName);
        return createElementNS(namespaceURI, qualifiedName, localPart);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.xerces.dom.DocumentImpl#cloneNode(boolean)
     */
    @Override
    public Node cloneNode(boolean deep) {
        WSPolicyDocumentImpl clone = new WSPolicyDocumentImpl();
        super.cloneNode(clone, true);
        return clone;
    }

    /*
     * (non-Javadoc)
     * @see org.apache.xerces.dom.NodeImpl#toString()
     */
    @Override
    public String toString() {
        StringWriter writer = new StringWriter();
        try {
            DOMSerializer.serializeNode(this, writer, "UTF-8", true);
        } catch (Exception ex) {
            return super.toString();
        }
        return writer.toString();
    }
    
    /*
     * (non-Javadoc)
     * set by PolicyDocumentParser
     */
    public void setDocumentContext(DocumentContext documentContext) {
        this.documentContext = documentContext;
    }

    /*
     * Set by factory. TargetNamespace is tns of wsdl document within which policy expression
     * was parsed.
     */
    void setTargetNamespace(String targetNamespace) {
        this.targetNamespace = targetNamespace;
    }

    String getTargetNamespace() {
        return targetNamespace;
    }
    
    /*
     * fix-up the namespace now so that it doesn't get fixed up when document is serialized. as
     * of xerces 2.9.0, the lsserializer modifies the serialized node, i.e. which violates dom 3
     * spec
     */
    private static void fixUpNamespace(Element element) {

        // TODO: file a bug report w/ xerces after testing against 2.9.1

        if (element.getPrefix() == null || element.getPrefix().equals("")) {
            element.setAttributeNS(Constants.XMLNS_NS, "xmlns", element.getNamespaceURI());
        } else {
            element.setAttributeNS(Constants.XMLNS_NS, "xmlns:" + element.getPrefix(), element.getNamespaceURI());
        }

    }

}