/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util;

/*
 * Adapted from org.apache.neethi.PolicyEngine which was released under the following
 * license:
 * <p>
 * Copyright 2001-2005 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

import javax.xml.namespace.QName;

import org.apache.neethi.All;
import org.apache.neethi.Constants;
import org.apache.neethi.ExactlyOne;
import org.apache.neethi.Policy;
import org.apache.neethi.PolicyOperator;
import org.apache.neethi.PolicyReference;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.PolicyReferenceImpl;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.WSPolicyAssertionImpl;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.WSPolicyAssertionWrapper;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;

/**
 * Adapted from neethi PolicyEngine. Changed policy namespace to ws-policy 1.5 namespace,
 * changed document impl to xerces (which has a much more robust implementation of
 * {@link Element#equals(Object)} which allows us to implement the filter operation), removed
 * support for extension attributes on policy element and support for assertion builder factory
 * plug-ins, i.e. we always construct an instance of WSPolicyAssertionWrapper.
 */
public class PolicyReader {

    public static final String POLICY_NAMESPACE = "http://www.w3.org/ns/ws-policy";

    public static Policy readPolicyDocument(EntityResolver entityResolver, Document document) {
        return getPolicyOperator(entityResolver, document.getDocumentElement());
    }

    private static Policy getPolicyOperator(EntityResolver entityResolver, Element el) {
        Policy policy = new Policy();
        QName qname = org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.Constants.ID_QNAME;
        String id = el.getAttributeNS(qname.getNamespaceURI(), qname.getLocalPart());
        if (!id.equals("")) {
            policy.setId(id);
        }
        qname = org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.Constants.NAME_QNAME;
        String name = el.getAttributeNS(qname.getNamespaceURI(), qname.getLocalPart());
        if (!name.equals("")) {
            policy.setName(name);
        }
        return (Policy)processOperationElement(entityResolver, el, policy);
    }

    private static ExactlyOne getExactlyOneOperator(EntityResolver entityResolver, Element element) {
        return (ExactlyOne)processOperationElement(entityResolver, element, new ExactlyOne());
    }

    private static All getAllOperator(EntityResolver entityResolver, Element element) {
        return (All)processOperationElement(entityResolver, element, new All());
    }

    private static PolicyReference getPolicyReference(EntityResolver entityResolver, Element element) {

        if (!(POLICY_NAMESPACE.equals(element.getNamespaceURI()) && Constants.ELEM_POLICY_REF.equals(element
                .getLocalName()))) {
            throw new RuntimeException("Specified element is not a <wsp:PolicyReference .. />  element");
        }

        // modified to use our policy reference implementation. see comments
        // there
        PolicyReference reference = new PolicyReferenceImpl(entityResolver);

        reference.setURI(element.getAttribute("URI"));
        return reference;

    }

    private static PolicyOperator processOperationElement(EntityResolver entityResolver, Element operationElement, PolicyOperator operator) {

        NodeList nodeList = operationElement.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }
            Element childElement = (Element)nodeList.item(i);
            if (POLICY_NAMESPACE.equals(childElement.getNamespaceURI())) {

                if (Constants.ELEM_POLICY.equals(childElement.getLocalName())) {
                    operator.addPolicyComponent(getPolicyOperator(entityResolver, childElement));

                } else if (Constants.ELEM_EXACTLYONE.equals(childElement.getLocalName())) {
                    operator.addPolicyComponent(getExactlyOneOperator(entityResolver, childElement));

                } else if (Constants.ELEM_ALL.equals(childElement.getLocalName())) {
                    operator.addPolicyComponent(getAllOperator(entityResolver, childElement));

                } else if (Constants.ELEM_POLICY_REF.equals(childElement.getLocalName())) {
                    operator.addPolicyComponent(getPolicyReference(entityResolver, childElement));
                }

            } else {
                
                // the assertion must remain attached to document from which
                // we are reading
                
                WSPolicyAssertionImpl assertionImpl = (WSPolicyAssertionImpl)childElement;
                operator.addPolicyComponent(new WSPolicyAssertionWrapper(assertionImpl));
                
            }
        }

        return operator;
    }

}
