/*
 * Copyright 2019: Christophe Saint-Marcel
 * This software is part of the Caseine project.
 * This software was developped with the support of the following organizations:
 * Université Grenoble Alpes
 * Institut Polytechnique de Grenoble
 * 
 * 
 * This file is part of the VPL Design Tool.
 * The VPL Design Tool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * The VPL Design Tool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the VPL Design Tool.  If not, see <https://www.gnu.org/licenses/>.
 */
package caseine;

import java.io.IOException;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;

import caseine.project.BadIDEException;
import caseine.project.CaseineJavaProject;
import caseine.project.CaseineProject;
import caseine.project.CaseineProjectAlreadyExistingException;
import caseine.project.MavenProjectException;
import caseine.project.NotACaseineProjectException;
import caseine.project.VPLIDMissingException;
import caseine.tools.vplwsclient.exception.VplException;

/**
 * The push command for the vpl design tool. It can be used to generate locally the caseine structure
 * and remotely the VPL files( Requested Files, Execution Files, Corrected Files).
 */
public class Caseine {

	private static String url = "https://moodle.caseine.org/webservice/rest/server.php";
	private static String token = "ca240f18444b233238a47c7fcff30df4";

	private static Options options;

	static {
		options = new Options();

		options.addOption("h", "help", false, "Prints usage and exits. ");
		options.addOption("v", "vplId", true, "specify the vplid. If unspecified, only generate gaseine files.");
	}

	/**
	 *
	 * @param args command line argument
	 * @throws Exception if something wrong
	 */
	public static void main(String[] args) throws Exception {

		CommandLineParser parser = new DefaultParser();
		CommandLine cmd = parser.parse(options, args);

		// Check for args
		if (cmd.hasOption("h")) {
			printUsage();
			System.exit(0);
		}

		String projectPath = ".";
		String vplId = "0";

		String[] remainingArgs = cmd.getArgs();
		if (remainingArgs != null && remainingArgs.length >= 1) {
			projectPath = remainingArgs[0];
		}

		if (cmd.hasOption("vplId")) {
			vplId = cmd.getOptionValue("vplId");
		}

		clean(projectPath);
		local(projectPath, vplId);

		if (!vplId.equals("0")) {
			push(projectPath, vplId, token, url);
		}
	}

	private static void printUsage() {
		HelpFormatter formatter = new HelpFormatter();
		formatter.printHelp("java -cp /path/to/vplplugin.jar caseine.Push [OPTIONS] [projectFile]", "With OPTIONS in:",
				options, "projectFile is the folder containing the original maven project. Defaults to \".\"", false);
	}
	
	/**
	 * Generates the project template.
	 * 
	 * @param projectPath
	 * @param vplId
	 * @param lang
	 * @param ide
	 * @param mvn 
	 * @throws CaseineProjectAlreadyExistingException
	 * @throws IOException
	 * @throws BadIDEException 
	 */
    public static void generate(String projectPath, String vplId, String lang, String ide, boolean mvn) throws CaseineProjectAlreadyExistingException, IOException, BadIDEException {
    	
    	CaseineProject proj = null;
    	switch(lang) {
    		default: // Java
    			proj = new CaseineJavaProject(projectPath, vplId, ide, url, token);
    	}
    	proj.generate(mvn);
    }

	/**
	 * Generates the local templates.
	 * 
	 * @param projectPath the project location
	 * @param vplId 
	 * @throws IOException if ioException
	 * @throws ClassNotFoundException if the class is not found
	 * @throws NotACaseineProjectException 
	 * @throws MavenProjectException 
	 */
	public static void local(String projectPath, String vplId) throws IOException, ClassNotFoundException, NotACaseineProjectException, MavenProjectException {

		CaseineProject proj = new CaseineJavaProject(projectPath, vplId, null, null, null);

        if (proj.isCaseine()) {
    		proj.local();        	
        } else {
        	throw new NotACaseineProjectException();
        }
	}
	
	/**
	 * Publishes the templates to the remote caseine server.
	 * 
	 * @param projectPath the local path
	 * @param vplId the id of the VPL
	 * @param url optionnal url 
	 * @param token your token to be authenticated
	 * @throws IOException if something wrong
	 * @throws ClassNotFoundException if something wrong
	 * @throws VplException if something wrong
	 * @throws VPLIDMissingException 
	 * @throws MavenProjectException 
	 */
	public static void push(String projectPath, String vplId, String url, String token)
			throws IOException, ClassNotFoundException, VplException, VPLIDMissingException, MavenProjectException {
		
		CaseineProject caseineProject  = new CaseineJavaProject(projectPath, vplId, null, url, token);

		caseineProject.push();
	}

	/**
	 * Cleans the local templates.
	 * 
	 * @param projectPath the project location
	 * @throws IOException if an error occurs with the deleted files
	 */
	public static void clean(String projectPath) throws IOException {
		
		CaseineProject caseineProject  = new CaseineJavaProject(projectPath, null, null, null, null);
		
		caseineProject.clean();
	}

	public static void nature(String projectPath, String vplId) throws CaseineProjectAlreadyExistingException, IOException {
		CaseineProject caseineProject  = new CaseineJavaProject(projectPath, vplId, null, null, null);
		
		caseineProject.nature();		
	}
}
