/*
 * Copyright 2019: Christophe Saint-Marcel
 * This software is part of the Caseine project.
 * This software was developped with the support of the following organizations:
 * Université Grenoble Alpes
 * Institut Polytechnique de Grenoble
 * 
 * 
 * This file is part of the VPL Design Tool.
 * The VPL Design Tool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * The VPL Design Tool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the VPL Design Tool.  If not, see <https://www.gnu.org/licenses/>.
 */
package caseine;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.json.JsonObject;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;

import caseine.publication.Publisher;
import caseine.tools.vplwsclient.FileUtils;
import caseine.tools.vplwsclient.RestJsonMoodleClient;
import caseine.tools.vplwsclient.RestJsonMoodleClient.VPLService;
import caseine.tools.vplwsclient.VplFile;
import caseine.tools.vplwsclient.exception.VplException;

/**
 * The push command for the vpl design tool. It can be used to generate locally the caseine structure
 * and remotely the VPL files( Requested Files, Execution Files, Corrected Files).
 */
public class Push {

	private static final String RF = "rf";
	private static final String EF = "ef";
	private static final String CF = "cf";
	/**
	 * By convention sources of a CaseInE project must be here.
	 */
	private static final String PATH_SRC = File.separator + "src";
	/**
	 * By convention test sources of a CaseInE project must be here.
	 */
	private static final String PATH_TEST = File.separator + "test";
	/**
	 * By convention resources of a CaseInE project must be here.
	 */
	private static final String PATH_RESOURCES_SRC = File.separator + "resources" + File.separator + "src";
	/**
	 * By convention test resources of a CaseInE project must be here.
	 */
	private static final String PATH_RESOURCES_TEST = File.separator + "resources" + File.separator + "test";
	/**
	 * By convention lib resources of a CaseInE project must be here.
	 */
	private static final String PATH_RESOURCES_LIB = File.separator + "resources" + File.separator + "lib";

	private static final String PLUGIN_LIB = "caseine.vpl.tools.plugin.jar";

	private static String url = "https://moodle.caseine.org/webservice/rest/server.php";
	private static String token = "ca240f18444b233238a47c7fcff30df4";

	private static Options options;

	static {
		options = new Options();

		options.addOption("h", "help", false, "Prints usage and exits. ");
		options.addOption("v", "vplId", true, "specify the vplid. If unspecified, only generate gaseine files.");
	}

	/**
	 *
	 * @param args command line argument
	 * @throws Exception if something wrong
	 */
	public static void main(String[] args) throws Exception {

		CommandLineParser parser = new DefaultParser();
		CommandLine cmd = parser.parse(options, args);

		// Check for args
		if (cmd.hasOption("h")) {
			printUsage();
			System.exit(0);
		}

		String projectPath = ".";
		String vplId = "0";

		String[] remainingArgs = cmd.getArgs();
		if (remainingArgs != null && remainingArgs.length >= 1) {
			projectPath = remainingArgs[0];
		}

		if (cmd.hasOption("vplId")) {
			vplId = cmd.getOptionValue("vplId");
		}

		clean(projectPath);
		publish(projectPath, null);

		if (!vplId.equals("0")) {
			publishMoodle(projectPath, vplId, token, url);
		}
	}

	private static void printUsage() {
		HelpFormatter formatter = new HelpFormatter();
		formatter.printHelp("java -cp /path/to/vplplugin.jar caseine.Push [OPTIONS] [projectFile]", "With OPTIONS in:",
				options, "projectFile is the folder containing the original maven project. Defaults to \".\"", false);
	}

	/**
	 * Generates the local templates.
	 * 
	 * @param projectPath the project location
	 * @param contextClassLoader 
	 * @throws IOException if ioException
	 * @throws ClassNotFoundException if the class is not found
	 */
	public static void publish(String projectPath, ClassLoader contextClassLoader) throws IOException, ClassNotFoundException {

		// Clean to be confident about the result
		clean(projectPath);

		// Add existing libraries to the classpath
		List<File> jarFiles = FileUtils.listFiles(projectPath + PATH_RESOURCES_LIB);

		URL[] urls = new URL[jarFiles.size() + 1];
		for (int i = 0; i < jarFiles.size(); i++) {
			urls[i] = jarFiles.get(i).toURI().toURL();
		}

		urls[jarFiles.size()] = new File(projectPath + getPATH_BIN() + "classes").toURI().toURL();

		URLClassLoader ucl = new URLClassLoader(urls, contextClassLoader);

		Publisher pbsher = new Publisher(new File(projectPath + PATH_SRC).toPath(),
				new File(projectPath + PATH_TEST).toPath(), new File(projectPath + PATH_RESOURCES_SRC).toPath(),
				new File(projectPath + PATH_RESOURCES_TEST).toPath(), new File(projectPath + getPATH_OUTPUT()).toPath(),
				contextClassLoader);

		try {
			pbsher.publishAll();
		} catch (ClassNotFoundException cnfe) {
			// Clean to prevent push of a bad caseine-output at next time
			clean(projectPath);
			throw cnfe;
		} catch (NullPointerException ex) {
			throw new NullPointerException("IN " + projectPath + "<-->"  );
		}

		// Copy the plugin as a lib (temporary solution)
		File libDirectory = new File(projectPath + getPATH_EXECUTION_FILES() + "lib");
		libDirectory.mkdir();
		Files.copy(new FileInputStream(new File(projectPath + getPATH_BIN() + "lib/" + PLUGIN_LIB)),
				Paths.get(libDirectory.getAbsolutePath(), PLUGIN_LIB));

		// Copy the optional libraries (under resources/lib)
		libDirectory = new File(projectPath + getPATH_REQUESTED_FILES() + "lib");
		libDirectory.mkdir();
		File libResourceDirectory = new File(projectPath + PATH_RESOURCES_LIB);
		if (libResourceDirectory.exists()) {
			for (int i = 0; i < jarFiles.size(); i++) {
				Files.copy(new FileInputStream(jarFiles.get(i)),
						Paths.get(libDirectory.getAbsolutePath(), jarFiles.get(i).getName()));
			}
		}

		File casesFile = new File(projectPath + getPATH_EXECUTION_FILES() + "vpl_evaluate.cases");
		casesFile.createNewFile();
		// List<String> dirs = new ArrayList<>();

		// listDirectory(new File(projectPath + PATH_EXECUTION_FILES), "", dirs);
		// Iterator<String> iter = dirs.iterator();
		Iterator<String> iter = pbsher.iterator();

		BufferedWriter writer = new BufferedWriter(new FileWriter(casesFile));
		writer.write("JunitFiles = ");
		while (iter.hasNext()) {
			String classes = iter.next();
			System.out.println(classes);
			writer.write(classes + ", ");
		}
		writer.close();

		for (Path p : pbsher.getPathToRemove()) {
			Path toRemove = new File(projectPath + getPATH_REQUESTED_FILES(), p.toString()).toPath();
			Files.deleteIfExists(toRemove);
		}
		String requestedTestPath = projectPath + getPATH_REQUESTED_FILES() + File.separator + "test";
		for (Path p : pbsher.getPathToTest()) {
			Path toTest = new File(projectPath + getPATH_REQUESTED_FILES(), p.toString()).toPath();
			Path toTest2 = new File(requestedTestPath, p.toString()).toPath();
			if (!toTest2.getParent().toFile().exists()) {
				toTest2.getParent().toFile().mkdirs();
			}
			Files.move(toTest, toTest2);
		}

		String correctedTestPath = projectPath + getPATH_CORRECTED_FILES() + File.separator + "test";
		for (Path p : pbsher.getPathToTest()) {
			Path toTest = new File(projectPath + getPATH_CORRECTED_FILES(), p.toString()).toPath();
			Path toTest2 = new File(correctedTestPath, p.toString()).toPath();
			if (!toTest2.getParent().toFile().exists()) {
				toTest2.getParent().toFile().mkdirs();
			}
			Files.move(toTest, toTest2);
		}

	}
	
	/**
	 * Publishes the templates to the remote caseine server.
	 * 
	 * @param projectPath the local path
	 * @param vplId the id of the VPL
	 * @param token your token to be authenticated
	 * @param url optionnal url 
	 * @throws IOException if something wrong
	 * @throws ClassNotFoundException if something wrong
	 * @throws VplException if something wrong
	 */
	public static void publishMoodle(String projectPath, String vplId, String token, String url)
			throws IOException, ClassNotFoundException, VplException {

		String[] ids = vplId.split(",");
		
		for (String id: ids) {
			RestJsonMoodleClient client = new RestJsonMoodleClient(id, token, url);
	
			saveFiles(client, projectPath + getPATH_REQUESTED_FILES(), VPLService.VPL_SAVE_RF, true, true);
	
			// Execution Files
			saveFiles(client, projectPath + getPATH_EXECUTION_FILES(), VPLService.VPL_SAVE_EF, true, false);
	
			// Corrected Files
			saveFiles(client, projectPath + getPATH_CORRECTED_FILES(), VPLService.VPL_SAVE_CF, true, true);
		}
	}

	/**
	 * 
	 * @param client the Caseine  client
	 * @param projectLocation the local root directory
	 * @param service the action key for the Caseine WS
	 * @param override is True if we want local files to override remote ones.
	 * @param checkMax is True if we must check the number of files to push 
	 * @throws IOException
	 * @throws VplException
	 */
	private static void saveFiles(RestJsonMoodleClient client, String projectLocation, VPLService service, boolean override, boolean checkMax) throws IOException, VplException {
		List<File> files = FileUtils.listFiles(projectLocation);
		if (files == null) {
			System.out.println("Directory " + projectLocation + " not found");		
		} else if (files.isEmpty()) {
			System.out.println("No files found in " + projectLocation);			
		} else {
			Stream<VplFile> vplFiles = files.stream().map(file-> {return fileToCaseinePath(file, projectLocation);});
			JsonObject result = client.callServiceWithFiles(service.toString(), vplFiles.collect(Collectors.toList()));
			System.out.println(result);
		}		
	}

	private static VplFile fileToCaseinePath(File f, String projectLocation) {
		projectLocation = new File(projectLocation).getAbsolutePath();
		if (!projectLocation.endsWith(File.separator)) {
			projectLocation = projectLocation + File.separator;
		}
		String fProjectRelativeName = f.getAbsolutePath().replace(projectLocation, "").replace(File.separator, "/");
		if(fProjectRelativeName.startsWith(PATH_SRC + "/"))
			fProjectRelativeName = fProjectRelativeName.substring(PATH_SRC.length()+1);
		try {
			return new VplFile(f, fProjectRelativeName);
		} catch (IOException e) {
			return null;
		}
	}


	/**
	 * Cleans the local templates.
	 * 
	 * @param projectPath the project location
	 * @throws IOException if an error occurs with the deleted files
	 */
	public static void clean(String projectPath) throws IOException {
		if (!new File(projectPath + PATH_SRC).exists()) {
			throw new FileNotFoundException("Directory \"" + projectPath + PATH_SRC + "\" does not exist");
		}
		if (!new File(projectPath + getPATH_BIN()).exists()) {
			throw new FileNotFoundException(
					"Directory \"" + projectPath + getPATH_BIN() + "\" does not exist, compile the project first");
		}
		if (new File(projectPath + getPATH_OUTPUT()).exists()) {
			Path folder = Paths.get(projectPath + getPATH_OUTPUT());
			CleanFolder.clean(folder);
		}
	}

	/**
	 * By convention requested files generated for a CaseInE project must be here.
	 */
	private static String getPATH_REQUESTED_FILES() {
		return getPATH_OUTPUT() + File.separator + RF + File.separator;
	}

	/**
	 * By convention corrected files generated for a CaseInE project must be here.
	 */
	private static String getPATH_CORRECTED_FILES() {
		return getPATH_OUTPUT() + File.separator + CF + File.separator;
	}

	/**
	 * By convention execution files generated for a CaseInE project must be here.
	 */
	private static String getPATH_EXECUTION_FILES() {
		return getPATH_OUTPUT() + File.separator + EF + File.separator;
	}

	/**
	 * By convention outputs of a CaseInE project must be here.
	 */
	private static String getPATH_OUTPUT() {
		return getPATH_BIN() + "caseine-output";
	}
	
	/*
	 * @param projectPath the project location
	 * @return True if the Caseine project is generated locally
	 */
	public static boolean isGenerated(String projectPath) {
		return new File(projectPath + getPATH_OUTPUT()).exists();
	}

	public static void listDirectory(File file, String prefix, List<String> liste) {
		if (file.exists()) {
			if (file.isFile() && file.getName().endsWith(".java")) {
				liste.add(prefix + file.getName().substring(0, file.getName().length() - 5));
			} else if (file.isDirectory() && !file.getName().startsWith(CF)) {
				File[] contenu = file.listFiles();
				for (int i = 0; i < contenu.length; i++) {
					String pref = prefix;
					if (!file.getName().equals(EF)) {
						pref += file.getName() + ".";
					}
					listDirectory(contenu[i], pref, liste);
				}
			}
		}
	}

	/**
	 * By convention binaries of a CaseInE project must be generated here.
	 */
	private static String getPATH_BIN() {
		return PATH_BIN;
	}

	private static String PATH_BIN = File.separator + "target" + File.separator;
}
