package caseine.project;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.json.JsonObject;

import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

import caseine.Caseine;
import caseine.CleanFolder;
import caseine.publication.Publisher;
import caseine.tools.vplwsclient.FileUtils;
import caseine.tools.vplwsclient.RestJsonMoodleClient;
import caseine.tools.vplwsclient.RestJsonMoodleClient.VPLService;
import caseine.tools.vplwsclient.VplFile;
import caseine.tools.vplwsclient.exception.VplException;

public class CaseineJavaProject implements CaseineProject {

	private static final String VPLID_0 = "0";
	
	private static final String RF = "rf";
	private static final String EF = "ef";
	private static final String CF = "cf";
	/**
	 * By convention sources of a CaseInE project must be here.
	 */
	private static final String PATH_SRC = File.separator + "src";

	/**
	 * By convention binaries of a CaseInE project must be here.
	 */
	private static String PATH_BIN = File.separator + "target" + File.separator;

	/**
	 * By convention test sources of a CaseInE project must be here.
	 */
	private static final String PATH_TEST = File.separator + "test";
	/**
	 * By convention resources of a CaseInE project must be here.
	 */
	private static final String PATH_RESOURCES_SRC = File.separator + "resources" + File.separator + "src";
	/**
	 * By convention test resources of a CaseInE project must be here.
	 */
	private static final String PATH_RESOURCES_TEST = File.separator + "resources" + File.separator + "test";
	/**
	 * By convention lib resources of a CaseInE project must be here.
	 */
	private static final String PATH_RESOURCES_LIB = File.separator + "resources" + File.separator + "lib";

	private static final String PLUGIN_LIB = "caseine.vpl.tools.plugin.jar";

	public static String CASEINE_VPL_ID = "CASEINE_VPL_ID";

	private static Logger log = Logger.getLogger(CaseineJavaProject.class.getName());

	private String projectPath;

	private String vplId;

	private String ide;

	private String url;

	private String token;

	private File caseineFile;

	private static String INTERNAL_FILE = ".caseine";

	/**
	 * 
	 * @param projectPath
	 * @param vplId
	 * @param ide
	 * @param url
	 * @param token
	 */
	public CaseineJavaProject(String projectPath, String vplId, String ide, String url, String token) {
		this.projectPath = projectPath;
		caseineFile = new File(projectPath + File.separator + INTERNAL_FILE);
		this.vplId = vplId;
		this.ide = ide;
		this.url = url;
		this.token = token;
	}

	/**
	 * 
	 * @return True if it is a Caseine project, Maven projects are not managed
	 */
	public boolean isCaseine() {
		boolean result = false;
		
		if (getType() == CaseineJavaProject.ProjectType.MAVEN) {
			result = true;
		}
		if (caseineFile.exists()) {
			result = true;
		}
		return result;
	}

	public ProjectType getType() {
		File pom = new File(projectPath + File.separator + "pom.xml");
		if (pom.exists()) {
			return ProjectType.MAVEN;
		}
		return ProjectType.JAVA;
	}

	public String getVplId() {
		String result = null;
		if (caseineFile.exists()) {
			Properties props = new Properties();
			try {
				props.load(new FileInputStream(caseineFile));
				if (props.containsKey(CASEINE_VPL_ID)) {
					result = (String) props.get(CASEINE_VPL_ID);
				}
			} catch (Exception e) {
				log.severe(e.getMessage());
			}
		} else {
			if (getType() == CaseineJavaProject.ProjectType.MAVEN) {
				// Get it in the pom.xml
				File pom = new File(projectPath + File.separator + "pom.xml");
				MavenXpp3Reader reader = new MavenXpp3Reader();
				try {
					Model model = reader.read(new FileReader(pom));
					Properties properties = model.getProperties();
					result = (String) properties.get(CASEINE_VPL_ID);
				} catch (IOException | XmlPullParserException e) {
					// Nothing found
				}				
			}
		}
		return result;
	}

	public void setVplId(String vplId) {
		if (caseineFile.exists() && !vplId.equals(VPLID_0)) {
			Properties props = new Properties();
			try {
				props.load(new FileInputStream(caseineFile));
				props.setProperty(CASEINE_VPL_ID, vplId);
				props.store(new FileOutputStream(caseineFile), null);
			} catch (Exception e) {
				log.severe(e.getMessage());
			}
		}
	}

	/**
	 * Generates the project template.
	 * 
	 * @param mvn True if we must generate the maven pom.xml
	 * 
	 * @throws CaseineProjectAlreadyExistingException
	 * @throws BadIDEException
	 * @throws IOException
	 */
	public void generate(boolean mvn) throws CaseineProjectAlreadyExistingException, BadIDEException, IOException {

		if (isCaseine()) {
			throw new CaseineProjectAlreadyExistingException("A project already exists in " + projectPath);
		}
		File root = new File(projectPath);
		if (!root.exists()) {
			root.mkdir();
		}

		caseine.FileUtils.copyAZipAndUnZipIt(Caseine.class.getResourceAsStream("/java/default/project.zip"), root, ".");

		if (ide != null) {
			try {
				caseine.FileUtils.copyAZipAndUnZipIt(Caseine.class.getResourceAsStream("/java/" + ide + "/project.zip"),
						root, ".");
			} catch (Exception e) {
				CleanFolder.clean(root.toPath());
				throw new BadIDEException("ide not supported: " + ide);
			}
		}

		// Adds vplId in the project
		setVplId(vplId);
		// Fills the .caseine with the project name
		fillCaseineProjectFile();
		
		if (mvn) {
			String pom = projectPath + File.separatorChar + "pom.xml";
			caseine.FileUtils.writeAResource(Caseine.class.getResourceAsStream("/java/mvn/pom.xml"),
					new File(pom));
			caseine.FileUtils.fillTemplate(pom, "${vpl.name}", getProjectName());
			String vplId = getVplId();
			if (vplId != null && !"<env>".equals(vplId)) {
				caseine.FileUtils.fillTemplate(pom, "${vpl.id}", vplId);				
			}
			// Finally, remove the caseine project file which is replaced by the pom.xml!
			caseineFile.delete();
		} else {
			// Adds the plugin to the lib
			caseine.FileUtils.writeAResource(Caseine.class.getResourceAsStream("/caseine.vpl.tools.plugin.jar"),
					new File(projectPath + File.separatorChar + "resources" + File.separatorChar + "lib"
							+ File.separatorChar + "caseine.vpl.tools.plugin.jar"));			
		}

	}

	/**
	 * Generates the local caseine templates into caseine-output.
	 * 
	 * @throws IOException            if ioException
	 * @throws ClassNotFoundException if the class is not found
	 * @throws MavenProjectException 
	 */
	public void local() throws IOException, ClassNotFoundException, MavenProjectException {
		
		if (getType() == CaseineJavaProject.ProjectType.MAVEN) {
			throw new MavenProjectException();
		}

		if (vplId.equals(VPLID_0)) {
			vplId = getVplId();
		} else { // Update the caseine project file
			setVplId(vplId);
		}

		// Clean to be confident about the result
		clean();

		// Add existing libraries to the classpath
		List<File> jarFiles = FileUtils.listFiles(projectPath + PATH_RESOURCES_LIB);

		URL[] urls = new URL[jarFiles.size() + 1];
		for (int i = 0; i < jarFiles.size(); i++) {
			urls[i] = jarFiles.get(i).toURI().toURL();
		}

		urls[jarFiles.size()] = new File(projectPath + getPATH_BIN() + "classes").toURI().toURL();
		
		URLClassLoader ucl = new URLClassLoader(urls, Caseine.class.getClassLoader());

		Publisher pbsher = new Publisher(new File(projectPath + PATH_SRC).toPath(),
				new File(projectPath + PATH_TEST).toPath(), new File(projectPath + PATH_RESOURCES_SRC).toPath(),
				new File(projectPath + PATH_RESOURCES_TEST).toPath(), new File(projectPath + getPATH_OUTPUT()).toPath(),
				ucl);

		try {
			pbsher.publishAll();
		} catch (ClassNotFoundException cnfe) {
			// Clean to prevent push of a bad caseine-output at next time
			clean();
			throw cnfe;
		} catch (NullPointerException ex) {
			throw new NullPointerException("IN " + projectPath + "<-->");
		}

		// Copy the plugin as a lib (temporary solution ?!?)
		File libEFDirectory = new File(projectPath + getPATH_EXECUTION_FILES() + "lib");
		if (!libEFDirectory.exists()) {
			// Try in lib
		}
		
		libEFDirectory.mkdir();
		if (getType() == CaseineJavaProject.ProjectType.MAVEN) {
			caseine.FileUtils.writeAResource(Caseine.class.getResourceAsStream("/caseine.vpl.tools.plugin.jar"),
					Paths.get(libEFDirectory.getAbsolutePath(), PLUGIN_LIB).toFile());			
		}

		// Copy the optional libraries (under resources/lib)
		File libRFDirectory = new File(projectPath + getPATH_REQUESTED_FILES() + "lib");
		libRFDirectory.mkdir();
		File libResourceDirectory = new File(projectPath + PATH_RESOURCES_LIB);
		if (libResourceDirectory.exists()) {
			for (int i = 0; i < jarFiles.size(); i++) {
				if (jarFiles.get(i).getName().equals(PLUGIN_LIB)) {
					Files.copy(new FileInputStream(jarFiles.get(i)),
							Paths.get(libEFDirectory.getAbsolutePath(), jarFiles.get(i).getName()));
				} else {
					Files.copy(new FileInputStream(jarFiles.get(i)),
							Paths.get(libRFDirectory.getAbsolutePath(), jarFiles.get(i).getName()));
				}
			}
		}

		File casesFile = new File(projectPath + getPATH_EXECUTION_FILES() + "vpl_evaluate.cases");
		casesFile.createNewFile();
		// List<String> dirs = new ArrayList<>();

		// listDirectory(new File(projectPath + PATH_EXECUTION_FILES), "", dirs);
		// Iterator<String> iter = dirs.iterator();
		Iterator<String> iter = pbsher.iterator();

		BufferedWriter writer = new BufferedWriter(new FileWriter(casesFile));
		writer.write("JunitFiles = ");
		while (iter.hasNext()) {
			String classes = iter.next();
			System.out.println(classes);
			writer.write(classes + ", ");
		}
		writer.close();

		for (Path p : pbsher.getPathToRemove()) {
			Path toRemove = new File(projectPath + getPATH_REQUESTED_FILES(), p.toString()).toPath();
			Files.deleteIfExists(toRemove);
		}
		String requestedTestPath = projectPath + getPATH_REQUESTED_FILES() + File.separator + "test";
		for (Path p : pbsher.getPathToTest()) {
			Path toTest = new File(projectPath + getPATH_REQUESTED_FILES(), p.toString()).toPath();
			Path toTest2 = new File(requestedTestPath, p.toString()).toPath();
			if (!toTest2.getParent().toFile().exists()) {
				toTest2.getParent().toFile().mkdirs();
			}
			Files.move(toTest, toTest2);
		}

		String correctedTestPath = projectPath + getPATH_CORRECTED_FILES() + File.separator + "test";
		for (Path p : pbsher.getPathToTest()) {
			Path toTest = new File(projectPath + getPATH_CORRECTED_FILES(), p.toString()).toPath();
			Path toTest2 = new File(correctedTestPath, p.toString()).toPath();
			if (!toTest2.getParent().toFile().exists()) {
				toTest2.getParent().toFile().mkdirs();
			}
			Files.move(toTest, toTest2);
		}

	}

	/**
	 * Publishes the templates to the remote caseine server.
	 * 
	 * @throws IOException            if something wrong
	 * @throws ClassNotFoundException if something wrong
	 * @throws VplException           if something wrong
	 * @throws VPLIDMissingException
	 * @throws MavenProjectException 
	 */
	public void push() throws IOException, ClassNotFoundException, VplException, VPLIDMissingException, MavenProjectException {

		// Run local to be sure
		local();
		
		vplId = getVplId();

		RestJsonMoodleClient client = new RestJsonMoodleClient(vplId, token, url);

		saveFiles(client, projectPath + getPATH_REQUESTED_FILES(), VPLService.VPL_SAVE_RF, true, true);

		// Execution Files
		saveFiles(client, projectPath + getPATH_EXECUTION_FILES(), VPLService.VPL_SAVE_EF, true, false);

		// Corrected Files
		saveFiles(client, projectPath + getPATH_CORRECTED_FILES(), VPLService.VPL_SAVE_CF, true, true);

	}

	/**
	 * Cleans the local templates.
	 * 
	 * @throws IOException if an error occurs with the deleted files
	 */
	public void clean() throws IOException {
		if (!new File(projectPath + PATH_SRC).exists()) {
			throw new FileNotFoundException("Directory \"" + projectPath + PATH_SRC + "\" does not exist");
		}
		new File(projectPath + getPATH_BIN()).mkdir();
		if (new File(projectPath + getPATH_OUTPUT()).exists()) {
			Path folder = Paths.get(projectPath + getPATH_OUTPUT());
			CleanFolder.clean(folder);
		}
	}

	@Override
	public void nature() throws CaseineProjectAlreadyExistingException, IOException {
		if (isCaseine()) {
			if (getType() == CaseineJavaProject.ProjectType.MAVEN) {
				throw new CaseineProjectAlreadyExistingException("Nature command is not implemented for Maven projects");
			}
			throw new CaseineProjectAlreadyExistingException("A project already exists in " + projectPath);
		}
		caseineFile.createNewFile();
		// Adds vplId in the project
		setVplId(vplId);		
	}

	/*
	 * 
	 * @return True if the Caseine project is generated locally
	 */
	public boolean isGenerated() {
		return new File(projectPath + getPATH_OUTPUT()).exists();
	}
		
	public String getProjectName() {
		String projectName = null;
		File projectFile = new File(projectPath);
		if (projectPath.endsWith(".")) {
			projectName = projectFile.getParentFile().getName();
		} else {
			projectName = projectFile.getName();
		}
		return projectName;
	}

	private void fillCaseineProjectFile() {
		try {
			caseine.FileUtils.fillTemplate(projectPath + File.separatorChar + ".project", "/*VPL_NAME*/", getProjectName());
		} catch (IOException e) {
			log.severe(e.getMessage());
		}
	}

	/**
	 * 
	 * @param client          the Caseine client
	 * @param projectLocation the local root directory
	 * @param service         the action key for the Caseine WS
	 * @param override        is True if we want local files to override remote
	 *                        ones.
	 * @param checkMax        is True if we must check the number of files to push
	 * @throws IOException
	 * @throws VplException
	 */
	private static void saveFiles(RestJsonMoodleClient client, String projectLocation, VPLService service,
			boolean override, boolean checkMax) throws IOException, VplException {
		List<File> files = FileUtils.listFiles(projectLocation);
		if (files == null) {
			System.out.println("Directory " + projectLocation + " not found");
		} else if (files.isEmpty()) {
			System.out.println("No files found in " + projectLocation);
		} else {
			Stream<VplFile> vplFiles = files.stream().map(file -> {
				return fileToCaseinePath(file, projectLocation);
			});
			JsonObject result = client.callServiceWithFiles(service.toString(), vplFiles.collect(Collectors.toList()));
			System.out.println(result);
		}
	}

	private static VplFile fileToCaseinePath(File f, String projectLocation) {
		projectLocation = new File(projectLocation).getAbsolutePath();
		if (!projectLocation.endsWith(File.separator)) {
			projectLocation = projectLocation + File.separator;
		}
		String fProjectRelativeName = f.getAbsolutePath().replace(projectLocation, "").replace(File.separator, "/");
		if (fProjectRelativeName.startsWith(PATH_SRC + "/"))
			fProjectRelativeName = fProjectRelativeName.substring(PATH_SRC.length() + 1);
		try {
			return new VplFile(f, fProjectRelativeName);
		} catch (IOException e) {
			return null;
		}
	}

	/**
	 * By convention requested files generated for a CaseInE project must be here.
	 */
	private static String getPATH_REQUESTED_FILES() {
		return getPATH_OUTPUT() + File.separator + RF + File.separator;
	}

	/**
	 * By convention corrected files generated for a CaseInE project must be here.
	 */
	private static String getPATH_CORRECTED_FILES() {
		return getPATH_OUTPUT() + File.separator + CF + File.separator;
	}

	/**
	 * By convention execution files generated for a CaseInE project must be here.
	 */
	private static String getPATH_EXECUTION_FILES() {
		return getPATH_OUTPUT() + File.separator + EF + File.separator;
	}

	/**
	 * By convention outputs of a CaseInE project must be here.
	 */
	private static String getPATH_OUTPUT() {
		return getPATH_BIN() + "caseine-output";
	}

	/**
	 * By convention binaries of a CaseInE project must be generated here.
	 */
	private static String getPATH_BIN() {
		return PATH_BIN;
	}
}
