/**
 * <h1>AWS CDK Resolver</h1>
 * <p>
 * The <code>AwsCdkResolver</code> is able to resolve any <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.CfnOutput.html"><code>CfnOutput</code></a>
 * defined by your AWS CDK application. In this example, we create an S3 <code>Bucket</code> with the AWS CDK, and pass its (deploy time generated)
 * name as an environment variable to a Kubernetes <code>CronJob</code> resource.
 * <p>
 * <blockquote><pre>
 * import * as aws from 'aws-cdk-lib';
 * import * as k8s from 'cdk8s';
 * import * as kplus from 'cdk8s-plus-27';
 * 
 * import { AwsCdkResolver } from '&#64;cdk8s/awscdk-resolver';
 * 
 * const awsApp = new aws.App();
 * const stack = new aws.Stack(awsApp, 'aws');
 * 
 * const k8sApp = new k8s.App({ resolvers: [new AwsCdkResolver()] });
 * const manifest = new k8s.Chart(k8sApp, 'Manifest');
 * 
 * const bucket = new aws.aws_s3.Bucket(stack, 'Bucket');
 * const bucketName = new aws.CfnOutput(stack, 'BucketName', {
 *   value: bucket.bucketName,
 * });
 * 
 * new kplus.CronJob(manifest, 'CronJob', {
 *   schedule: k8s.Cron.daily(),
 *   containers: [{
 *     image: 'job',
 *     envVariables: {
 *       // directly passing the value of the `CfnOutput` containing
 *       // the deploy time bucket name
 *       BUCKET_NAME: kplus.EnvValue.fromValue(bucketName.value),
 *     }
 *  }]
 * });
 * 
 * awsApp.synth();
 * k8sApp.synth();
 * </pre></blockquote>
 * <p>
 * During cdk8s synthesis, the custom resolver will detect that <code>bucketName.value</code> is not a concrete value,
 * but rather a value of a <code>CfnOutput</code>. It will then perform AWS service calls in order to fetch the
 * actual value from the deployed infrastructure in your account. This means that in order
 * for <code>cdk8s synth</code> to succeed, it must be executed <em>after</em> the AWS CDK resources
 * have been deployed. So your deployment workflow should (conceptually) be:
 * <p>
 * <ol>
 * <li><code>cdk deploy</code></li>
 * <li><code>cdk8s synth</code></li>
 * </ol>
 * <p>
 * <blockquote>
 * <p>
 * Note that the <code>AwsCdkResolver</code> is <strong>only</strong> able to fetch tokens that have a <code>CfnOutput</code> defined for them.
 * <p>
 * </blockquote>
 * <p>
 * <h5>Permissions</h5>
 * <p>
 * Since running <code>cdk8s synth</code> will now require performing AWS service calls, it must have access
 * to a set of AWS credentials. Following are the set of actions the credentials must allow:
 * <p>
 * <ul>
 * <li><code>cloudformation:DescribeStacks</code></li>
 * </ul>
 * <p>
 * Note that the actions cdk8s require are far more scoped down than those normally required for the
 * deployment of AWS CDK applications. It is therefore recommended to not reuse the same set of credentials,
 * and instead create a scoped down <code>ReadOnly</code> role dedicated for cdk8s resolvers.
 * <p>
 * <h2>Cross Repository Workflow</h2>
 * <p>
 * As we've seen, your <code>cdk8s</code> application needs access to the objects defined in your cloud application. If both applications
 * are defined within the same file, this is trivial to achieve. If they are in different files, a simple <code>import</code> statement will suffice.
 * However, what if the applications are managed in two separate repositories? This makes it a little trickier, but still possible.
 * <p>
 * In this scenario, <code>cdk.ts</code> in the AWS CDK application, stored in a dedicated repository.
 * <p>
 * <blockquote><pre>
 * import * as aws from 'aws-cdk-lib';
 * 
 * const awsApp = new aws.App();
 * const stack = new aws.Stack(awsApp, 'aws');
 * 
 * const bucket = new aws.aws_s3.Bucket(stack, 'Bucket');
 * const bucketName = new aws.CfnOutput(stack, 'BucketName', {
 *   value: bucket.bucketName,
 * });
 * 
 * awsApp.synth();
 * </pre></blockquote>
 * <p>
 * In order for the <code>cdk8s</code> application to have cross repository access, the AWS CDK object instances that we want to expose need to be available
 * via a package repository. To do this, break up the AWS CDK application into the following files:
 * <p>
 * <code>app.ts</code>
 * <p>
 * <blockquote><pre>
 * import * as aws from 'aws-cdk-lib';
 * 
 * const awsApp = new aws.App();
 * const stack = new aws.Stack(awsApp, 'aws');
 * 
 * const bucket = new aws.aws_s3.Bucket(stack, 'Bucket');
 * // export the thing we want to have available for cdk8s applications
 * export const bucketName = new aws.CfnOutput(stack, 'BucketName', {
 *   value: bucket.bucketName,
 * });
 * 
 * // note that we don't call awsApp.synth here
 * </pre></blockquote>
 * <p>
 * <code>main.ts</code>
 * <p>
 * <blockquote><pre>
 * import { awsApp } from './app.ts'
 * 
 * awsApp.synth();
 * </pre></blockquote>
 * <p>
 * Now, publish the <code>app.ts</code> file to a package manager, so that your <code>cdk8s</code> application can install and import it.
 * This approach might be somewhat counter intuitive, because normally we only publish classes to the package manager,
 * not instances. Indeed, these types of applications introduce a new use-case that requires the sharing of instances.
 * Conceptually, this is no different than writing state<sup>*</sup> to an SSM parameter or an S3 bucket, and it allows us to remain
 * in the boundaries of our programming language, and the typing guarantees it provides.
 * <p>
 * <blockquote>
 * <p>
 * <sup>*</sup> Actually, we are only publishing instructions for fetching state, not the state itself.
 * <p>
 * </blockquote>
 * <p>
 * Assuming <code>app.ts</code> was published as the <code>my-cdk-app</code> package, our <code>cdk8s</code> application will now look like so:
 * <p>
 * <blockquote><pre>
 * import * as k8s from 'cdk8s';
 * import * as kplus from 'cdk8s-plus-27';
 * 
 * // import the desired instance from the AWS CDK app.
 * import { bucketName } from 'my-cdk-app';
 * 
 * import { AwsCdkResolver } from '&#64;cdk8s/awscdk-resolver';
 * 
 * const k8sApp = new k8s.App({ resolvers: [new AwsCdkResolver()] });
 * const manifest = new k8s.Chart(k8sApp, 'Manifest');
 * 
 * new kplus.CronJob(manifest, 'CronJob', {
 *   schedule: k8s.Cron.daily(),
 *   containers: [{
 *     image: 'job',
 *     envVariables: {
 *       // directly passing the value of the `CfnOutput` containing
 *       // the deploy time bucket name
 *       BUCKET_NAME: kplus.EnvValue.fromValue(bucketName.value),
 *     }
 *  }]
 * });
 * 
 * k8sApp.synth();
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package org.cdk8s.awscdkresolver;
