/**
 * <h1>cdk8s-operator</h1>
 * <p>
 * <blockquote>
 * <p>
 * Create Kubernetes CRD Operators using CDK8s Constructs
 * <p>
 * </blockquote>
 * <p>
 * This is a multi-language (jsii) library and a command-line tool that allows you
 * to create Kubernetes operators for CRDs (Custom Resource Definitions) using
 * CDK8s.
 * <p>
 * <h2>Getting Started</h2>
 * <p>
 * Let's create our first CRD served by a CDK8s construct using TypeScript.
 * <p>
 * <h3>Install CDK8s</h3>
 * <p>
 * Make sure your system has the required CDK8s <a href="https://cdk8s.io/docs/latest/getting-started/#prerequisites">prerequisites</a>.
 * <p>
 * Install the CDK8s CLI globally through npm:
 * <p>
 * <blockquote><pre>
 * $ npm i -g cdk8s-cli
 * Installing...
 * 
 * # Verify installation
 * $ cdk8s --version
 * 1.0.0-beta.3
 * </pre></blockquote>
 * <p>
 * <h3>Create a new CDK8s app</h3>
 * <p>
 * Now, let's create a new CDK8s typescript app:
 * <p>
 * <blockquote><pre>
 * mkdir hello-operator &amp;&amp; cd hello-operator
 * git init
 * cdk8s init typescript-app
 * </pre></blockquote>
 * <p>
 * <h3>Install cdk8s-operator</h3>
 * <p>
 * Next, let's install this module as a dependency of our TypeScript project:
 * <p>
 * <blockquote><pre>
 * npm install cdk8s-operator
 * </pre></blockquote>
 * <p>
 * <h3>Construct</h3>
 * <p>
 * We will start by creating the construct that implements the abstraction. This is
 * is just a normal CDK8s custom construct:
 * <p>
 * Let's create a construct called <code>PodCollection</code> which represents a collection of
 * pods:
 * <p>
 * <code>pod-collection.ts</code>:
 * <p>
 * <blockquote><pre>
 * import { Pod } from 'cdk8s-plus-17';
 * import { Construct } from 'constructs';
 * 
 * export interface PodCollectionProps {
 *   /** Number of pods *&#47;
 *   readonly count: number;
 *   /** The docker image to deploy *&#47;
 *   readonly image: string;
 * }
 * 
 * export class PodCollection extends Construct {
 *   constructor(scope: Construct, id: string, props: PodCollectionProps) {
 *     super(scope, id);
 * 
 *     for (let i = 0; i &lt; props.count; ++i) {
 *       new Pod(this, `pod-${i}`, {
 *         containers: [ { image: props.image } ]
 *       });
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Operator App</h3>
 * <p>
 * Now, we will need to replace out <code>main.ts</code> file with an "operator app", which is
 * a special kind of CDK8s app designed to be executed by the <code>cdk8s-server</code> CLI
 * which is included in this module.
 * <p>
 * The <code>Operator</code> app construct can be used to create "CDK8s Operators" which are
 * CDK8s apps that accept input from a file (or STDIN) with a Kubernetes manifest,
 * instantiates a construct with the <code>spec</code> as its input and emits the resulting
 * manifest to STDOUT.
 * <p>
 * Replace the contents of <code>main.ts</code> with the following. We initialize an
 * <code>Operator</code> app and then register a provider which handles resources of API
 * version <code>samples.cdk8s.org/v1alpha1</code> and kind <code>PodCollection</code>.
 * <p>
 * <code>main.ts</code>:
 * <p>
 * <blockquote><pre>
 * import { Operator } from 'cdk8s-operator';
 * import { PodCollection } from './pod-collection';
 * 
 * const app = new Operator();
 * 
 * app.addProvider({
 *   apiVersion: 'samples.cdk8s.org/v1alpha1',
 *   kind: 'PodCollection',
 *   handler: {
 *     apply: (scope, id, props) =&gt; new PodCollection(scope, id, props)
 *   }
 * })
 * 
 * app.synth();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * A single operator can handle any number of resource kinds. Simply call
 * <code>addProvider()</code> for each apiVersion/kind.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Using Operators</h2>
 * <p>
 * To use this operator, create an <code>input.json</code> file, e.g:
 * <p>
 * <code>input.json</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "apiVersion": "samples.cdk8s.org/v1alpha1",
 *   "kind": "PodCollection",
 *   "metadata": {
 *     "name": "my-collection"
 *   },
 *   "spec": {
 *     "image": "paulbouwer/hello-kubernetes",
 *     "count": 5
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * Compile your code:
 * <p>
 * <blockquote><pre>
 * # delete `main.test.ts` since it has some code that won't compile
 * $ rm -f main.test.*
 * 
 * # compile
 * $ npm run compile
 * </pre></blockquote>
 * <p>
 * And run:
 * <p>
 * <blockquote><pre>
 * $ node main.js input.json
 * </pre></blockquote>
 * <p>
 * <details>
 *   <summary>STDOUT</summary>
 * <p>
 * <blockquote><pre>
 * apiVersion: "v1"
 * kind: "Pod"
 * metadata:
 *   name: "my-collection-pod-0-c8735c52"
 * spec:
 *   containers:
 *     - env: []
 *       image: "paulbouwer/hello-kubernetes"
 *       imagePullPolicy: "Always"
 *       name: "main"
 *       ports: []
 *       volumeMounts: []
 *   volumes: []
 * ---
 * apiVersion: "v1"
 * kind: "Pod"
 * metadata:
 *   name: "my-collection-pod-1-c89f58d7"
 * spec:
 *   containers:
 *     - env: []
 *       image: "paulbouwer/hello-kubernetes"
 *       imagePullPolicy: "Always"
 *       name: "main"
 *       ports: []
 *       volumeMounts: []
 *   volumes: []
 * ---
 * apiVersion: "v1"
 * kind: "Pod"
 * metadata:
 *   name: "my-collection-pod-2-c88d4268"
 * spec:
 *   containers:
 *     - env: []
 *       image: "paulbouwer/hello-kubernetes"
 *       imagePullPolicy: "Always"
 *       name: "main"
 *       ports: []
 *       volumeMounts: []
 *   volumes: []
 * ---
 * apiVersion: "v1"
 * kind: "Pod"
 * metadata:
 *   name: "my-collection-pod-3-c86866b1"
 * spec:
 *   containers:
 *     - env: []
 *       image: "paulbouwer/hello-kubernetes"
 *       imagePullPolicy: "Always"
 *       name: "main"
 *       ports: []
 *       volumeMounts: []
 *   volumes: []
 * ---
 * apiVersion: "v1"
 * kind: "Pod"
 * metadata:
 *   name: "my-collection-pod-4-c8b74b1d"
 * spec:
 *   containers:
 *     - env: []
 *       image: "paulbouwer/hello-kubernetes"
 *       imagePullPolicy: "Always"
 *       name: "main"
 *       ports: []
 *       volumeMounts: []
 *   volumes: []
 * </pre></blockquote>
 * <p>
 * </details>
 * <p>
 * <h2><code>cdk8s-server</code></h2>
 * <p>
 * This library is shipped with a program called <code>cdk8s-server</code> which can be used
 * to host your operator inside an HTTP server. This server can be used as a
 * sidecar container with a generic CRD operator (TBD).
 * <p>
 * <blockquote><pre>
 * $ PORT=8080 npx cdk8s-server
 * Listening on 8080
 * - App command: node main.js
 * - Request body should include a single k8s resource in JSON format
 * - Request will be piped through STDIN to "node main.js"
 * - Response is the STDOUT and expected to be a multi-resource yaml manifest
 * </pre></blockquote>
 * <p>
 * Now, you can send <code>input.json</code> over HTTP:
 * <p>
 * <blockquote><pre>
 * $ curl -d &#64;input.json http://localhost:8080
 * MANIFEST...
 * </pre></blockquote>
 * <p>
 * <h2>License</h2>
 * <p>
 * Apache 2.0
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package org.cdk8s.cdk8soperator;
