/*
 * Decompiled with CFR 0.152.
 */
package org.cloudbus.cloudsim.resources;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.cloudbus.cloudsim.distributions.ContinuousDistribution;
import org.cloudbus.cloudsim.resources.File;
import org.cloudbus.cloudsim.resources.FileStorage;
import org.cloudbus.cloudsim.resources.Storage;
import org.cloudbus.cloudsim.util.Log;

public class HarddriveStorage
implements FileStorage {
    private Storage storage;
    private Storage reservedStorage;
    private List<String> fileNameList;
    private List<File> fileList;
    private final String name;
    private ContinuousDistribution gen;
    private double maxTransferRate;
    private double latency;
    private double avgSeekTime;

    public HarddriveStorage(String name, long capacity) throws IllegalArgumentException {
        this.storage = new Storage(capacity);
        this.reservedStorage = new Storage(capacity);
        if (Objects.isNull(name) || name.trim().isEmpty()) {
            throw new IllegalArgumentException("HarddriveStorage(): Error - invalid storage name.");
        }
        this.name = name;
        this.init();
    }

    public HarddriveStorage(long capacity) throws IllegalArgumentException {
        this("HarddriveStorage", capacity);
    }

    private void init() {
        this.fileList = new ArrayList<File>();
        this.fileNameList = new ArrayList<String>();
        this.gen = null;
        this.latency = 0.00417;
        this.avgSeekTime = 0.009;
        this.maxTransferRate = 133.0;
    }

    @Override
    public int getNumStoredFile() {
        return this.fileList.size();
    }

    @Override
    public boolean reserveSpace(int fileSize) {
        if (this.storage.allocateResource(fileSize)) {
            this.reservedStorage.allocateResource(fileSize);
            return true;
        }
        return false;
    }

    @Override
    public double addReservedFile(File file) {
        if (Objects.isNull(file)) {
            return 0.0;
        }
        if (!this.reservedStorage.isResourceAmountBeingUsed(file.getSize())) {
            throw new RuntimeException("The file size wasn't previously reserved in order to add a reserved file.");
        }
        long fileSize = file.getSize();
        this.storage.deallocateResource(fileSize);
        this.reservedStorage.deallocateResource(fileSize);
        double result = this.addFile(file);
        if (result == 0.0) {
            this.storage.allocateResource(fileSize);
        }
        return result;
    }

    @Override
    public boolean hasPotentialAvailableSpace(int fileSize) {
        if (fileSize <= 0) {
            return false;
        }
        if (this.storage.isResourceAmountAvailable(fileSize)) {
            return true;
        }
        return this.getDeletedFilesTotalSize() > fileSize;
    }

    private int getDeletedFilesTotalSize() {
        return this.fileList.stream().filter(File::isDeleted).mapToInt(File::getSize).sum();
    }

    @Override
    public String getName() {
        return this.name;
    }

    public boolean setLatency(double latency) {
        if (latency < 0.0) {
            return false;
        }
        this.latency = latency;
        return true;
    }

    public double getLatency() {
        return this.latency;
    }

    @Override
    public boolean setMaxTransferRate(int rate) {
        if (rate <= 0) {
            return false;
        }
        this.maxTransferRate = rate;
        return true;
    }

    @Override
    public double getMaxTransferRate() {
        return this.maxTransferRate;
    }

    public boolean setAvgSeekTime(double seekTime) {
        return this.setAvgSeekTime(seekTime, null);
    }

    public boolean setAvgSeekTime(double seekTime, ContinuousDistribution gen) {
        if (seekTime <= 0.0) {
            return false;
        }
        this.avgSeekTime = seekTime;
        this.gen = gen;
        return true;
    }

    public double getAvgSeekTime() {
        return this.avgSeekTime;
    }

    @Override
    public File getFile(String fileName) {
        if (!File.isValid(fileName)) {
            Log.printConcatLine(this.name, ".getFile(): Warning - invalid file name.");
            return null;
        }
        int size = 0;
        for (File currentFile : this.fileList) {
            size += currentFile.getSize();
            if (!currentFile.getName().equals(fileName)) continue;
            double seekTime = this.getSeekTime(size);
            double transferTime = this.getTransferTime(currentFile.getSize());
            currentFile.setTransactionTime(seekTime + transferTime);
            return currentFile;
        }
        return null;
    }

    @Override
    public List<String> getFileNameList() {
        return Collections.unmodifiableList(this.fileNameList);
    }

    @Override
    public List<File> getFileList() {
        return Collections.unmodifiableList(this.fileList);
    }

    private double getSeekTime(int fileSize) {
        double result = 0.0;
        if (!Objects.isNull(this.gen)) {
            result += this.gen.sample();
        }
        if (fileSize > 0 && this.storage.getCapacity() != 0L) {
            result += (double)fileSize / (double)this.storage.getCapacity();
        }
        return result;
    }

    private double getTransferTime(int fileSize) {
        double result = 0.0;
        if (fileSize > 0 && this.storage.getCapacity() != 0L) {
            result = (double)fileSize * this.maxTransferRate / (double)this.storage.getCapacity();
        }
        return result;
    }

    @Override
    public double addFile(File file) {
        double result = 0.0;
        if (!File.isValid(file)) {
            Log.printConcatLine(this.name, ".addFile(): Invalid file ", file);
            return result;
        }
        if (!this.storage.isResourceAmountAvailable(file.getSize())) {
            Log.printConcatLine(this.name, ".addFile(): Warning - not enough space to store ", file.getName());
            return result;
        }
        if (!this.contains(file.getName())) {
            double seekTime = this.getSeekTime(file.getSize());
            double transferTime = this.getTransferTime(file.getSize());
            this.fileList.add(file);
            this.fileNameList.add(file.getName());
            this.storage.allocateResource(file.getSize());
            result = seekTime + transferTime;
            file.setTransactionTime(result);
        }
        return result;
    }

    @Override
    public double addFile(List<File> list) {
        if (Objects.isNull(list) || list.isEmpty()) {
            Log.printConcatLine(this.getName(), ".addFile(): Warning - list is empty.");
            return 0.0;
        }
        return list.stream().mapToDouble(this::addFile).sum();
    }

    @Override
    public File deleteFile(String fileName) {
        if (!File.isValid(fileName)) {
            return null;
        }
        int i = this.fileNameList.indexOf(fileName);
        if (i != -1) {
            File file = this.fileList.get(i);
            double result = this.deleteFile(file);
            file.setTransactionTime(result);
            return file;
        }
        return null;
    }

    @Override
    public double deleteFile(File file) {
        if (!File.isValid(file)) {
            return 0.0;
        }
        double seekTime = this.getSeekTime(file.getSize());
        double transferTime = this.getTransferTime(file.getSize());
        if (this.contains(file)) {
            this.fileList.remove(file);
            this.fileNameList.remove(file.getName());
            this.storage.deallocateResource(file.getSize());
            double result = seekTime + transferTime;
            file.setTransactionTime(result);
            return result;
        }
        return 0.0;
    }

    @Override
    public boolean contains(String fileName) {
        if (Objects.isNull(fileName) || fileName.isEmpty()) {
            Log.printConcatLine(this.name, ".contains(): Warning - invalid file name");
            return false;
        }
        return this.fileNameList.contains(fileName);
    }

    @Override
    public boolean contains(File file) {
        if (!File.isValid(file)) {
            return false;
        }
        return this.contains(file.getName());
    }

    @Override
    public boolean renameFile(File file, String newName) {
        if (this.contains(newName)) {
            return false;
        }
        String oldName = file.getName();
        File renamedFile = this.getFile(oldName);
        if (!Objects.isNull(renamedFile)) {
            renamedFile.setName(newName);
            renamedFile.setTransactionTime(0.0);
            this.fileNameList.remove(oldName);
            this.fileNameList.add(newName);
            return true;
        }
        return false;
    }

    @Override
    public long getCapacity() {
        return this.storage.getCapacity();
    }

    @Override
    public long getAvailableResource() {
        return this.storage.getAvailableResource();
    }

    @Override
    public long getAllocatedResource() {
        return this.storage.getAllocatedResource();
    }

    @Override
    public boolean isResourceAmountAvailable(long amountToCheck) {
        return this.storage.isResourceAmountAvailable(amountToCheck);
    }

    @Override
    public boolean isResourceAmountAvailable(double amountToCheck) {
        return this.isResourceAmountAvailable((long)amountToCheck);
    }

    @Override
    public boolean isFull() {
        return this.storage.isFull();
    }
}

