/*
 * Decompiled with CFR 0.152.
 */
package org.cloudbus.cloudsim.datacenters;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.cloudbus.cloudsim.allocationpolicies.VmAllocationPolicy;
import org.cloudbus.cloudsim.allocationpolicies.VmAllocationPolicyAbstract;
import org.cloudbus.cloudsim.allocationpolicies.VmAllocationPolicySimple;
import org.cloudbus.cloudsim.cloudlets.Cloudlet;
import org.cloudbus.cloudsim.core.CloudSimEntity;
import org.cloudbus.cloudsim.core.Simulation;
import org.cloudbus.cloudsim.core.events.PredicateType;
import org.cloudbus.cloudsim.core.events.SimEvent;
import org.cloudbus.cloudsim.datacenters.Datacenter;
import org.cloudbus.cloudsim.datacenters.DatacenterCharacteristics;
import org.cloudbus.cloudsim.datacenters.DatacenterCharacteristicsSimple;
import org.cloudbus.cloudsim.datacenters.DatacenterPowerSupply;
import org.cloudbus.cloudsim.hosts.Host;
import org.cloudbus.cloudsim.network.IcmpPacket;
import org.cloudbus.cloudsim.resources.DatacenterStorage;
import org.cloudbus.cloudsim.resources.FileStorage;
import org.cloudbus.cloudsim.schedulers.cloudlet.CloudletScheduler;
import org.cloudbus.cloudsim.util.Conversion;
import org.cloudbus.cloudsim.util.MathUtil;
import org.cloudbus.cloudsim.vms.Vm;
import org.cloudsimplus.autoscaling.VerticalVmScaling;
import org.cloudsimplus.faultinjection.HostFaultInjection;
import org.cloudsimplus.listeners.EventListener;
import org.cloudsimplus.listeners.HostEventInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatacenterSimple
extends CloudSimEntity
implements Datacenter {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)DatacenterSimple.class.getSimpleName());
    private double bandwidthPercentForMigration;
    private boolean migrationsEnabled;
    private List<? extends Host> hostList;
    private final DatacenterCharacteristics characteristics;
    private VmAllocationPolicy vmAllocationPolicy;
    private double lastProcessTime;
    private double schedulingInterval;
    private DatacenterStorage datacenterStorage;
    private List<EventListener<HostEventInfo>> onHostAvailableListeners;
    private DatacenterPowerSupply powerSupply;

    public DatacenterSimple(Simulation simulation, List<? extends Host> hostList) {
        this(simulation, hostList, (VmAllocationPolicy)new VmAllocationPolicySimple(), new DatacenterStorage());
    }

    public DatacenterSimple(Simulation simulation, List<? extends Host> hostList, VmAllocationPolicy vmAllocationPolicy) {
        this(simulation, hostList, vmAllocationPolicy, new DatacenterStorage());
    }

    public DatacenterSimple(Simulation simulation, VmAllocationPolicy vmAllocationPolicy) {
        this(simulation, new ArrayList(), vmAllocationPolicy, new DatacenterStorage());
    }

    public DatacenterSimple(Simulation simulation, List<? extends Host> hostList, VmAllocationPolicy vmAllocationPolicy, List<FileStorage> storageList) {
        this(simulation, hostList, vmAllocationPolicy, new DatacenterStorage(storageList));
    }

    public DatacenterSimple(Simulation simulation, List<? extends Host> hostList, VmAllocationPolicy vmAllocationPolicy, DatacenterStorage storage) {
        super(simulation);
        this.setHostList(hostList);
        this.powerSupply = DatacenterPowerSupply.NULL;
        this.setLastProcessTime(0.0);
        this.setSchedulingInterval(0.0);
        this.setDatacenterStorage(storage);
        this.onHostAvailableListeners = new ArrayList<EventListener<HostEventInfo>>();
        this.characteristics = new DatacenterCharacteristicsSimple(this);
        this.bandwidthPercentForMigration = 0.5;
        this.migrationsEnabled = true;
        this.setVmAllocationPolicy(vmAllocationPolicy);
    }

    private void setHostList(List<? extends Host> hostList) {
        this.hostList = Objects.requireNonNull(hostList);
        this.setupHosts();
    }

    private void setupHosts() {
        for (Host host : this.hostList) {
            host.setDatacenter(this);
            host.setSimulation(this.getSimulation());
        }
        Simulation.setIdForEntitiesWithoutOne(this.hostList);
    }

    @Override
    public void processEvent(SimEvent evt) {
        if (this.processCloudletEvents(evt) || this.processVmEvents(evt) || this.processNetworkEvents(evt) || this.processHostEvents(evt)) {
            return;
        }
        LOGGER.trace("{}: {}: Unknown event {} received.", new Object[]{this.getSimulation().clock(), this, evt.getTag()});
    }

    private boolean processHostEvents(SimEvent evt) {
        if (evt.getTag() == 60) {
            this.processHostAdditionRequest(evt);
            return true;
        }
        if (evt.getTag() == 61) {
            this.processHostRemovalRequest(evt);
            return true;
        }
        return false;
    }

    private void processHostAdditionRequest(SimEvent evt) {
        this.getHostFromHostEvent(evt).ifPresent(host -> {
            this.addHost(host);
            LOGGER.info("{}: {}: Host {} added to {} during simulation runtime", new Object[]{this.getSimulation().clock(), this.getClass().getSimpleName(), host.getId(), this});
            this.notifyOnHostAvailableListeners(host);
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processHostRemovalRequest(SimEvent srcEvt) {
        long hostId = (Long)srcEvt.getData();
        Host host = this.getHostById(hostId);
        if (host == Host.NULL) {
            LOGGER.warn("{}: {}: Host {} was not found to be removed from {}.", new Object[]{this.getSimulation().clock(), this.getClass().getSimpleName(), hostId, this});
            return;
        }
        HostFaultInjection fault = new HostFaultInjection(this);
        try {
            LOGGER.error("{}: {}: Host {} removed from {} due to injected failure.", new Object[]{this.getSimulation().clock(), this.getClass().getSimpleName(), host.getId(), this});
            fault.generateHostFault(host);
        }
        finally {
            fault.shutdownEntity();
        }
        this.getSimulation().cancelAll(this.getSimulation().getCloudInfoService(), evt -> MathUtil.same(evt.getTime(), srcEvt.getTime()) && evt.getTag() == 61 && ((Long)evt.getData()).longValue() == host.getId());
    }

    private Optional<Host> getHostFromHostEvent(SimEvent evt) {
        if (evt.getData() instanceof Host) {
            return Optional.of((Host)evt.getData());
        }
        return Optional.empty();
    }

    private boolean processNetworkEvents(SimEvent evt) {
        if (evt.getTag() == 105) {
            this.processPingRequest(evt);
            return true;
        }
        return false;
    }

    private boolean processVmEvents(SimEvent evt) {
        switch (evt.getTag()) {
            case 31: {
                this.processVmCreate(evt, false);
                return true;
            }
            case 32: {
                this.processVmCreate(evt, true);
                return true;
            }
            case 42: {
                this.requestVmVerticalScaling(evt);
                return true;
            }
            case 33: {
                this.processVmDestroy(evt, false);
                return true;
            }
            case 34: {
                this.processVmDestroy(evt, true);
                return true;
            }
            case 35: {
                this.finishVmMigration(evt, false);
                return true;
            }
            case 36: {
                this.finishVmMigration(evt, true);
                return true;
            }
            case 41: {
                this.updateCloudletProcessing();
                return true;
            }
        }
        return false;
    }

    private boolean requestVmVerticalScaling(SimEvent evt) {
        if (!(evt.getData() instanceof VerticalVmScaling)) {
            return false;
        }
        return this.vmAllocationPolicy.scaleVmVertically((VerticalVmScaling)evt.getData());
    }

    private boolean processCloudletEvents(SimEvent evt) {
        switch (evt.getTag()) {
            case 16: {
                this.processCloudletSubmit(evt, false);
                return true;
            }
            case 17: {
                this.processCloudletSubmit(evt, true);
                return true;
            }
            case 18: {
                this.processCloudlet(evt, 18);
                return true;
            }
            case 19: {
                this.processCloudlet(evt, 19);
                return true;
            }
            case 20: {
                this.processCloudlet(evt, 20);
                return true;
            }
            case 21: {
                this.processCloudlet(evt, 21);
                return true;
            }
            case 22: {
                this.processCloudlet(evt, 22);
                return true;
            }
        }
        return false;
    }

    protected void processPingRequest(SimEvent evt) {
        IcmpPacket pkt = (IcmpPacket)evt.getData();
        pkt.setTag(106);
        pkt.setDestination(pkt.getSource());
        this.sendNow(pkt.getSource(), 106, pkt);
    }

    protected void processCloudlet(SimEvent evt, int type) {
        Cloudlet cloudlet;
        try {
            cloudlet = (Cloudlet)evt.getData();
        }
        catch (ClassCastException e) {
            LOGGER.error("{}: Error in processing Cloudlet: {}", (Object)super.getName(), (Object)e.getMessage());
            return;
        }
        switch (type) {
            case 18: {
                this.processCloudletCancel(cloudlet);
                break;
            }
            case 19: {
                this.processCloudletPause(cloudlet, false);
                break;
            }
            case 20: {
                this.processCloudletPause(cloudlet, true);
                break;
            }
            case 21: {
                this.processCloudletResume(cloudlet, false);
                break;
            }
            case 22: {
                this.processCloudletResume(cloudlet, true);
                break;
            }
            default: {
                LOGGER.trace("{}: Unable to handle a request from {} with event tag = {}", new Object[]{this, evt.getSource().getName(), evt.getTag()});
            }
        }
    }

    protected void processCloudletSubmit(SimEvent evt, boolean ack) {
        Cloudlet cloudlet = (Cloudlet)evt.getData();
        if (cloudlet.isFinished()) {
            this.notifyBrokerAboutAlreadyFinishedCloudlet(cloudlet, ack);
            return;
        }
        cloudlet.assignToDatacenter(this);
        this.submitCloudletToVm(cloudlet, ack);
    }

    private void submitCloudletToVm(Cloudlet cloudlet, boolean ack) {
        double fileTransferTime = this.getDatacenterStorage().predictFileTransferTime(cloudlet.getRequiredFiles());
        CloudletScheduler scheduler = cloudlet.getVm().getCloudletScheduler();
        double estimatedFinishTime = scheduler.cloudletSubmit(cloudlet, fileTransferTime);
        if (estimatedFinishTime > 0.0 && !Double.isInfinite(estimatedFinishTime)) {
            this.send(this, this.getCloudletProcessingUpdateInterval(estimatedFinishTime), 41);
        }
        this.sendCloudletSubmitAckToBroker(cloudlet, ack);
    }

    protected double getCloudletProcessingUpdateInterval(double nextFinishingCloudletTime) {
        if (this.schedulingInterval == 0.0) {
            return nextFinishingCloudletTime;
        }
        double time = Math.floor(this.getSimulation().clock());
        double mod = time % this.schedulingInterval;
        double delay = mod == 0.0 ? this.schedulingInterval : time - mod + this.schedulingInterval - time;
        return Math.min(nextFinishingCloudletTime, delay);
    }

    protected void processCloudletResume(Cloudlet cloudlet, boolean ack) {
        double estimatedFinishTime = cloudlet.getVm().getCloudletScheduler().cloudletResume(cloudlet);
        if (estimatedFinishTime > 0.0 && estimatedFinishTime > this.getSimulation().clock()) {
            this.schedule(this, this.getCloudletProcessingUpdateInterval(estimatedFinishTime), 41);
        }
        this.sendAck(ack, cloudlet, 22);
    }

    private void sendAck(boolean ack, Cloudlet cloudlet, int cloudSimTagAck) {
        if (ack) {
            this.sendNow(cloudlet.getBroker(), cloudSimTagAck, cloudlet);
        }
    }

    protected void processCloudletPause(Cloudlet cloudlet, boolean ack) {
        cloudlet.getVm().getCloudletScheduler().cloudletPause(cloudlet);
        this.sendAck(ack, cloudlet, 20);
    }

    protected void processCloudletCancel(Cloudlet cloudlet) {
        cloudlet.getVm().getCloudletScheduler().cloudletCancel(cloudlet);
        this.sendNow(cloudlet.getBroker(), 18, cloudlet);
    }

    protected boolean processVmCreate(SimEvent evt, boolean ackRequested) {
        Vm vm = (Vm)evt.getData();
        boolean hostAllocatedForVm = this.vmAllocationPolicy.allocateHostForVm(vm);
        if (hostAllocatedForVm) {
            if (!vm.isCreated()) {
                vm.setCreated(true);
            }
            vm.updateProcessing(vm.getHost().getVmScheduler().getAllocatedMips(vm));
        }
        if (ackRequested) {
            this.send(vm.getBroker(), this.getSimulation().getMinTimeBetweenEvents(), 32, vm);
        }
        return hostAllocatedForVm;
    }

    protected void processVmDestroy(SimEvent evt, boolean ack) {
        Vm vm = (Vm)evt.getData();
        this.vmAllocationPolicy.deallocateHostForVm(vm);
        if (ack) {
            this.sendNow(vm.getBroker(), 34, vm);
        }
        String warningMsg = this.generateNotFinishedCloudletsWarning(vm);
        String msg = String.format("%.2f: %s: %s destroyed on %s. %s", this.getSimulation().clock(), this.getClass().getSimpleName(), vm, vm.getHost(), warningMsg);
        if (warningMsg.isEmpty()) {
            LOGGER.info(msg);
        } else {
            LOGGER.warn(msg);
        }
    }

    private String generateNotFinishedCloudletsWarning(Vm vm) {
        int cloudletsNoFinished = vm.getCloudletScheduler().getCloudletList().size();
        if (cloudletsNoFinished == 0) {
            return "";
        }
        return String.format("It had a total of %d cloudlets (running + waiting). %s", cloudletsNoFinished, "Some events may have been missed. You can try: (a) decreasing CloudSim's minTimeBetweenEvents and/or Datacenter's schedulingInterval attribute; (b) increasing broker's Vm destruction delay for idle VMs if you set it to zero; (c) defining Cloudlets with smaller length (your Datacenter's scheduling interval may be smaller than the time to finish some Cloudlets).");
    }

    protected void finishVmMigration(SimEvent evt, boolean ack) {
        if (!(evt.getData() instanceof Map.Entry)) {
            throw new ClassCastException("The data object must be Map.Entry<Vm, Host>");
        }
        Map.Entry entry = (Map.Entry)evt.getData();
        Vm vm = (Vm)entry.getKey();
        Host targetHost = (Host)entry.getValue();
        this.updateHostsProcessing();
        this.vmAllocationPolicy.deallocateHostForVm(vm);
        targetHost.removeMigratingInVm(vm);
        boolean result = this.vmAllocationPolicy.allocateHostForVm(vm, targetHost);
        if (ack) {
            this.sendNow(evt.getSource(), 32, vm);
        }
        vm.setInMigration(false);
        SimEvent event = this.getSimulation().findFirstDeferred(this, new PredicateType(35));
        if (event == null || event.getTime() > this.getSimulation().clock()) {
            this.updateHostsProcessing();
        }
        if (result) {
            LOGGER.info("{}: Migration of {} to {} is completed", new Object[]{this.getSimulation().clock(), vm, targetHost});
        } else {
            LOGGER.error("{}: Allocation of {} to the destination Host failed!", (Object)this, (Object)vm);
        }
    }

    private void notifyBrokerAboutAlreadyFinishedCloudlet(Cloudlet cloudlet, boolean ack) {
        LOGGER.warn("{}: {} owned by {} is already completed/finished. It won't be executed again.", new Object[]{this.getName(), cloudlet, cloudlet.getBroker()});
        this.sendCloudletSubmitAckToBroker(cloudlet, ack);
        this.sendNow(cloudlet.getBroker(), 15, cloudlet);
    }

    private void sendCloudletSubmitAckToBroker(Cloudlet cloudlet, boolean ack) {
        if (!ack) {
            return;
        }
        this.sendNow(cloudlet.getBroker(), 17, cloudlet);
    }

    private double updateHostsProcessing() {
        double nextSimulationTime = Double.MAX_VALUE;
        for (Host host : this.getHostList()) {
            double time = host.updateProcessing(this.getSimulation().clock());
            nextSimulationTime = Math.min(time, nextSimulationTime);
        }
        double minTimeBetweenEvents = this.getSimulation().getMinTimeBetweenEvents() + 0.01;
        if ((nextSimulationTime = Math.max(nextSimulationTime, minTimeBetweenEvents)) == Double.MAX_VALUE) {
            return nextSimulationTime;
        }
        this.powerSupply.computePowerUtilizationForTimeSpan(this.lastProcessTime);
        return nextSimulationTime;
    }

    protected double updateCloudletProcessing() {
        if (!this.isTimeToUpdateCloudletsProcessing()) {
            return Double.MAX_VALUE;
        }
        double nextSimulationTime = this.updateHostsProcessing();
        if (nextSimulationTime != Double.MAX_VALUE) {
            nextSimulationTime = this.getCloudletProcessingUpdateInterval(nextSimulationTime);
            this.schedule(this, nextSimulationTime, 41);
        }
        this.setLastProcessTime(this.getSimulation().clock());
        this.checkIfVmMigrationsAreNeeded();
        return nextSimulationTime;
    }

    private boolean isTimeToUpdateCloudletsProcessing() {
        return this.getSimulation().clock() < 0.111 || this.getSimulation().clock() >= this.lastProcessTime + this.getSimulation().getMinTimeBetweenEvents();
    }

    private void checkIfVmMigrationsAreNeeded() {
        if (!this.isMigrationsEnabled()) {
            return;
        }
        Map<Vm, Host> migrationMap = this.getVmAllocationPolicy().getOptimizedAllocationMap(this.getVmList());
        for (Map.Entry<Vm, Host> entry : migrationMap.entrySet()) {
            this.requestVmMigration(entry);
        }
    }

    private void requestVmMigration(Map.Entry<Vm, Host> entry) {
        double currentTime = this.getSimulation().clock();
        Host sourceHost = entry.getKey().getHost();
        Host targetHost = entry.getValue();
        double delay = this.timeToMigrateVm(entry.getKey(), targetHost);
        String msg1 = sourceHost == Host.NULL ? String.format("%.2f: Migration of %s to %s is started.", currentTime, entry.getKey(), targetHost) : String.format("%.2f: Migration of %s from %s to %s is started.", currentTime, entry.getKey(), sourceHost, targetHost);
        String msg2 = String.format("It's expected to finish in %.2f seconds, considering the %.0f%% of bandwidth allowed for migration and the VM RAM size.", delay, this.getBandwidthPercentForMigration() * 100.0);
        LOGGER.info("{}{}{}", new Object[]{msg1, System.lineSeparator(), msg2});
        sourceHost.addVmMigratingOut(entry.getKey());
        targetHost.addMigratingInVm(entry.getKey());
        this.send(this, delay, 35, entry);
    }

    private double timeToMigrateVm(Vm vm, Host targetHost) {
        return (double)vm.getRam().getCapacity() / Conversion.bitesToBytes((double)targetHost.getBw().getCapacity() * this.getBandwidthPercentForMigration());
    }

    @Override
    public void shutdownEntity() {
        super.shutdownEntity();
        LOGGER.info("{}: {} is shutting down...", (Object)this.getSimulation().clock(), (Object)this.getName());
    }

    @Override
    protected void startEntity() {
        LOGGER.info("{} is starting...", (Object)this.getName());
        this.sendNow(this.getSimulation().getCloudInfoService(), 2, this);
    }

    @Override
    public <T extends Host> List<T> getHostList() {
        return Collections.unmodifiableList(this.hostList);
    }

    @Override
    public DatacenterCharacteristics getCharacteristics() {
        return this.characteristics;
    }

    @Override
    public VmAllocationPolicy getVmAllocationPolicy() {
        return this.vmAllocationPolicy;
    }

    public final Datacenter setVmAllocationPolicy(VmAllocationPolicy vmAllocationPolicy) {
        Objects.requireNonNull(vmAllocationPolicy);
        if (vmAllocationPolicy.getDatacenter() != null && vmAllocationPolicy.getDatacenter() != Datacenter.NULL && !this.equals(vmAllocationPolicy.getDatacenter())) {
            throw new IllegalStateException("The given VmAllocationPolicy is already used by another Datacenter.");
        }
        vmAllocationPolicy.setDatacenter(this);
        this.vmAllocationPolicy = vmAllocationPolicy;
        return this;
    }

    protected double getLastProcessTime() {
        return this.lastProcessTime;
    }

    protected final void setLastProcessTime(double lastProcessTime) {
        this.lastProcessTime = lastProcessTime;
    }

    @Override
    public DatacenterStorage getDatacenterStorage() {
        return this.datacenterStorage;
    }

    @Override
    public final void setDatacenterStorage(DatacenterStorage datacenterStorage) {
        datacenterStorage.setDatacenter(this);
        this.datacenterStorage = datacenterStorage;
    }

    private <T extends Vm> List<T> getVmList() {
        return Collections.unmodifiableList(this.getHostList().stream().flatMap(h -> h.getVmList().stream()).collect(Collectors.toList()));
    }

    @Override
    public double getSchedulingInterval() {
        return this.schedulingInterval;
    }

    @Override
    public final Datacenter setSchedulingInterval(double schedulingInterval) {
        this.schedulingInterval = Math.max(schedulingInterval, 0.0);
        return this;
    }

    @Override
    public Host getHost(int index) {
        if (index >= 0 && index < this.getHostList().size()) {
            return (Host)this.getHostList().get(index);
        }
        return Host.NULL;
    }

    @Override
    public long getActiveHostsNumber() {
        return this.hostList.stream().filter(Host::isActive).count();
    }

    @Override
    public Host getHostById(long id) {
        return this.hostList.stream().filter(host -> host.getId() == id).findFirst().map(host -> host).orElse(Host.NULL);
    }

    @Override
    public <T extends Host> Datacenter addHostList(List<T> hostList) {
        Objects.requireNonNull(hostList);
        hostList.forEach(this::addHost);
        return this;
    }

    @Override
    public <T extends Host> Datacenter addHost(T host) {
        if (this.vmAllocationPolicy == null || this.vmAllocationPolicy == VmAllocationPolicy.NULL) {
            throw new IllegalStateException("A VmAllocationPolicy must be set before adding a new Host to the Datacenter.");
        }
        if (host.getId() <= -1L) {
            host.setId(this.getHostList().size());
        }
        host.setDatacenter(this);
        this.hostList.add(host);
        this.vmAllocationPolicy.setDatacenter(this);
        return this;
    }

    private <T extends Host> void notifyOnHostAvailableListeners(T host) {
        this.onHostAvailableListeners.forEach(listener -> listener.update(HostEventInfo.of(listener, host, this.getSimulation().clock())));
    }

    @Override
    public <T extends Host> Datacenter removeHost(T host) {
        this.hostList.remove(host);
        ((VmAllocationPolicyAbstract)this.vmAllocationPolicy).addPesFromHost(host);
        return this;
    }

    public String toString() {
        return String.format("Datacenter %d", this.getId());
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (object == null || this.getClass() != object.getClass()) {
            return false;
        }
        if (!super.equals(object)) {
            return false;
        }
        DatacenterSimple that = (DatacenterSimple)object;
        return !this.characteristics.equals(that.characteristics);
    }

    @Override
    public int hashCode() {
        int result = super.hashCode();
        result = 31 * result + this.characteristics.hashCode();
        return result;
    }

    @Override
    public double getBandwidthPercentForMigration() {
        return this.bandwidthPercentForMigration;
    }

    @Override
    public void setBandwidthPercentForMigration(double bandwidthPercentForMigration) {
        if (bandwidthPercentForMigration <= 0.0) {
            throw new IllegalArgumentException("The bandwidth migration percentage must be greater than 0.");
        }
        if (bandwidthPercentForMigration > 1.0) {
            throw new IllegalArgumentException("The bandwidth migration percentage must be lower or equal to 1.");
        }
        this.bandwidthPercentForMigration = bandwidthPercentForMigration;
    }

    @Override
    public double getPower() throws UnsupportedOperationException {
        double power = this.powerSupply.getPower();
        if (power < 0.0) {
            throw new UnsupportedOperationException("The power consumption for " + this + " cannot be computed because a DatacenterPowerSupply object was not given. Call the setPowerSupply() before the simulation start to provide one. This enables power consumption computation.");
        }
        return power;
    }

    @Override
    public Datacenter addOnHostAvailableListener(EventListener<HostEventInfo> listener) {
        this.onHostAvailableListeners.add(Objects.requireNonNull(listener));
        return this;
    }

    public boolean isMigrationsEnabled() {
        return this.migrationsEnabled;
    }

    public final Datacenter enableMigrations() {
        this.migrationsEnabled = true;
        return this;
    }

    public final Datacenter disableMigrations() {
        this.migrationsEnabled = false;
        return this;
    }

    @Override
    public void setPowerSupply(DatacenterPowerSupply powerSupply) {
        this.powerSupply = powerSupply == null ? DatacenterPowerSupply.NULL : powerSupply.setDatacenter(this);
    }
}

