/*
 * Decompiled with CFR 0.152.
 */
package org.cloudbus.cloudsim.resources;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.apache.commons.lang3.StringUtils;
import org.cloudbus.cloudsim.distributions.ContinuousDistribution;
import org.cloudbus.cloudsim.resources.File;
import org.cloudbus.cloudsim.resources.FileStorage;
import org.cloudbus.cloudsim.resources.Storage;
import org.cloudbus.cloudsim.util.Conversion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HarddriveStorage
implements FileStorage {
    private static final double DEF_LATENCY_SECS = 0.00417;
    private static final double DEF_SEEK_TIME_SECS = 0.009;
    private static final int DEF_MAX_TRANSFER_RATE_MBITS_SEC = 1064;
    private static final Logger LOGGER = LoggerFactory.getLogger((String)HarddriveStorage.class.getSimpleName());
    private final Storage storage;
    private final Storage reservedStorage;
    private List<String> fileNameList;
    private List<File> fileList;
    private final String name;
    private ContinuousDistribution gen;
    private double maxTransferRate;
    private double latency;
    private double avgSeekTime;

    public HarddriveStorage(String name, long capacity) throws IllegalArgumentException {
        if (StringUtils.isBlank((CharSequence)name)) {
            throw new IllegalArgumentException("Storage name cannot be empty.");
        }
        this.fileList = new ArrayList<File>();
        this.fileNameList = new ArrayList<String>();
        this.storage = new Storage(capacity);
        this.reservedStorage = new Storage(capacity);
        this.name = name;
        this.init();
    }

    public HarddriveStorage(long capacity) throws IllegalArgumentException {
        this("HarddriveStorage", capacity);
    }

    private void init() {
        this.setLatency(0.00417);
        this.setAvgSeekTime(0.009);
        this.setMaxTransferRate(1064.0);
    }

    @Override
    public int getNumStoredFile() {
        return this.fileList.size();
    }

    @Override
    public boolean reserveSpace(int fileSize) {
        if (this.storage.allocateResource(fileSize)) {
            this.reservedStorage.allocateResource(fileSize);
            return true;
        }
        return false;
    }

    @Override
    public double addReservedFile(File file) {
        Objects.requireNonNull(file);
        if (!this.reservedStorage.isResourceAmountBeingUsed(file.getSize())) {
            throw new IllegalStateException("The file size wasn't previously reserved in order to add a reserved file.");
        }
        long fileSize = file.getSize();
        this.storage.deallocateResource(fileSize);
        this.reservedStorage.deallocateResource(fileSize);
        double result = this.addFile(file);
        if (result == 0.0) {
            this.storage.allocateResource(fileSize);
        }
        return result;
    }

    @Override
    public boolean hasPotentialAvailableSpace(int fileSize) {
        if (fileSize <= 0) {
            return false;
        }
        if (this.storage.isAmountAvailable(fileSize)) {
            return true;
        }
        return this.getDeletedFilesTotalSize() > fileSize;
    }

    @Override
    public boolean hasFile(String fileName) {
        return this.getFile(fileName) != null;
    }

    private int getDeletedFilesTotalSize() {
        return this.fileList.stream().filter(File::isDeleted).mapToInt(File::getSize).sum();
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public void setLatency(double latency) {
        if (latency < 0.0) {
            throw new IllegalArgumentException("Latency must be greater than zero.");
        }
        this.latency = latency;
    }

    @Override
    public double getLatency() {
        return this.latency;
    }

    @Override
    public void setMaxTransferRate(double maxTransferRate) {
        if (maxTransferRate <= 0.0) {
            throw new IllegalArgumentException("Max transfer rate must be greater than zero.");
        }
        this.maxTransferRate = maxTransferRate;
    }

    @Override
    public double getMaxTransferRate() {
        return this.maxTransferRate;
    }

    public boolean setAvgSeekTime(double seekTime) {
        return this.setAvgSeekTime(seekTime, null);
    }

    public boolean setAvgSeekTime(double seekTime, ContinuousDistribution gen) {
        if (seekTime <= 0.0) {
            return false;
        }
        this.avgSeekTime = seekTime;
        this.gen = gen;
        return true;
    }

    public double getAvgSeekTime() {
        return this.avgSeekTime;
    }

    @Override
    public File getFile(String fileName) {
        if (!File.isValid(fileName)) {
            LOGGER.warn("{}.getFile(): Invalid file name {}.", (Object)this.name, (Object)fileName);
            return null;
        }
        int size = 0;
        for (File currentFile : this.fileList) {
            size += currentFile.getSize();
            if (!currentFile.getName().equals(fileName)) continue;
            double seekTime = this.getSeekTime(size);
            double transferTime = this.getTransferTime(currentFile.getSize());
            currentFile.setTransactionTime(seekTime + transferTime);
            return currentFile;
        }
        return null;
    }

    @Override
    public List<String> getFileNameList() {
        return Collections.unmodifiableList(this.fileNameList);
    }

    @Override
    public List<File> getFileList() {
        return Collections.unmodifiableList(this.fileList);
    }

    private double getSeekTime(int fileSize) {
        double result = 0.0;
        if (this.gen != null) {
            result += this.gen.sample();
        }
        if (fileSize > 0 && this.storage.getCapacity() != 0L) {
            result += (double)fileSize / (double)this.storage.getCapacity();
        }
        return result;
    }

    @Override
    public double getTransferTime(String fileName) {
        File file = this.getFile(fileName);
        if (file == null) {
            return -1.0;
        }
        return this.getTransferTime(file);
    }

    @Override
    public double getTransferTime(File file) {
        return this.getTransferTime(file.getSize());
    }

    @Override
    public double getTransferTime(int fileSize) {
        return this.getTransferTime(fileSize, this.getMaxTransferRate()) + this.getLatency();
    }

    protected final double getTransferTime(int fileSize, double speed) {
        return Conversion.bytesToBits(fileSize) / speed;
    }

    @Override
    public double addFile(File file) {
        double result = 0.0;
        if (!File.isValid(file)) {
            LOGGER.warn("{}.addFile(): Invalid file {}", (Object)this.name, (Object)file);
            return result;
        }
        if (!this.storage.isAmountAvailable(file.getSize())) {
            LOGGER.error("{}.addFile(): Not enough space to store {}", (Object)this.name, (Object)file.getName());
            return result;
        }
        if (!this.contains(file.getName())) {
            this.fileList.add(file);
            this.fileNameList.add(file.getName());
            this.storage.allocateResource(file.getSize());
            result = this.getTotalFileAddTime(file);
            file.setTransactionTime(result);
        }
        return result;
    }

    private double getTotalFileAddTime(File file) {
        double seekTime = this.getSeekTime(file.getSize());
        double transferTime = this.getTransferTime(file.getSize());
        return seekTime + transferTime;
    }

    @Override
    public double addFile(List<File> list) {
        Objects.requireNonNull(list);
        if (list.isEmpty()) {
            LOGGER.debug("{}.addFile(): File list is empty.", (Object)this.getName());
            return 0.0;
        }
        return list.stream().mapToDouble(this::addFile).sum();
    }

    @Override
    public File deleteFile(String fileName) {
        if (!File.isValid(fileName)) {
            return null;
        }
        int i = this.fileNameList.indexOf(fileName);
        if (i != -1) {
            File file = this.fileList.get(i);
            double result = this.deleteFile(file);
            file.setTransactionTime(result);
            return file;
        }
        return null;
    }

    @Override
    public double deleteFile(File file) {
        if (!File.isValid(file)) {
            return 0.0;
        }
        if (this.contains(file)) {
            this.fileList.remove(file);
            this.fileNameList.remove(file.getName());
            this.storage.deallocateResource(file.getSize());
            double result = this.getTotalFileAddTime(file);
            file.setTransactionTime(result);
            return result;
        }
        return 0.0;
    }

    @Override
    public boolean contains(String fileName) {
        if (StringUtils.isBlank((CharSequence)fileName)) {
            LOGGER.warn("{}.contains(): Invalid file name {}", (Object)this.name, (Object)fileName);
            return false;
        }
        return this.fileNameList.contains(fileName);
    }

    @Override
    public boolean contains(File file) {
        if (!File.isValid(file)) {
            return false;
        }
        return this.contains(file.getName());
    }

    @Override
    public boolean renameFile(File file, String newName) {
        if (this.contains(newName)) {
            return false;
        }
        String oldName = file.getName();
        File renamedFile = this.getFile(oldName);
        if (renamedFile != null) {
            renamedFile.setName(newName);
            renamedFile.setTransactionTime(0.0);
            this.fileNameList.remove(oldName);
            this.fileNameList.add(newName);
            return true;
        }
        return false;
    }

    @Override
    public long getCapacity() {
        return this.storage.getCapacity();
    }

    @Override
    public long getAvailableResource() {
        return this.storage.getAvailableResource();
    }

    @Override
    public long getAllocatedResource() {
        return this.storage.getAllocatedResource();
    }

    @Override
    public boolean isAmountAvailable(long amountToCheck) {
        return this.storage.isAmountAvailable(amountToCheck);
    }

    @Override
    public boolean isAmountAvailable(double amountToCheck) {
        return this.isAmountAvailable((long)amountToCheck);
    }

    @Override
    public boolean isFull() {
        return this.storage.isFull();
    }
}

