/*
 * Sonar, entreprise quality control tool.
 * Copyright (C) 2007-2008 Hortis-GRC SA
 * mailto:be_agile HAT hortis DOT ch
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package ch.hortis.sonar.mvn;

import ch.hortis.sonar.model.Snapshot;
import ch.hortis.sonar.service.WebInterfaceService;
import ch.hortis.sonar.service.WebInterfaceServiceImpl;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.sonar.commons.DaoFacade;

import javax.persistence.Query;
import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

/**
 * Creates the snapshots hierarchy and launch the batch on the sonar server side
 *
 * @goal batch
 * @aggregator
 */
public class BatchLaunchMojo extends CoreMojo {

  /**
   * Sonar host URL
   *
   * @parameter expression="${sonar.host.url}" default-value="http://localhost:9000" alias="sonar.host.url"
   */
  private String sonarHostURL;

  private WebInterfaceService webInterfaceService;
  private DaoFacade daoFacade;

  public void setWebInterfaceService(WebInterfaceService webInterfaceService) {
    this.webInterfaceService = webInterfaceService;
  }

  public void doExecute() throws MojoExecutionException {
    try {
      daoFacade = (DaoFacade) getContainer().getComponent(DaoFacade.class);

      Set<Integer> snapshotIds = processSnapshots();

      Snapshot topLevel = getTopLevelSnapshot();
      if (topLevel == null) {
        throw new MojoExecutionException("Unable to find top level snapshot");
      }

      updateRootColumn(snapshotIds, topLevel);

      notifyServer(topLevel);

    } catch (Exception e) {
      throw new MojoExecutionException("Error occured when triggering measures calculation job on remote server", e);

    }
  }

  private void notifyServer(Snapshot topLevel) throws IOException {
    if (webInterfaceService == null) {
      webInterfaceService = new WebInterfaceServiceImpl(sonarHostURL);
    }
    webInterfaceService.triggerMeasuresCalculations(topLevel.getId());
  }

  private void updateRootColumn(Set<Integer> snapshotIds, Snapshot rootSnapshot) {
    // the root snapshot has root==null
    snapshotIds.remove(rootSnapshot.getId());
    if (!snapshotIds.isEmpty()) {
      Query query = daoFacade.getDatabaseManager().createQuery("update Snapshot s set s.root=:root where s.id in (:snapshotIds) or s.root in (:snapshotIds)");
      query.setParameter("root", rootSnapshot);
      query.setParameter("snapshotIds", snapshotIds);
      query.executeUpdate();
      daoFacade.getDatabaseManager().commit();
    }
  }


  private Set<Integer> processSnapshots() throws MojoExecutionException {
    Set<Integer> snapshotIds = new HashSet<Integer>();
    for (MavenProject reactorProject : getReactorProjects()) {
      processSnapshot(reactorProject, null, snapshotIds);
    }
    daoFacade.getDatabaseManager().commit();
    return snapshotIds;
  }

  private void processSnapshot(MavenProject reactorProject, Snapshot parent, Set<Integer> snapshotIds) throws MojoExecutionException {
    Integer projectBoundSnapshotId = getSnapshotId(reactorProject);
    Snapshot projectBoundSnapshot = daoFacade.getDatabaseManager().getEntityManager().find(Snapshot.class, projectBoundSnapshotId);
    if (projectBoundSnapshot == null) {
      throw new MojoExecutionException("Unable to find Snapshot " + projectBoundSnapshotId + " bound to the pom");
    }
    snapshotIds.add(projectBoundSnapshotId);
    if (parent != null) {
      projectBoundSnapshot.setParent(parent);
    }

    daoFacade.getDatabaseManager().merge(projectBoundSnapshot);
    saveSnapshotHierarchyModules(projectBoundSnapshot, reactorProject, snapshotIds);
  }

  private void saveSnapshotHierarchyModules(Snapshot snapshot, MavenProject reactorProject, Set<Integer> snapshotIds) throws MojoExecutionException {
    for (Object moduleNames : reactorProject.getModules()) {
      String moduleName = (String) moduleNames;
      File moduleFile = new File(reactorProject.getBasedir() + "/" + moduleName + "/pom.xml");
      // cannot use the readen module Project to create the hierarchy, must use one of
      // the maven object in the getReactorProjects() array
      MavenProject targetReactorProject = null;
      for (MavenProject project : getReactorProjects()) {
        try {
          if (project.getFile().getCanonicalFile().equals(moduleFile.getCanonicalFile())) {
            targetReactorProject = project;
            break;
          }
        } catch (IOException e) {
          throw new MojoExecutionException("Unable to resolve canonical path of pom.xml", e);
        }
      }
      if (targetReactorProject != null) {
	    // see SONAR-294
        processSnapshot(targetReactorProject, snapshot, snapshotIds);
      }      
    }
  }

  private Snapshot getTopLevelSnapshot() throws MojoExecutionException {
    for (org.apache.maven.project.MavenProject reactorProject : getReactorProjects()) {
      Integer projectBoundSnapshotId = super.getSnapshotId(reactorProject);
      Snapshot projectBoundSnapshot = daoFacade.getDatabaseManager().getEntityManager().find(Snapshot.class, projectBoundSnapshotId);
      if (projectBoundSnapshot.getParent() == null) {
        if (getReactorProjects().size() == 1 ||
            daoFacade.getMeasuresDao().getChildSnapshots(projectBoundSnapshot, false).size() > 0) {
          return projectBoundSnapshot;
        }
      }
    }
    return null;
  }
}