/*
 * Sonar, entreprise quality control tool.
 * Copyright (C) 2007-2008 Hortis-GRC SA
 * mailto:be_agile HAT hortis DOT ch
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package ch.hortis.sonar.mvn;

import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.sonar.commons.database.DatabaseConnector;
import org.sonar.commons.database.DatabaseProperties;
import org.sonar.commons.database.DriverDatabaseConnector;
import org.sonar.maven.MavenContainer;
import org.sonar.plugins.api.maven.model.MavenPom;

import java.util.List;

public abstract class CoreMojo extends AbstractMojo {

  private final static String SNAPSHOT_ID_PROP = "sonar.snaphsotId";

  /**
   * The projects in the reactor.
   *
   * @parameter expression="${reactorProjects}"
   * @readonly
   */
  private List<MavenProject> reactorProjects;

  /**
   * The maven project running this plugin
   *
   * @parameter expression="${project}"
   * @required
   * @readonly
   */
  protected MavenProject mavenProject;

  /**
   * JDBC URL
   *
   * @parameter expression="${sonar.jdbc.url}" default-value="jdbc:derby://localhost:1527/sonar" alias="sonar.jdbc.url"
   */
  private String jdbcURL;

  /**
   * JDBC driver class
   *
   * @parameter expression="${sonar.jdbc.driver}" default-value="org.apache.derby.jdbc.ClientDriver" alias="sonar.jdbc.driver"
   */
  private String jdbcDriverClassName;

  /**
   * JDBC user name
   *
   * @parameter expression="${sonar.jdbc.username}" default-value="sonar" alias="sonar.jdbc.username"
   */
  private String jdbcUserName;

  /**
   * JDBC password
   *
   * @parameter expression="${sonar.jdbc.password}" default-value="sonar" alias="sonar.jdbc.password"
   */
  private String jdbcPassword;

  /**
   * JDBC dialect (values : db2, oracle, postgre, hsqldb, mysql, mssql). Do not set, it should be autodetected.
   *
   * @parameter expression="${sonar.jdbc.dialect}" alias="sonar.jdbc.dialect"
   */
  private String jdbcDialect;

  /**
   * Hibernate dialect class. Do not set, it should be autodetected.
   *
   * @parameter expression="${sonar.jdbc.dialectClass}" alias="sonar.jdbc.dialectClass"
   */
  private String jdbcDialectClass;

  /**
   * Project branch
   *
   * @parameter expression="${branch}" alias="branch"
   */
  private String branch;

  private MavenContainer container;

  private DatabaseConnector databaseConnector = null;

  public void setDatabaseConnector(DatabaseConnector databaseConnector) {
    this.databaseConnector = databaseConnector;
  }

  public final void execute() throws MojoExecutionException, MojoFailureException {
    try {
      doExecute();

    } finally {
      if (container != null) {
        try {
          container.stop();
        } catch (Throwable e) {
          // don't care
        }
        container = null;
      }
    }
  }

  protected abstract void doExecute() throws MojoExecutionException, MojoFailureException;

  protected void setReactorProjects(List<MavenProject> reactorProjects) {
    this.reactorProjects = reactorProjects;
  }

  protected List<MavenProject> getReactorProjects() {
    return reactorProjects;
  }

  protected String getBranch() {
    return branch;
  }

  protected void setBranch(String branch) {
    this.branch = branch;
  }

  protected void setProject(MavenProject mavenProject) {
    this.mavenProject = mavenProject;
  }

  protected void setJdbcURL(String jdbcURL) {
    this.jdbcURL = jdbcURL;
  }

  protected void setJdbcDriverClassName(String jdbcDriverClassName) {
    this.jdbcDriverClassName = jdbcDriverClassName;
  }

  protected void setJdbcUserName(String jdbcUserName) {
    this.jdbcUserName = jdbcUserName;
  }

  protected void setJdbcPassword(String jdbcPassword) {
    this.jdbcPassword = jdbcPassword;
  }

  protected Integer getSnapshotId(MavenProject pom) {
    String propValue = pom.getProperties().getProperty(SNAPSHOT_ID_PROP);
    if (propValue == null) {
      throw new IllegalStateException("Unable to retrieve property '" + SNAPSHOT_ID_PROP + "' from pom");
    }
    return Integer.valueOf(propValue);
  }

  protected void setSnapshotId(Integer id, MavenProject pom) {
    pom.getProperties().setProperty(SNAPSHOT_ID_PROP, id.toString());
  }

  protected DatabaseConnector getDatabaseConnector() {
    if (databaseConnector == null) {
      PropertiesConfiguration config = new PropertiesConfiguration();
      config.setProperty(DatabaseProperties.PROP_URL, jdbcURL);
      config.setProperty(DatabaseProperties.PROP_DRIVER, jdbcDriverClassName);
      config.setProperty(DatabaseProperties.PROP_USER, jdbcUserName);
      config.setProperty(DatabaseProperties.PROP_PASSWORD, jdbcPassword);
      //config.setProperty(DatabaseProperties.PROP_ISOLATION, Connection.TRANSACTION_READ_COMMITTED);
      config.setProperty(DatabaseProperties.PROP_DIALECT, jdbcDialect);
      config.setProperty(DatabaseProperties.PROP_DIALECT_CLASS, jdbcDialectClass);
      config.setProperty(DatabaseProperties.PROP_HIBERNATE_SECOND_LEVEL_CACHE, Boolean.FALSE);
      databaseConnector = new DriverDatabaseConnector(config);
    }
    return databaseConnector;
  }

  protected MavenContainer getContainer() {
    if (container == null) {
      // components are started/stopped by reflection. It executed the following methods : start, stop, dispos
      // one instance per component
      container = new MavenContainer(getLog(), getDatabaseConnector(), mavenProject);
      container.start();
    }
    return container;
  }

  private MavenPom mavenPom = null;

  protected MavenPom getMavenPom() {
    if (mavenPom == null) {
      mavenPom = new MavenPom(mavenProject);
    }
    return mavenPom;
  }
}