/*
 * Sonar, entreprise quality control tool.
 * Copyright (C) 2007-2008 Hortis-GRC SA
 * mailto:be_agile HAT hortis DOT ch
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.maven;

import ch.hortis.sonar.model.MavenProject;
import ch.hortis.sonar.model.ProjectLink;
import ch.hortis.sonar.model.Snapshot;
import ch.hortis.sonar.mvn.CoreMojo;
import org.apache.maven.model.CiManagement;
import org.apache.maven.model.IssueManagement;
import org.apache.maven.model.Scm;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.commons.configuration.Configuration;
import org.sonar.commons.DaoFacade;
import org.sonar.commons.database.DatabaseManager;
import org.sonar.plugins.api.maven.AbstractMeasuresRecorder;
import org.sonar.plugins.api.maven.MavenCollector;
import org.sonar.plugins.api.maven.MavenUtils;

import javax.persistence.NoResultException;
import javax.persistence.Query;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

/**
 * @goal collect
 */
public class CollectMojo extends CoreMojo {
  public static final String PARAM_DATE = "sonar.projectDate";
  public static final String PARAM_VERSION = "sonar.projectVersion";

  public void doExecute() throws MojoExecutionException {
    Snapshot snapshot = createSnapshot();

    for (AbstractMeasuresRecorder recorder : getContainer().getComponents(AbstractMeasuresRecorder.class)) {
      recorder.setRootSnapshot(snapshot);
    }

    List<MavenCollector> collectors = getContainer().getMavenCollectors();
    for (MavenCollector collector : collectors) {
      try {
        getLog().info("Starting " + collector.getClass());
        collector.collect(getMavenPom());
      }
      catch (RuntimeException e) {
        if (collector.shouldStopOnFailure()) {
          throw e;
        }
        getLog().error("error while collecting " + collector.getClass().getSimpleName());
      }
    }

    setSnapshotId(snapshot.getId(), mavenProject);
  }


  private Snapshot createSnapshot() {
    DaoFacade daoFacade = (DaoFacade) getContainer().getComponent(DaoFacade.class);
    DatabaseManager dbManager = daoFacade.getDatabaseManager();

    MavenProject dbProject = loadProjectFromDatabase(mavenProject.getGroupId(), mavenProject.getArtifactId(), MavenUtils.getBranch(mavenProject), dbManager);
    if (dbProject == null) {
      dbProject = MavenProject.newMavenProject(mavenProject.getGroupId(), mavenProject.getArtifactId(), MavenUtils.getBranch(mavenProject), mavenProject.getName());
      dbProject.setQualifier(getMavenPom().getLanguage().getProjectQualifier());
    }
    dbProject.setName(mavenProject.getName());
    dbProject.setDescription(mavenProject.getDescription());
    dbProject = (MavenProject) dbManager.save(dbProject);

    updateProject(dbProject, mavenProject);

    Snapshot snapshot = new Snapshot();
    snapshot.setMavenProject(dbProject);
    snapshot.setVersion(getProjectVersion(getContainer().getConfiguration(), mavenProject));
    snapshot.setCreatedAt(getProjectDate(getContainer().getConfiguration()));

    dbManager.save(snapshot);
    return snapshot;
  }

  protected String getProjectVersion(Configuration configuration, org.apache.maven.project.MavenProject pom) {
    String version = configuration.getString(PARAM_VERSION);
    if (version==null) {
      version= pom.getVersion();
    }
    return version;
  }

  protected Date getProjectDate(Configuration configuration) {
    String formattedDate = configuration.getString(PARAM_DATE);
    if (formattedDate == null) {
      return new Date();
    }

    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
    try {
      return format.parse(formattedDate);

    } catch (ParseException e) {
      throw new RuntimeException("The property " + PARAM_DATE + " does not respect the format yyyy-MM-dd (for example 2008-05-23) : " + formattedDate, e);
    }
  }

  private MavenProject loadProjectFromDatabase(String groupId, String artifactId, String branch, DatabaseManager dbManager) {
    try {
      Query query = dbManager.getEntityManager().createQuery("SELECT p FROM MavenProject p WHERE p.scope=:scope AND p.key=:key AND p.enabled=true");
      query.setParameter("scope", MavenProject.SCOPE_PROJECT);
      query.setParameter("key", MavenProject.toMavenKey(groupId, artifactId, branch));
      return (MavenProject) query.getSingleResult();

    } catch (NoResultException e) {
      return null;
    }
  }

  private void updateProject(MavenProject sonarProject, org.apache.maven.project.MavenProject pom) {
    updateProjectLink(ProjectLink.LINK_HOME_PAGE, pom.getUrl(), sonarProject);
    Scm scmConfig = pom.getScm();
    if (scmConfig == null) {
      scmConfig = new Scm();
    }
    updateProjectLink(ProjectLink.LINK_SCM_URL, scmConfig.getUrl(), sonarProject);
    updateProjectLink(ProjectLink.LINK_SCM_DEV_CONNECTION, scmConfig.getDeveloperConnection(), sonarProject);
    updateProjectLink(ProjectLink.LINK_SCM_RO_CONNECTION, scmConfig.getConnection(), sonarProject);

    CiManagement cimConfig = pom.getCiManagement();
    if (cimConfig == null) {
      cimConfig = new CiManagement();
    }
    updateProjectLink(ProjectLink.LINK_CONTINUOUS_INTEGRATION, cimConfig.getUrl(), sonarProject);

    IssueManagement imConfig = pom.getIssueManagement();
    if (imConfig == null) {
      imConfig = new IssueManagement();
    }
    updateProjectLink(ProjectLink.LINK_ISSUES_TRACKER, imConfig.getUrl(), sonarProject);
  }

  private void updateProjectLink(String linkType, String href, MavenProject sonarProject) {
    if (href != null && !"".equals(href)) {
      ProjectLink link = sonarProject.getProjectLinkByType(linkType);
      if (link == null) {
        link = new ProjectLink();
        link.setMavenProject(sonarProject);
        link.setType(linkType);
        sonarProject.getProjectLinks().add(link);
      }
      link.setHref(href);
    }
  }
}
