/*
 * Sonar, entreprise quality control tool.
 * Copyright (C) 2007-2008 Hortis-GRC SA
 * mailto:be_agile HAT hortis DOT ch
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.maven;

import org.apache.commons.configuration.Configuration;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.picocontainer.Characteristics;
import org.picocontainer.MutablePicoContainer;
import org.sonar.commons.Context;
import org.sonar.commons.DaoFacade;
import org.sonar.commons.IocContainer;
import org.sonar.commons.database.DatabaseConnector;
import org.sonar.commons.database.DatabaseManagerFactory;
import org.sonar.plugins.api.Plugins;
import org.sonar.plugins.api.Extension;
import org.sonar.plugins.api.maven.JavaMeasuresRecorder;
import org.sonar.plugins.api.maven.MavenCollector;
import org.sonar.plugins.api.maven.MavenCollectorFilter;
import org.sonar.plugins.api.maven.MavenExtension;
import org.sonar.plugins.api.maven.MavenPluginHandler;
import org.sonar.plugins.api.maven.PlsqlMeasuresRecorder;
import org.sonar.plugins.api.maven.model.MavenPom;
import org.sonar.plugins.api.rules.RulesManagerImpl;

import java.util.ArrayList;
import java.util.List;

public class MavenContainer {
  private MutablePicoContainer pico;

  public MavenContainer(Log mavenLog, DatabaseConnector databaseConnector, MavenProject pom) {
    pico = IocContainer.buildPicoContainer();
    Plugins plugins = new Plugins();
    pico.as(Characteristics.CACHE).addComponent(Context.MAVEN);
    pico.as(Characteristics.CACHE).addComponent(pom);
    pico.as(Characteristics.CACHE).addComponent(mavenLog);
    pico.as(Characteristics.CACHE).addComponent(databaseConnector);
    pico.as(Characteristics.CACHE).addComponent(DaoFacade.class);
    pico.as(Characteristics.CACHE).addComponent(plugins);
    pico.as(Characteristics.CACHE).addComponent(RulesManagerImpl.class);
    pico.as(Characteristics.CACHE).addComponent(DatabaseManagerFactory.class);
    pico.as(Characteristics.CACHE).addComponent(MavenConfiguration.class);
    pico.as(Characteristics.CACHE).addComponent(MavenPom.class);
    pico.as(Characteristics.CACHE).addComponent(JavaMeasuresRecorder.class);
    pico.as(Characteristics.CACHE).addComponent(PlsqlMeasuresRecorder.class);
    loadExtensions(pico, plugins.getExtensions(MavenExtension.class));
  }

  public void start() {
    pico.start();
  }

  public void stop() {
    pico.stop();
    pico = null;
  }

  private MavenPom getMavenPom() {
    return pico.getComponent(MavenPom.class);
  }

  public List<MavenCollector> getMavenCollectors() {
    List<MavenCollector> result = new ArrayList<MavenCollector>();
    MavenPom pom = getMavenPom();
    for (MavenCollector mavenCollector : pico.getComponents(MavenCollector.class)) {
      if (mavenCollector.shouldCollectOn(pom)) {
        result.add(mavenCollector);
      }
    }

    for (MavenCollectorFilter filter : pico.getComponents(MavenCollectorFilter.class)) {
      filter.filter(result, pom);
    }

    return result;
  }

  /**
   * warning : each call creates and starts a new pico container that will be destroyed at the end of the MOJO execution.
   */
  public List<MavenPluginHandler> getMavenPluginHandlers() {
    MutablePicoContainer childPico = pico.makeChildContainer();
    for (MavenCollector mavenCollector : getMavenCollectors()) {
      Class<? extends MavenPluginHandler> mavenPluginHandlerClass = mavenCollector.dependsOnMavenPlugin(getMavenPom());
      if (mavenPluginHandlerClass != null) {
        childPico.addComponent(mavenPluginHandlerClass);
      }
    }
    childPico.start();

    return childPico.getComponents(MavenPluginHandler.class);
  }

  public Configuration getConfiguration() {
    return pico.getComponent(Configuration.class);
  }

  public Object getComponent(Object key) {
    return pico.getComponent(key);
  }

  public <T> List<T> getComponents(Class<T> key) {
    return pico.getComponents(key);
  }

  public void loadExtensions(MutablePicoContainer container, List<Class<? extends Extension>> extensionClasses) {
    for (Class<? extends Extension> extensionClass : extensionClasses) {
      container.as(Characteristics.CACHE).addComponent(extensionClass);
    }
  }
}
