package org.sonar.dev;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.LineIterator;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.StringUtils;

import java.io.File;
import java.io.IOException;

/**
 * @goal trim
 */
public class TrimMojo extends AbstractMojo {

  /**
   * @parameter
   * @required
   */
  private File directory;

  /**
   * List of ant-style patterns. If
   * this is not specified, allfiles in the project source directories are included.
   *
   * @parameter
   */
  private String[] includes;

  /**
   * @parameter
   */
  private String[] excludes;


  /**
   * Specifies the encoding of the source files.
   *
   * @parameter expression="${encoding}" default-value="${project.build.sourceEncoding}"
   */
  private String sourceEncoding;

  public void execute() throws MojoExecutionException, MojoFailureException {
    if (directory == null || !directory.exists()) {
      return;
    }

    File[] files = scanFiles();
    for (File file : files) {
      StringBuilder sb = new StringBuilder();
      try {
        LineIterator lines = FileUtils.lineIterator(file, sourceEncoding);
        while (lines.hasNext()) {
          String line = lines.nextLine();
          sb.append(StringUtils.trim(line));
          sb.append(IOUtils.LINE_SEPARATOR);
        }
        FileUtils.writeStringToFile(file, sb.toString(), sourceEncoding);

      } catch (IOException e) {
        throw new MojoExecutionException("Can not trim the file " + file, e);
      }
    }
    getLog().info("Trimmed files: " + files.length);
  }

  /**
   * gets a list of all files in the source directory.
   *
   * @return the list of all files in the source directory;
   */
  private File[] scanFiles() {
    String[] defaultIncludes = {"**\\*"};
    DirectoryScanner ds = new DirectoryScanner();
    if (includes == null) {
      ds.setIncludes(defaultIncludes);
    } else {
      ds.setIncludes(includes);
    }
    ds.addDefaultExcludes(); // .svn, ...
    if (excludes != null) {
      ds.setExcludes(excludes);
    }
    ds.setBasedir(directory);
    getLog().info("Scanning directory " + directory);
    ds.scan();
    int maxFiles = ds.getIncludedFiles().length;
    File[] result = new File[maxFiles];
    for (int i = 0; i < maxFiles; i++) {
      result[i] = new File(directory, ds.getIncludedFiles()[i]);
    }
    return result;
  }

  public File getDirectory() {
    return directory;
  }

  public void setDirectory(File directory) {
    this.directory = directory;
  }

  public String[] getIncludes() {
    return includes;
  }

  public void setIncludes(String[] includes) {
    this.includes = includes;
  }

  public String[] getExcludes() {
    return excludes;
  }

  public void setExcludes(String[] excludes) {
    this.excludes = excludes;
  }

  public String getSourceEncoding() {
    return sourceEncoding;
  }

  public void setSourceEncoding(String sourceEncoding) {
    this.sourceEncoding = sourceEncoding;
  }
}
