/*
 * Sonar, entreprise quality control tool.
 * Copyright (C) 2007-2008 Hortis-GRC SA
 * mailto:be_agile HAT hortis DOT ch
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.plugins.api.web.gwt.client;

import com.google.gwt.core.client.EntryPoint;
import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.http.client.URL;
import com.google.gwt.i18n.client.Dictionary;
import com.google.gwt.json.client.JSONArray;
import com.google.gwt.json.client.JSONObject;
import com.google.gwt.json.client.JSONString;
import com.google.gwt.json.client.JSONValue;
import com.google.gwt.user.client.ui.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public abstract class AbstractSourceViewer implements EntryPoint {

  public static final String ERROR_STYLE = "sources-error";
  public static final String WARNING_STYLE = "sources-warning";
  public static final String INFO_STYLE = "sources-info";
  public static final String HTML_ROOT_ID = "sourceviewers";

  private Panel container = null;
  private Grid grid = null;

  private boolean linesRetrieved;
  private boolean decoratorsRetrieved;
  protected boolean expandLinkActivation = true;
  protected String loadedSource = null;
  private List<String> lines;
  private Map<Integer, LineDecorator> decorators;

  private ResourceDictionary resourceDictionary;

  protected abstract void exportJavascript();

  protected abstract String getGwtId();

  protected abstract boolean showValueColumn();

  public void onModuleLoad() {
    this.resourceDictionary = new ResourceDictionary();
    exportJavascript();
  }

  protected Widget loadContainer() {
    if (container == null) {
      RootPanel rootPanel = RootPanel.get(HTML_ROOT_ID);
      if (rootPanel != null) {
        container = new FlowPanel();
        container.setStyleName("sourceviewer");

        final TabBar tabs = new TabBar();
        tabs.setWidth("100%");
        container.add(tabs);

        final Dictionary tabsDictionnary = resourceDictionary.getSourceViewers();
        int tabIndex = 0;
        for (String tabTitle : tabsDictionnary.keySet()) {
          tabs.addTab(tabTitle);
          if (tabsDictionnary.get(tabTitle).equals(getGwtId())) {
            tabs.selectTab(tabIndex);
          }
          tabIndex++;
        }
        tabs.addTabListener(new TabListener() {
          public boolean onBeforeTabSelected(SourcesTabEvents sourcesTabEvents, int i) {
            String gwtId = (String) tabsDictionnary.values().toArray()[i];
            refreshSourceViewer(gwtId);
            return false;
          }

          public void onTabSelected(SourcesTabEvents sourcesTabEvents, int i) {

          }
        });

        grid = new Grid();
        grid.setStyleName("sources");
        container.add(grid);

        rootPanel.add(container);
      }
    }
    return container;
  }

  private native void refreshSourceViewer(String pluginKey) /*-{
    $wnd.config['sourceviewer_plugin_key'] = pluginKey;
    $wnd.callGwtViewer('org.sonar.plugins.core.resourceviewer.GwtResourceViewer');
  }-*/;

  protected void refreshLines() {
    if (!linesRetrieved || !decoratorsRetrieved) {
      return;
    }

    grid.resize(lines.size() + decorators.size(), 3);

    int row = 0;
    int lineId = 1;
    for (String line : lines) {
      final LineDecorator decorator = decorators.get(lineId);

      // line id
      grid.setText(row, 0, String.valueOf(lineId));
      grid.getCellFormatter().setStyleName(row, 0, "ln");

      // value column
      int sourceColumnIndex = 1;
      if (showValueColumn()) {
        grid.getCellFormatter().setStyleName(row, 1, "val");
        sourceColumnIndex = 2;

        if (decorator != null) {
          grid.setText(row, 1, decorator.getValue());
          if (decorator.getValueStyle() != null) {
            grid.getCellFormatter().addStyleName(row, 1, decorator.getValueStyle());
          }
        } else {
          grid.setText(row, 1, null);
        }
      }

      // source column
      grid.getCellFormatter().setStyleName(row, sourceColumnIndex, "src");
      grid.setText(row, sourceColumnIndex, line);
      if (decorator != null && decorator.getSourceStyle() != null) {
        grid.getCellFormatter().addStyleName(row, sourceColumnIndex, decorator.getSourceStyle());
      }

      // message
      if (decorator != null && decorator.getMessage() != null) {
        row++;
        grid.getCellFormatter().setStyleName(row, 0, "ln");
        grid.getCellFormatter().setStyleName(row, 1, "val");
        grid.setWidget(row, sourceColumnIndex, decorator.getMessage());
      }

      row++;
      lineId++;
    }
    grid.setVisible(true);
    container.setVisible(true);
  }

  protected void loadSource() {
    loadContainer();
    setDecoratorsRetrieved(false);
    setLinesRetrieved(false);
    grid.setVisible(false);
    String sourceKey = resourceDictionary.getSourceKey();
    if (sourceKey != null) {

      // first time or source has changed
      if (loadedSource == null || !sourceKey.equals(loadedSource)) {
        grid.clear();
        grid.resize(0,0);

        requestLines(sourceKey);
        requestLineDecorators(sourceKey);
        loadedSource = sourceKey;

        // source has not changed
      } else {
        grid.setVisible(true);
        container.setVisible(true);
      }
    }
  }

  protected void requestLines(String sourceKey) {
    String url = URL.encode(Utils.getServerUrl() + "/api/projects/" + sourceKey + "/sources?format=json&callback=");
    JsonUtils.requestJson(url, new LinesHandler());
  }

  protected abstract void requestLineDecorators(String sourceKey);

  public class LinesHandler implements JsonUtils.JSONHandler {

    public void handleJSON(JavaScriptObject obj) {
      if (obj == null) {
        Utils.showError("no JSON data");
        return;
      }
      ArrayList<String> lines = new ArrayList<String>();
      JSONArray jsonArray = new JSONArray(obj);
      for (int i = 0; i < jsonArray.size(); i++) {
        JSONObject jsStock;
        JSONValue jsonValue;
        if ((jsStock = jsonArray.get(i).isObject()) == null) continue;
        if ((jsonValue = jsStock.get("lines")) != null) {
          JSONArray measures = jsonValue.isArray();
          for (int j = 0; j < measures.size(); j++) {
            JSONString value = null;
            if ((jsStock = measures.get(j).isObject()) == null) continue;
            if ((jsonValue = jsStock.get("value")) != null) {
              value = jsonValue.isString();
            }
            lines.add(value != null ? value.stringValue() : "");
          }
        }
      }
      setLines(lines);
      setLinesRetrieved(true);
      refreshLines();
    }
  }

  protected boolean isLinesRetrieved() {
    return linesRetrieved;
  }

  protected void setLinesRetrieved(boolean linesRetrieved) {
    this.linesRetrieved = linesRetrieved;
  }

  protected boolean isDecoratorsRetrieved() {
    return decoratorsRetrieved;
  }

  protected void setDecoratorsRetrieved(boolean decoratorsRetrieved) {
    this.decoratorsRetrieved = decoratorsRetrieved;
  }

  protected void setLines(List<String> lines) {
    this.lines = lines;
  }

  protected void setExpandLinkActivation(boolean expandLinkActivation) {
    this.expandLinkActivation = expandLinkActivation;
  }

  public Map<Integer, LineDecorator> getDecorators() {
    return decorators;
  }

  public void setDecorators(Map<Integer, LineDecorator> decorators) {
    this.decorators = decorators;
  }
}
