/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.api.batch;

import org.sonar.api.measures.FormulaData;
import org.sonar.api.measures.Measure;
import org.sonar.api.measures.Metric;
import org.sonar.api.resources.Project;
import org.sonar.api.resources.Resource;

import java.util.List;

/**
 * A pre-implementation of a decorator using a simple calculation formula
 * @since 1.11
 */
public class FormulaDecorator implements Decorator {

  private Metric metric;
  private DefaultFormulaContext formulaContext;

  /**
   * Creates a FormulaDecorator
   *
   * @param metric the metric should have an associated formula
   * 
   * @throws IllegalArgumentException if no formula is associated to the metric
   */
  public FormulaDecorator(Metric metric) {
    if (metric.getFormula() == null) {
      throw new IllegalArgumentException("No formula defined on metric");
    }
    this.metric = metric;
    this.formulaContext = new DefaultFormulaContext(metric);
  }

  /**
   * {@inheritDoc}
   */
  public boolean shouldExecuteOnProject(Project project) {
    return true;
  }

  /**
   * @return metric generated by the decorator
   */
  @DependedUpon
  public Metric generatesMetric() {
    return metric;
  }

  /**
   * @return metric the decorator depends upon
   */
  @DependsUpon
  public List<Metric> dependsUponMetrics() {
    return metric.getFormula().dependsUponMetrics();
  }

  /**
   * {@inheritDoc}
   */
  public void decorate(Resource resource, DecoratorContext context) {
    if (context.getMeasure(metric) != null) {
      return;
    }

    formulaContext.setDecoratorContext(context);
    FormulaData data = new DefaultFormulaData(context);
    Measure measure = metric.getFormula().calculate(data, formulaContext);
    if (measure != null) {
      context.saveMeasure(measure);
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    FormulaDecorator that = (FormulaDecorator) o;

    if (metric != null ? !metric.equals(that.metric) : that.metric != null) {
      return false;
    }
    return true;
  }

  @Override
  public int hashCode() {
    return metric != null ? metric.hashCode() : 0;
  }

  @Override
  public String toString() {
    return new StringBuilder().append("f(").append(metric).append(")").toString();
  }
}
