/*
 * Decompiled with CFR 0.152.
 */
package org.codelibs.elasticsearch.auth.service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import org.apache.commons.codec.digest.DigestUtils;
import org.codelibs.elasticsearch.auth.AuthException;
import org.codelibs.elasticsearch.auth.filter.ContentFilter;
import org.codelibs.elasticsearch.auth.filter.LoginFilter;
import org.codelibs.elasticsearch.auth.filter.LogoutFilter;
import org.codelibs.elasticsearch.auth.security.Authenticator;
import org.codelibs.elasticsearch.auth.security.LoginConstraint;
import org.codelibs.elasticsearch.auth.util.MapUtil;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.health.ClusterHealthResponse;
import org.elasticsearch.action.admin.cluster.health.ClusterHealthStatus;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsResponse;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.netty.handler.codec.http.Cookie;
import org.elasticsearch.common.netty.handler.codec.http.CookieDecoder;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.rest.RestController;
import org.elasticsearch.rest.RestFilter;
import org.elasticsearch.rest.RestRequest;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;

public class AuthService
extends AbstractLifecycleComponent<AuthService> {
    private static final String DEFAULT_CONSTRAINT_TYPE = "constraint";
    private static final String DEFAULT_CONSTRAINT_INDEX_NAME = "security";
    private static final String DEFAULT_COOKIE_TOKEN_NAME = "eaid";
    private static final String DEFAULT_GUEST_ROLE = "guest";
    private RestController restController;
    private Map<String, Authenticator> authenticatorMap = new LinkedHashMap<String, Authenticator>();
    private Client client;
    private String constraintIndex;
    private String constraintType;
    private String authTokenIndex = "auth";
    private String tokenType = "token";
    private String tokenKey = "token";
    private String guestRole;
    private ContentFilter contentFilter;
    private boolean cookieToken = true;
    private String cookieTokenName;
    private boolean updateToken;

    @Inject
    public AuthService(Settings settings, Client client, RestController restController) {
        super(settings);
        this.client = client;
        this.restController = restController;
        this.logger.info("Creating authenticators.", new Object[0]);
        this.constraintIndex = settings.get("auth.constraint.index", DEFAULT_CONSTRAINT_INDEX_NAME);
        this.constraintType = settings.get("auth.constraint.type", DEFAULT_CONSTRAINT_TYPE);
        this.cookieTokenName = settings.get("auth.token.cookie", DEFAULT_COOKIE_TOKEN_NAME);
        this.updateToken = settings.getAsBoolean("auth.token.update_by_request", Boolean.valueOf(true));
        this.guestRole = settings.get("auth.role.guest", DEFAULT_GUEST_ROLE);
        if (this.cookieTokenName.trim().length() == 0 || "false".equalsIgnoreCase(this.cookieTokenName)) {
            this.cookieToken = false;
        }
    }

    protected void doStart() throws ElasticsearchException {
        String[] logoutMethodValues;
        String[] loginMethodValues;
        this.logger.info("Starting AuthService.", new Object[0]);
        LoginFilter loginFilter = new LoginFilter(this, this.authenticatorMap);
        String loginPath = this.settings.get("auth.login.path");
        if (loginPath != null) {
            loginFilter.setLoginPath(loginPath);
        }
        if ((loginMethodValues = this.settings.getAsArray("auth.login.methods")) != null && loginMethodValues.length > 0) {
            loginFilter.setHttpMethods(this.createMethods(loginMethodValues));
        }
        this.restController.registerFilter((RestFilter)loginFilter);
        LogoutFilter logoutFilter = new LogoutFilter(this);
        String logoutPath = this.settings.get("auth.logout.path");
        if (logoutPath != null) {
            logoutFilter.setLogoutPath(logoutPath);
        }
        if ((logoutMethodValues = this.settings.getAsArray("auth.logout.methods")) != null && logoutMethodValues.length > 0) {
            logoutFilter.setHttpMethods(this.createMethods(logoutMethodValues));
        }
        this.restController.registerFilter((RestFilter)logoutFilter);
        this.contentFilter = new ContentFilter(this);
        this.restController.registerFilter((RestFilter)this.contentFilter);
    }

    protected void doStop() throws ElasticsearchException {
        this.logger.info("Stopping AuthService", new Object[0]);
    }

    protected void doClose() throws ElasticsearchException {
        this.logger.info("Closing AuthService.", new Object[0]);
    }

    public void registerAuthenticator(String name, Authenticator authenticator) {
        this.authenticatorMap.put(name, authenticator);
    }

    public void init(final ActionListener<Void> listener) {
        this.client.admin().cluster().prepareHealth(new String[0]).setWaitForYellowStatus().execute((ActionListener)new ActionListener<ClusterHealthResponse>(){

            public void onResponse(ClusterHealthResponse response) {
                if (response.getStatus() == ClusterHealthStatus.RED) {
                    listener.onFailure((Throwable)new AuthException(RestStatus.SERVICE_UNAVAILABLE, "This cluster is not ready."));
                } else {
                    AuthService.this.createConstraintIndexIfNotExist((ActionListener<Void>)listener);
                }
            }

            public void onFailure(Throwable e) {
                listener.onFailure(e);
            }
        });
    }

    protected void createConstraintIndexIfNotExist(final ActionListener<Void> listener) {
        this.client.admin().indices().prepareExists(new String[]{this.constraintIndex}).execute((ActionListener)new ActionListener<IndicesExistsResponse>(){

            public void onResponse(IndicesExistsResponse response) {
                if (response.isExists()) {
                    AuthService.this.reload((ActionListener<Void>)listener);
                } else {
                    AuthService.this.createConstraintIndex((ActionListener<Void>)listener);
                }
            }

            public void onFailure(Throwable e) {
                listener.onFailure(e);
            }
        });
    }

    protected void createConstraintIndex(final ActionListener<Void> listener) {
        this.client.admin().indices().prepareCreate(this.constraintIndex).execute((ActionListener)new ActionListener<CreateIndexResponse>(){

            public void onResponse(CreateIndexResponse response) {
                AuthService.this.reload((ActionListener<Void>)listener);
            }

            public void onFailure(Throwable e) {
                listener.onFailure(e);
            }
        });
    }

    public void reload(final ActionListener<Void> listener) {
        this.client.admin().indices().prepareRefresh(new String[]{this.constraintIndex}).setForce(true).execute((ActionListener)new ActionListener<RefreshResponse>(){

            public void onResponse(RefreshResponse response) {
                AuthService.this.loadLoginConstraints(new ActionListener<LoginConstraint[]>(){

                    public void onResponse(LoginConstraint[] constraints) {
                        if (AuthService.this.logger.isDebugEnabled()) {
                            AuthService.this.logger.debug("Load {} constraint(s).", new Object[]{constraints.length});
                        }
                        AuthService.this.contentFilter.setLoginConstraints(constraints);
                        listener.onResponse(null);
                    }

                    public void onFailure(Throwable e) {
                        listener.onFailure(e);
                    }
                });
            }

            public void onFailure(Throwable e) {
                listener.onFailure(e);
            }
        });
    }

    public void createToken(Set<String> roleSet, final ActionListener<String> listener) {
        if (roleSet == null || roleSet.isEmpty()) {
            listener.onFailure((Throwable)new AuthException(RestStatus.BAD_REQUEST, "Role is empty."));
            return;
        }
        final String token = this.generateToken();
        HashMap<String, Object> sourceMap = new HashMap<String, Object>();
        sourceMap.put("roles", roleSet);
        sourceMap.put("lastModified", new Date());
        this.client.prepareIndex(this.authTokenIndex, this.tokenType, token).setSource(sourceMap).setRefresh(true).execute((ActionListener)new ActionListener<IndexResponse>(){

            public void onResponse(IndexResponse response) {
                listener.onResponse((Object)token);
            }

            public void onFailure(Throwable e) {
                listener.onFailure(e);
            }
        });
    }

    public void authenticate(final String token, final String[] roles, final ActionListener<Boolean> listener) {
        if (token == null) {
            if (roles != null) {
                for (String role : roles) {
                    if (!this.guestRole.equals(role)) continue;
                    listener.onResponse((Object)true);
                    return;
                }
            }
            listener.onResponse((Object)false);
        } else {
            this.client.prepareGet(this.authTokenIndex, this.tokenType, token).execute((ActionListener)new ActionListener<GetResponse>(){

                public void onResponse(GetResponse response) {
                    Map sourceMap = response.getSource();
                    if (sourceMap != null) {
                        String[] tokenRoles = MapUtil.getAsArray(sourceMap, "roles", new String[0]);
                        for (String role : roles) {
                            for (String tokenRole : tokenRoles) {
                                if (!role.equals(tokenRole)) continue;
                                listener.onResponse((Object)true);
                                if (AuthService.this.updateToken) {
                                    AuthService.this.updateToken(token, sourceMap);
                                }
                                return;
                            }
                        }
                    }
                    listener.onResponse((Object)false);
                }

                public void onFailure(Throwable e) {
                    listener.onFailure(e);
                }
            });
        }
    }

    private void updateToken(final String token, Map<String, Object> sourceMap) {
        sourceMap.put("lastModified", new Date());
        this.client.prepareIndex(this.authTokenIndex, this.tokenType, token).setSource(sourceMap).execute((ActionListener)new ActionListener<IndexResponse>(){

            public void onResponse(IndexResponse response) {
            }

            public void onFailure(Throwable e) {
                AuthService.this.logger.warn("Failed to update token: " + token, e, new Object[0]);
            }
        });
    }

    public void createUser(String authenticatorName, String username, String password, String[] roles, ActionListener<Void> listener) {
        if (authenticatorName == null || username == null || password == null || roles == null) {
            listener.onFailure((Throwable)new AuthException(RestStatus.BAD_REQUEST, "authenticator, username, passowrd or roles is null."));
        } else {
            this.getAuthenticator(authenticatorName).createUser(username, password, roles, listener);
        }
    }

    public void updateUser(String authenticatorName, String username, String password, String[] roles, ActionListener<Void> listener) {
        if (authenticatorName == null || username == null || password == null && roles == null) {
            listener.onFailure((Throwable)new AuthException(RestStatus.BAD_REQUEST, "authenticator, username, passowrd or roles are null."));
        } else {
            this.getAuthenticator(authenticatorName).updateUser(username, password, roles, listener);
        }
    }

    public void deleteUser(String authenticatorName, String username, ActionListener<Void> listener) {
        if (authenticatorName == null || username == null) {
            listener.onFailure((Throwable)new AuthException(RestStatus.BAD_REQUEST, "authenticator or username are null."));
        } else {
            this.getAuthenticator(authenticatorName).deleteUser(username, listener);
        }
    }

    public void deleteToken(String token, final ActionListener<Void> listener) {
        this.client.prepareDelete(this.authTokenIndex, this.tokenType, token).setRefresh(true).execute((ActionListener)new ActionListener<DeleteResponse>(){

            public void onResponse(DeleteResponse response) {
                if (response.isFound()) {
                    listener.onResponse(null);
                } else {
                    listener.onFailure((Throwable)new AuthException(RestStatus.BAD_REQUEST, "The token does not exist."));
                }
            }

            public void onFailure(Throwable e) {
                listener.onFailure(e);
            }
        });
    }

    public String getToken(RestRequest request) {
        String cookieString;
        String token = request.param(this.tokenKey);
        if (token == null && this.cookieToken && (cookieString = request.header("Cookie")) != null) {
            CookieDecoder cookieDecoder = new CookieDecoder();
            Set cookies = cookieDecoder.decode(cookieString);
            for (Cookie cookie : cookies) {
                if (!this.cookieTokenName.equals(cookie.getName())) continue;
                token = cookie.getValue();
                break;
            }
        }
        return token;
    }

    private String generateToken() {
        return DigestUtils.sha512Hex((String)UUID.randomUUID().toString());
    }

    private Authenticator getAuthenticator(String authenticatorName) {
        Authenticator authenticator = this.authenticatorMap.get(authenticatorName);
        if (authenticator == null) {
            throw new AuthException(RestStatus.BAD_REQUEST, "Unknown authenticator: " + authenticatorName);
        }
        return authenticator;
    }

    protected void loadLoginConstraints(final ActionListener<LoginConstraint[]> listener) {
        this.client.prepareSearch(new String[]{this.constraintIndex}).setTypes(new String[]{this.constraintType}).setQuery((QueryBuilder)QueryBuilders.queryString((String)"*:*")).execute((ActionListener)new ActionListener<SearchResponse>(){

            public void onResponse(SearchResponse response) {
                TreeMap<String, LoginConstraint> constraintMap = new TreeMap<String, LoginConstraint>(new Comparator<String>(){

                    @Override
                    public int compare(String path1, String path2) {
                        int length2;
                        int length1 = path1.length();
                        if (length1 == (length2 = path2.length())) {
                            return -1 * path1.compareTo(path2);
                        }
                        return length1 < length2 ? -1 : 1;
                    }
                });
                SearchHits hits = response.getHits();
                if (hits.totalHits() != 0L) {
                    for (SearchHit hit : hits) {
                        Map sourceMap = hit.sourceAsMap();
                        List<String> methodList = MapUtil.getAsList(sourceMap, "methods", Collections.emptyList());
                        List<String> pathList = MapUtil.getAsList(sourceMap, "paths", Collections.emptyList());
                        List<String> roleList = MapUtil.getAsList(sourceMap, "roles", Collections.emptyList());
                        if (!pathList.isEmpty() && !roleList.isEmpty()) {
                            for (String path : pathList) {
                                LoginConstraint constraint = (LoginConstraint)constraintMap.get(path);
                                if (constraint == null) {
                                    constraint = new LoginConstraint();
                                    constraint.setPath(path);
                                    constraintMap.put(path, constraint);
                                }
                                constraint.addCondition(methodList.toArray(new String[methodList.size()]), roleList.toArray(new String[roleList.size()]));
                            }
                            continue;
                        }
                        AuthService.this.logger.warn("Invaid login settings: " + sourceMap, new Object[0]);
                    }
                }
                listener.onResponse((Object)constraintMap.values().toArray(new LoginConstraint[constraintMap.size()]));
            }

            public void onFailure(Throwable e) {
                listener.onFailure((Throwable)new AuthException(RestStatus.INTERNAL_SERVER_ERROR, AuthService.this.constraintIndex + ":" + AuthService.this.constraintType + " is not found.", e));
            }
        });
    }

    private RestRequest.Method[] createMethods(String[] methodValues) {
        ArrayList<RestRequest.Method> methodList = new ArrayList<RestRequest.Method>();
        for (String method : methodValues) {
            if ("get".equalsIgnoreCase(method)) {
                methodList.add(RestRequest.Method.GET);
                continue;
            }
            if ("post".equalsIgnoreCase(method)) {
                methodList.add(RestRequest.Method.POST);
                continue;
            }
            if ("head".equalsIgnoreCase(method)) {
                methodList.add(RestRequest.Method.HEAD);
                continue;
            }
            if ("options".equalsIgnoreCase(method)) {
                methodList.add(RestRequest.Method.OPTIONS);
                continue;
            }
            if ("put".equalsIgnoreCase(method)) {
                methodList.add(RestRequest.Method.PUT);
                continue;
            }
            if (!"delete".equalsIgnoreCase(method)) continue;
            methodList.add(RestRequest.Method.DELETE);
        }
        return methodList.toArray(new RestRequest.Method[methodList.size()]);
    }
}

