/*
 * Copyright (C) 2012 Red Hat, Inc. (jcasey@redhat.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.commonjava.maven.ext.manip;

import java.util.AbstractList;
import java.util.List;

/**
 * From Apache Commons Collections. Avoids a dependency (which means shading it in) on the entire collections for a single
 * method.
 *
 * https://github.com/apache/commons-collections/blob/trunk/src/main/java/org/apache/commons/collections4/ListUtils.java#L651
 *
 * This was originally based upon http://www.vogella.com/tutorials/JavaAlgorithmsPartitionCollection/article.html and
 * background here http://www.baeldung.com/java-list-split
 */
public final class ListUtils
{
    /**
     * Returns consecutive {@link List#subList(int, int) sublists} of a
     * list, each of the same size (the final list may be smaller). For example,
     * partitioning a list containing {@code [a, b, c, d, e]} with a partition
     * size of 3 yields {@code [[a, b, c], [d, e]]} -- an outer list containing
     * two inner lists of three and two elements, all in the original order.
     * <p>
     * The outer list is unmodifiable, but reflects the latest state of the
     * source list. The inner lists are sublist views of the original list,
     * produced on demand using {@link List#subList(int, int)}, and are subject
     * to all the usual caveats about modification as explained in that API.
     * <p>
     * Adapted from http://code.google.com/p/guava-libraries/
     *
     * @param <T> the element type
     * @param list  the list to return consecutive sublists of
     * @param size  the desired size of each sublist (the last may be smaller)
     * @return a list of consecutive sublists
     * @throws NullPointerException if list is null
     * @throws IllegalArgumentException if size is not strictly positive
     */
    public static <T> List<List<T>> partition( final List<T> list, final int size )
    {
        if ( list == null )
        {
            throw new NullPointerException( "List must not be null" );
        }
        if ( size <= 0 )
        {
            throw new IllegalArgumentException( "Size must be greater than 0" );
        }
        return new Partition<>( list, size );
    }

    /**
     * Provides a partition view on a {@link List}.
     */
    private static class Partition<T> extends AbstractList<List<T>>
    {
        private final List<T> list;

        private final int size;

        private Partition( final List<T> list, final int size )
        {
            this.list = list;
            this.size = size;
        }

        @Override
        public List<T> get( final int index )
        {
            final int listSize = size();
            if ( listSize < 0 )
            {
                throw new IllegalArgumentException( "negative size: " + listSize );
            }
            if ( index < 0 )
            {
                throw new IndexOutOfBoundsException( "Index " + index + " must not be negative" );
            }
            if ( index >= listSize )
            {
                throw new IndexOutOfBoundsException( "Index " + index + " must be less than size " +
                                                                     listSize );
            }
            final int start = index * size;
            final int end = Math.min( start + size, list.size() );
            return list.subList( start, end );
        }

        @Override
        public int size()
        {
            return ( list.size() + size - 1 ) / size;
        }

        @Override
        public boolean isEmpty()
        {
            return list.isEmpty();
        }
    }
}
