/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.actorframe;

import org.coos.actorframe.messages.AFConstants;
import org.coos.javaframe.*;

import org.coos.javaframe.CompositeState;
import org.coos.javaframe.messages.AFPropertyMsg;
import org.coos.javaframe.messages.ActorMsg;
import org.coos.javaframe.messages.JFConstants;

import java.util.Enumeration;
import java.util.Vector;

/**
 * The behavior for for an actor. It mainly consists of the ActorFrame protocol.
 * 
 * @author Geir Melby, Tellu AS
 * @author Knut Eilif Husa, Tellu AS
 */
public class SuspendedCS extends CompositeState implements AFConstants {

	/**
	 * The constructor
	 * 
	 * @param sn
	 *            The name of the state
	 * @param cs
	 *            The enclosing state
	 */
	public SuspendedCS(String sn, CompositeState cs) {
		super(sn, cs);
	}

	public SuspendedCS() {
		super("");
	}

	public SuspendedCS(String sn) {
		super(sn);
	}

	public void enterState(StateMachine curfsm) {
		if (curfsm instanceof ActorSM) {
			ActorSM asm = (ActorSM) curfsm;
			asm.suspendedActors = curfsm.context.getChildrenRoles();
			if (asm.suspendedActors.isEmpty()) {
				curfsm.sendMessage(SUSPENDED_MSG, asm.suspendActor);
				asm.suspendedActors = null;
				suspended.enterState(curfsm);
				entry(curfsm);
				return;
			} else {
				curfsm.sendMessage(new AFPropertyMsg(SUSPEND_MSG, true), asm.suspendedActors);
				suspending.enterState(curfsm);
				entry(curfsm);
				return;
			}
		} else {
			exitState(2, curfsm);
		}
	}

	// protected State waitConfirmPorts = new State("waitConfirmPorts", this);
	protected State waitConfirmUpdatePorts = new State("waitConfirmUpdatePorts", this);
	protected State waitConfirmUpdateParts = new State("waitConfirmUpdateParts", this);
	protected State waitConfirmCreateParts = new State("waitConfirmCreateParts", this);
	protected State waitConfirmDeleteConnectors = new State("waitConfirmDeleteConnectors", this);
	protected State waitConfirmDeletePorts = new State("waitConfirmDeletePorts", this);
	protected State waitConfirmDeleteParts = new State("waitConfirmDeleteParts", this);
	protected State waitAddedConnectors = new State("waitAddedConnectors", this);
	protected State suspended = new State("suspended", this);
	protected State suspending = new State("suspending", this);

	/**
	 * Execute transition (if defined) for ActorMsg sig and State st. For each
	 * CompositeState class execTrans has to be defined. It defines the
	 * transitions (and saves) handled by its inner States (not included those
	 * inside inner CompositeStates). NB: execTrans is not called from user
	 * code.
	 * 
	 * @param sig
	 *            The consumed ActorMsg.
	 * @param st
	 *            The State to search for defined transition.
	 * @param curfsm
	 *            The performing StateMachine.
	 */
	public void execTrans(ActorMsg sig, State st, StateMachine curfsm) {

		ActorSM asm = (ActorSM) curfsm;

		if (st == waitConfirmDeleteConnectors) {
			// wait for deleting connectors internal among internal parts and
			// eventually between ports inside to my ports
			if (sig.equals(CHECK_SPEC_MSG)) {
				performExit(curfsm);
				// read changes in connectors specs

				Vector removedConnectors = compareSpecs(asm.actorSpec.getConnectorDesc(), asm.newActorSpec
						.getConnectorDesc(), false);
				removedConnectors = asm.rewriteContextualConnectors(removedConnectors);

				// Remove connectors connecting port between ports belong to my
				// parts and me.-
				// Send removeConnector to "To" Port in the connector
				// description to parts belong to me
				for (int i = 0; i < removedConnectors.size(); i++) {
					ConnectorSpec connectorSpec = (ConnectorSpec) removedConnectors.elementAt(i);
					AFPropertyMsg msg = new AFPropertyMsg(CONNECTOR_REMOVE_MSG, true);
					msg.setFrameworkMsg(true);
					msg.setProperty(CONNECTOR_REMOVE_MSG_CONNECTORS, connectorSpec);
					asm.sendMessage(msg, connectorSpec.getTo());
				}
				asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
				nextState(waitConfirmDeletePorts, curfsm);
				return;
			}
		} else if (st == waitConfirmDeletePorts) {
			// wait for deleteing parts,
			if (sig.equals(CHECK_SPEC_MSG)) {
				performExit(curfsm);
				// Remove all or some of my ports connecting me to the outside
				// world and inside parts
				// Send PortRemove to the ports, Assume that the enclosing
				// actors has removed the connectors
				Vector removedPorts = compareSpecs(asm.actorSpec.getPortDesc(), asm.newActorSpec.getPortDesc(), false);
				Enumeration ports = removedPorts.elements();
				while (ports.hasMoreElements()) {
					Port port = (Port) ports.nextElement();
					port.removePort(asm);
					asm.removePort(port);
				}
				asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
				nextState(waitConfirmDeleteParts, curfsm);
				return;

			}
		} else if (st == waitConfirmDeleteParts) {
			// wait for deleteing parts,
			if (sig.equals(CHECK_SPEC_MSG)) {
				Vector removed = compareSpecs(asm.actorSpec.getPartDesc(), asm.newActorSpec.getPartDesc(), false);

				if (!removed.isEmpty()) {
					for (int i = 0; i < removed.size(); i++) {
						PartSpec partSpec = (PartSpec) removed.elementAt(i);
						asm.removedParts = asm.context.getChildrenRoles(partSpec.getRoleType());
						asm.sendMessage(ROLE_REMOVE_MSG, asm.removedParts);
					}
					sameState(curfsm);
					return;

				} else {
					performExit(curfsm);
					asm.removedParts = null;
					asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
					nextState(waitConfirmUpdatePorts, curfsm);
					return;
				}

			} else if (sig.equals(ROLE_PLAY_ENDED_MSG)) {
				for (int i = 0; i < asm.removedParts.size(); i++) {
					ActorAddress address = (ActorAddress) asm.removedParts.elementAt(i);
					if (address.equals(sig.getSenderRole())) {
						asm.removedParts.removeElementAt(i);
						// remove the child from my actor context
						curfsm.context.remove(sig.getSenderRole());
						break;
					}
				}

				if (asm.removedParts.isEmpty()) {
					// all parts are deleted
					performExit(curfsm);
					asm.removedParts = null;
					asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
					nextState(waitConfirmUpdatePorts, curfsm);
					return;
				}
				sameState(curfsm);
				return;
			}
		} else if (st == waitConfirmUpdatePorts) {
			if (sig.equals(CHECK_SPEC_MSG)) {
				Vector addedPorts = compareSpecs(asm.newActorSpec.getPortDesc(), asm.actorSpec.getPortDesc(), false);

				Vector c = (Vector) sig.getProperty(ROLE_UPDATE_MSG_CONNECTORS);
				boolean newPortsCreated = curfsm.createPorts(c); // create the
				// ports as
				// defined.

				if (newPortsCreated) {
					curfsm.context.setTmpPorts(addedPorts, c);
					sameState(curfsm);
					return;
				}
				// no ports created
				// curfsm.context.setTmpPorts(addedPorts, c);
				performExit(curfsm);
				asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
				nextState(waitConfirmUpdateParts, curfsm);
				return;
			} else if (sig.equals(PORT_CREATE_ACK_MSG)) {
				// A port has been created.
				if (asm.context.allPortsAcked(sig.getSenderRole().getActorPort())) {
					performExit(curfsm);
					asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
					nextState(waitConfirmUpdateParts, curfsm);
					return;
				} else {
					// more ports to wait for.
					sameState(curfsm);
					return;
				}

			} else if (sig.equals(PORT_CREATE_NACK_MSG)) {
				// creation of a port failed.
				performExit(curfsm);
				if (asm.scheduler.isTraceOn()) {
					asm.trace.traceError("ActorCS: Port creation failed, creation of ports aborted");
				}

				if (asm.context.getParentAddress().getActorID() != null) {
					asm.sendMessage(new AFPropertyMsg(ROLE_CREATE_NACK_MSG, true), asm.context.getParentAddress());
				}

				asm.stopTimer(StateMachineCS.CREATION_TIMER_ID);
				AFPropertyMsg rpm = (AFPropertyMsg) curfsm.context.getRootPlayMsg();
				if (rpm != null) {
					// root for the RolePlayMsg
					// curfsm.sendMessage(new
					// RolePlayNackMsg(rpm.getRoleRequestMsg()),
					// rpm.getSenderRole());
					AFPropertyMsg msg = new AFPropertyMsg(ROLE_PLAY_NACK_MSG, true);
					msg.setProperty(ROLE_REQUEST_MSG, rpm.getProperty(ROLE_REQUEST_MSG));
					curfsm.sendMessage(msg, rpm.getSenderRole());
				}

				// remove its children and itself
				asm.sendMessage(new AFPropertyMsg(ROLE_REMOVE_MSG, true), asm.context.getActorAddress());

				if (asm.isVisible()) {
					asm.stopTimer(ActorSM.ROUTER_UPDATE_INTERVAL_ID);
				}
				exitState(0, asm);
				// nextState(init, curfsm);

				return;
			} else if (sig.equals(TIMER_MSG)
					&& ((AFPropertyMsg) sig).getProperty(JFConstants.TIMER_ID).equals(StateMachineCS.CREATION_TIMER_ID)) {
				// todo this is incomplete?
				// Resend the port connector requests to inner children
				asm.createConnectors();
				sameState(curfsm);

				return;
			}
		} else if (st == waitConfirmUpdateParts) {
			// send an update msg to all instances, wait ack message
			if (sig.equals(CHECK_SPEC_MSG)) {
				// asm.actorSpec = asm.newActorSpec;
				// remove all instances of that partType inside me
				Vector children = asm.getContext().getChildrenRoles();
				if (!children.isEmpty()) {
					asm.sendMessage(ROLE_UPDATE_MSG, children);
					asm.changedParts = children;
					sameState(curfsm);
					return;
				} else {
					performExit(curfsm);
					asm.changedParts = null;
					asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
					nextState(waitAddedConnectors, curfsm);
					return;

				}

			} else if (sig.equals(ROLE_UPDATE_ACK_MSG)) {
				for (int i = 0; i < asm.changedParts.size(); i++) {
					ActorAddress actorAddress = (ActorAddress) asm.changedParts.elementAt(i);
					if (actorAddress.equals(sig.getSenderRole())) {
						asm.changedParts.removeElementAt(i);
						break;
					}
				}

				if (asm.changedParts.isEmpty()) {
					performExit(curfsm);
					asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
					nextState(waitConfirmCreateParts, curfsm);
					return;
				}
				sameState(curfsm);
				return;

				// } else if (sig instanceof RoleCreateNackMsg) {
			} else if (sig.equals(ROLE_UPDATE_NACK_MSG)) {
				if (asm.scheduler.isTraceOn()) {
					asm.trace.traceError("SuspenedCS: Inner part update failed");
				}

				performExit(curfsm);
				if (asm.context.getParentAddress().getActorID() != null) {
					// asm.sendMessage(new RoleCreateNackMsg(),
					// asm.context.getParentAddress());
					asm.sendMessage(ROLE_CREATE_NACK_MSG, asm.context.getParentAddress());
				}
				asm.stopTimer(StateMachineCS.CREATION_TIMER_ID);
				exitState(0, asm);

				return;
			}
		} else if (st == waitConfirmCreateParts) {
			// new parts are created, wait until all parts are confirmed created
			// before returning
			// back to history state. If errors occur, send remove to self.
			// if (sig instanceof RoleCreateAckMsg) {
			if (sig.equals(CHECK_SPEC_MSG)) {
				// create eventually inner parts, RoleCreateMsg is send to all
				// actors that should be created
				boolean partsCreated = asm.createParts(asm.newActorSpec);
				if (partsCreated) {

					// add all createchildren
					Enumeration enumer = asm.context.getCreationOfChildren().elements();

					while (enumer.hasMoreElements()) {
						asm.context.getTmpPorts().addElement(enumer.nextElement());
					}
					if (asm.scheduler.isTraceOn()) {
						asm.trace.traceTask("New parts defined");
					}
					asm.setNoOfTrialsLeft(StateMachineCS.NO_OF_RETRIALS);
					asm.startTimer(StateMachineCS.CREATION_TIMER_SPEC, StateMachineCS.CREATION_TIMER_ID);
					sameState(curfsm);
					return;
				} else {
					performExit(curfsm);
					asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
					nextState(waitAddedConnectors, curfsm);
					return;

				}
			} else if (sig.equals(ROLE_CREATE_ACK_MSG)) {
				Vector v = asm.context.getCreationOfChildren();
				asm.context.removeCreationChild(sig.getSenderRole());
				sig.getSenderRole().setActorDomain(curfsm.getScheduler().getSchedulerData().getActorDomainName());
				asm.context.addPersistentChildrenRole(sig.getSenderRole());
				// Send suspend to the actor
				asm.sendMessage(SUSPEND_MSG, sig.getSenderRole());
				// add the this actor to the list to be acked
				if (asm.suspendedActors == null) {
					asm.suspendedActors = new Vector();
				}
				asm.suspendedActors.addElement(sig.getSenderRole());
				if (v.isEmpty()) {
					performExit(curfsm);
					asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
					nextState(waitAddedConnectors, curfsm);
					return;
				}
				sameState(curfsm);
				return;

				// } else if (sig instanceof RoleCreateNackMsg) {
			} else if (sig.equals(ROLE_CREATE_NACK_MSG)) {
				if (asm.scheduler.isTraceOn()) {
					asm.trace.traceError("ActorCS: Inner part creation rejected -- creation of inner parts aborted");
				}

				performExit(curfsm);
				if (asm.context.getParentAddress().getActorID() != null) {
					// asm.sendMessage(new RoleCreateNackMsg(),
					// asm.context.getParentAddress());
					asm.sendMessage(ROLE_CREATE_NACK_MSG, asm.context.getParentAddress());
				}

				asm.sendMessage(ROLE_REMOVE_MSG, asm.context.getActorAddress());
				asm.stopTimer(StateMachineCS.CREATION_TIMER_ID);
				exitState(0, asm);

				return;
			} else if (sig.equals(SUSPENDED_MSG)) {

				for (int i = 0; i < asm.suspendedActors.size(); i++) {
					ActorAddress actorAddress = (ActorAddress) asm.suspendedActors.elementAt(i);
					if (actorAddress.equals(sig.getSenderRole())) {
						asm.suspendedActors.removeElementAt(i);
						break;
					}
				}

				if (asm.suspendedActors.isEmpty()) {
					// all new created actors are suspended
					performExit(curfsm);
					nextState(suspended, curfsm);
					return;
				}

				sameState(curfsm);
				return;
			} else if (sig.equals(TIMER_MSG) && sig.getProperty(TIMER_ID).equals(StateMachineCS.CREATION_TIMER_ID)) {
				// creation failed, repeat 3 times
				// First get the not acknowledged children
				Vector v = asm.context.getCreationOfChildren();

				if (!v.isEmpty()) {
					// some children remains that are not acked, try some times
					// before giving up
					if (asm.getNoOfTrialsLeft() > 0) {
						// send new message to the parts again and decrement no
						// of trials
						asm.setNoOfTrialsLeft(asm.getNoOfTrialsLeft() - 1);

						if (asm.createParts(v)) {
							asm.startTimer(new AFPropertyMsg(JFConstants.TIMER_MSG, true),
									StateMachineCS.CREATION_TIMER_SPEC, StateMachineCS.CREATION_TIMER_ID);
						}
					} else {
						// give up tried some times, but respons
						if (asm.scheduler.isTraceOn()) {
							asm.trace.traceError("ActorCS: Creation Timeout, creation of inner parts aborted");
						}
						// if root node, the requestor of the RolePlayMsg should
						// be informed
						AFPropertyMsg pm = new AFPropertyMsg(ROLE_REMOVE_MSG, true);
						asm.sendMessage(pm, asm.context.getActorAddress());
					}
				}

				sameState(asm);

				return;
			} else if (!(sig.equals(ROLE_REMOVE_MSG) || sig.equals(ROLE_RELEASE_MSG) || sig.equals(ROLE_PLAY_ENDED_MSG)
					|| sig.equals(SET_ACTOR_TRACE_MSG) || sig.equals(REPORT_REQUEST_MSG)
					|| sig.equals(REPORT_RESPONS_MSG) || sig.equals(ACTOR_REPORT_TIMER_MSG))) {
				// save the message until other state is reached.
				save(sig, curfsm);
				sameState(curfsm);

				return;
			}
		} else if (st == waitAddedConnectors)

		{
			if (sig.equals(CHECK_SPEC_MSG)) {

				Vector addedConnectors = compareSpecs(asm.newActorSpec.getConnectorDesc(), asm.actorSpec
						.getConnectorDesc(), false);
				addedConnectors = asm.rewriteContextualConnectors(addedConnectors);

				if (addedConnectors.isEmpty()) {
					// no connectors to be created
					performExit(curfsm);
					// Send ack to reconfiguration is done
					asm.sendMessage(ROLE_UPDATE_ACK_MSG, asm.roleUpdateSender);
					asm.setActorSpec(asm.newActorSpec);
					nextState(suspended, curfsm);
					return;
				}
				// create connectors
				for (int i = 0; i < addedConnectors.size(); i++) {
					ConnectorSpec connectorSpec = (ConnectorSpec) addedConnectors.elementAt(i);
					AFPropertyMsg msg = new AFPropertyMsg(CONNECTOR_Add_MSG, true);
					msg.setFrameworkMsg(true);
					msg.setProperty(CONNECTOR_ADD_MSG_CONNECTORS, connectorSpec);
					asm.sendMessage(msg, connectorSpec.getFrom());
				}

				// asm.context.setTmpPorts(asm.addedConnectors,
				// asm.addedConnectors);
				sameState(curfsm);
				return;
			} else if (sig.equals(PORT_CREATE_ACK_MSG)) {
				// A port has been created.
				if (asm.context.allPortsAcked(sig.getSenderRole().getActorPort())) {
					performExit(curfsm);
					// Send ack to reconfiguration is done
					asm.sendMessage(ROLE_UPDATE_ACK_MSG, asm.roleUpdateSender);
					asm.setActorSpec(asm.newActorSpec);
					nextState(suspended, curfsm);
					return;
				} else {
					// more ports to wait for.
					sameState(curfsm);
					return;
				}

			} else if (sig.equals(PORT_CREATE_NACK_MSG)) {
				// creation of a port failed.
				performExit(curfsm);
				if (asm.scheduler.isTraceOn()) {
					asm.trace.traceError("ActorCS: Port creation failed, creation of ports aborted");
				}
				exitState(0, asm);
				return;
			}

		} else if (st == suspending)

		{
			if (sig.equals(SUSPENDED_MSG)) {
				for (int i = 0; i < asm.suspendedActors.size(); i++) {
					ActorAddress address = (ActorAddress) asm.suspendedActors.elementAt(i);
					if (address.equals(sig.getSenderRole())) {
						asm.suspendedActors.removeElementAt(i);
						break;
					}
				}

				// check if all children has suspendedActors
				if (asm.suspendedActors.isEmpty()) {
					performExit(curfsm);
					curfsm.sendMessage(SUSPENDED_MSG, asm.suspendActor);
					asm.suspendedActors = null;
					nextState(suspended, curfsm);
					return;
				}
				sameState(curfsm);
				return;

			} else {
				save(sig, curfsm);
				return;
			}

		} else if (st == suspended)

		{
			if (sig.equals(RESUME_MSG)) {
				asm.resumeActor = sig.getSenderRole();
				asm.resumedActors = curfsm.context.getChildrenRoles();
				if (asm.resumedActors.isEmpty()) {
					performExit(curfsm);
					curfsm.sendMessage(RESUMED_MSG, asm.resumeActor);
					// asm.sendMessage(ROLE_UPDATE_MSG,
					// asm.getMyActorAddress());
					// sameState(curfsm);
					exitState(1, asm);
					return;
					// nextState(curfsm.getHistoryState(this), curfsm);
					// return;
				} else {
					// send to all children
					curfsm.sendMessage(new AFPropertyMsg(RESUME_MSG, true), asm.resumedActors);
				}

				sameState(curfsm);
				return;

			} else if (sig.equals(RESUMED_MSG)) {
				for (int i = 0; i < asm.resumedActors.size(); i++) {
					ActorAddress address = (ActorAddress) asm.resumedActors.elementAt(i);
					if (address.equals(sig.getSenderRole())) {
						asm.resumedActors.removeElementAt(i);
						break;
					}
				}

				// check if all children have been resumed
				if (asm.resumedActors.isEmpty()) {
					performExit(curfsm);
					curfsm.sendMessage(RESUMED_MSG, asm.resumeActor);
					// asm.sendMessage(ROLE_UPDATE_MSG,
					// asm.getMyActorAddress());
					asm.resumedActors = null;
					// sameState(curfsm);
					exitState(1, asm);
					return;
					// nextState(curfsm.getHistoryState(this), curfsm);
					// return;
				}
				sameState(curfsm);
				return;

			} else if (sig.equals(SUSPEND_MSG)) {
				// todo Shall be removed, only an fix to accept double Suspend
				// msg from parent
				asm.sendMessage(SUSPENDED_MSG, sig.getSenderRole());
				sameState(curfsm);
				return;
			} else if (sig.equals(ROLE_REMOVE_MSG)) {
				return;

			} else if (sig.equals(ROLE_UPDATE_MSG)) {
				// received message about that my actor descriptor file has been
				// changed. That means that new parts
				// may be created, exiasting ports shall be removed and new
				// ports shal be created.
				// Mark that existing parts are not deleted. It may therefore
				// tempory exists to many parts
				// first read the changed actor sescriptor file
				asm.roleUpdateSender = sig.getSenderRole();
				asm.newActorSpec = asm.getApplicationSpec().getClonedActorSpec(asm.getType());

				if (asm.getActorSpec().equals(asm.newActorSpec)) {
					asm.newActorSpec = null;
					asm.sendMessage(ROLE_UPDATE_ACK_MSG, asm.roleUpdateSender);
					sameState(curfsm);
					return;
				}

				// analyse

				performExit(curfsm);

				asm.sendMessage(CHECK_SPEC_MSG, asm.getMyActorAddress());
				nextState(waitConfirmDeleteConnectors, curfsm);
				return;

			} else if (!(sig.equals(ROLE_REMOVE_MSG) || sig.equals(ROLE_RELEASE_MSG) || sig.equals(ROLE_PLAY_ENDED_MSG)
					|| sig.equals(SET_ACTOR_TRACE_MSG) || sig.equals(REPORT_REQUEST_MSG)
					|| sig.equals(REPORT_RESPONS_MSG) || sig.equals(ACTOR_REPORT_TIMER_MSG))) {
				// save the message until other state is reached.
				save(sig, curfsm);
				return;
			}
		}

		// all state

		// let the enclosing state take care of this timer message
		if (sig.equals(TIMER_MSG) && sig.getProperty(TIMER_ID).

		equals(TIMER_ACTOR_REG_ID)

		)

		{
			return;
		}

	}

	/**
	 * Check if there is a difference between to vectors
	 * 
	 * @param from
	 *            ia vector pf elements
	 * @param to
	 *            is vector of elements of same kind
	 * @return the elements in from vector that is not in the to vector
	 */

	private Vector compareSpecs(Vector from, Vector to, boolean equal) {
		Vector res = new Vector();

		for (int i = 0; i < from.size(); i++) {
			Object oldcs = from.elementAt(i);
			boolean found = false;
			for (int j = 0; j < to.size(); j++) {
				Object newcs = to.elementAt(j);
				if (oldcs.equals(newcs)) {
					// same element found, continue width next Connector Spec
					found = true;
					break;
				}
			}
			// element not found, which means that this element is removed
			if (equal && found) {
				res.addElement(oldcs);
			} else if (!equal && !found) {
				res.addElement(oldcs);
			}
		}
		return res;
	}

	/*
	 * private State getNextState(Hashtable comparedSpec){ State st = suspended;
	 * Vector v = (Vector)comparedSpec.get(ActorSpec.DELETED_CONNECTORS); if
	 * (!v.isEmpty()) return waitConfirmDeleteConnectors;
	 * 
	 * v = (Vector)comparedSpec.get(ActorSpec.DELETED_PARTS); if (!v.isEmpty())
	 * return waitConfirmDeleteParts;
	 * 
	 * 
	 * v = (Vector)comparedSpec.get(ActorSpec.ADDED_PARTS); if (!v.isEmpty())
	 * return waitConfirmDeleteConnectors;
	 * 
	 * v = (Vector)comparedSpec.get(ActorSpec.ADDED_CONNECTOR); if
	 * (!v.isEmpty()) return waitAddedConnectors;
	 * 
	 * return st; }
	 */

}
