/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import org.coos.javaframe.Logger;
import org.coos.javaframe.LoggerFactory;
import org.coos.javaframe.TraceConstants;

import java.util.Date;

/**
 * Logger implementation for CLDC platform.
 * 
 * @author Knut Eilif Husa, Tellu AS
 */
public class CLDCLoggerImpl implements Logger {
	private static boolean consoleLog = true;
	private static boolean remoteLog = false;
	private static String remoteHost = "";
	private static int remotePort;
	String loggerName;
	int traceLevel;

	/*
	 * static { try { RecordStore store =
	 * RecordStore.openRecordStore(ConfigureActorFrameMidlet.storeName, false);
	 * byte[] data = store.getRecord(1); ByteArrayInputStream bais = new
	 * ByteArrayInputStream(data); DataInputStream dis = new
	 * DataInputStream(bais); dis.readUTF(); dis.readUTF(); dis.readUTF();
	 * dis.readUTF();
	 * 
	 * consoleLog = dis.readBoolean(); remoteLog = dis.readBoolean(); remoteHost
	 * = dis.readUTF(); remotePort = (Integer.parseInt(dis.readUTF())); } catch
	 * (RecordStoreException e) { System.out.println("LoggerImpl:" +
	 * e.getMessage()); } catch (IOException e) {
	 * System.out.println("LoggerImpl:" + e.getMessage()); }
	 * 
	 * 
	 * }
	 */
	public CLDCLoggerImpl() {
	}

	public void log(int level, int categories, String value) {
		if (level >= traceLevel) {
			if (level < TraceConstants.tlError) {
				// Debug, info and warnings to stdout
				if (consoleLog) {
					switch (level) {
					case TraceConstants.tlWarn:
						System.out.println(getTimeString() + " WARN " + loggerName + " " + value);

						break;

					case TraceConstants.tlInfo:
						System.out.println(getTimeString() + " INFO " + loggerName + " " + value);

						break;

					case TraceConstants.tlDebug:
						System.out.println(getTimeString() + " DEBUG " + loggerName + " " + value);

						break;
					}
				}

				if (remoteLog) {
					CLDCLoggingEvent le = new CLDCLoggingEvent(remoteHost, remotePort);
					le.log(value, loggerName, level);
				}
			} else {
				// Error and fatals to stderr
				if (consoleLog) {
					switch (level) {
					case TraceConstants.tlError:
						System.err.println(getTimeString() + " ERROR " + loggerName + " " + value);

						break;

					case TraceConstants.tlFatal:
						System.err.println(getTimeString() + " FATAL " + loggerName + " " + value);

						break;
					}
				}

				if (remoteLog) {
					CLDCLoggingEvent le = new CLDCLoggingEvent(remoteHost, remotePort);
					le.log(value, loggerName, level);
				}
			}
		}
	}

	private static String getTimeString() {
		return new Date(System.currentTimeMillis()).toString();
	}

	public void log(int level, String value) {
		log(level, 0, value);
	}

	public String getLoggerName() {
		return loggerName;
	}

	public void setTraceLevel(int traceLevel) {
		this.traceLevel = traceLevel;
	}

	public void setLoggerName(String loggerName) {
		this.loggerName = loggerName;
	}

	public static void setConsoleLog(boolean consoleLog) {
		CLDCLoggerImpl.consoleLog = consoleLog;
	}

	public static void setRemoteLog(boolean remoteLog) {
		CLDCLoggerImpl.remoteLog = remoteLog;
	}

	public static void setRemoteHost(String remoteHost) {
		CLDCLoggerImpl.remoteHost = remoteHost;
	}

	public static void setRemotePort(int remotePort) {
		CLDCLoggerImpl.remotePort = remotePort;
	}

	public boolean isDebugEnabled() {
		return LoggerFactory.isTraceOn() && (traceLevel <= TraceConstants.tlDebug);
	}

	public boolean isErrorEnabled() {
		return LoggerFactory.isTraceOn() && (traceLevel <= TraceConstants.tlError);
	}

	public boolean isInfoEnabled() {
		return LoggerFactory.isTraceOn() && (traceLevel <= TraceConstants.tlInfo);
	}

	public boolean isTraceEnabled() {
		return LoggerFactory.isTraceOn() && isDebugEnabled(); // We are using
		// Debug for our
		// trace...
	}

	public boolean isWarnEnabled() {
		return LoggerFactory.isTraceOn() && (traceLevel <= TraceConstants.tlWarn);
	}
}
