/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import org.coos.util.serialize.AFClassLoader;
import org.coos.util.serialize.AFSerializer;
import org.coos.javaframe.messages.ActorAddressHelper;

import java.io.*;
import java.util.Vector;

/**
 * This class represents the specification of a connector between two Ports.
 * 
 * @author Viggo Fredriksen
 */
public class ConnectorSpec implements AFSerializer {
	private boolean isBidirectional; // flag indicating whether the connector
	// should be bidirectional
	private ActorAddress to;
	private ActorAddress from;
	private String name;

	/**
	 * Default empty constructor.
	 */
	public ConnectorSpec() {
		isBidirectional = true; // Default behaviour is true
		to = null;
	}

	/**
	 * Constructs a connector with no bi-directionality.
	 * 
	 * @param from
	 *            actoraddress.
	 * @param to
	 *            actoraddress.
	 */
	public ConnectorSpec(ActorAddress from, ActorAddress to) {
		this.isBidirectional = false;
		this.from = from;
		this.to = to;
	}

	public ConnectorSpec(DataInputStream din) throws IOException {
		isBidirectional = din.readBoolean();
		from = ActorAddressHelper.resurrect(din);
		to = ActorAddressHelper.resurrect(din);
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Constructor.
	 * 
	 * @param to
	 *            ActorAddress to request the connector to.
	 * @param isBidirectional
	 *            flag indicating whether the connector should be bidirectional.
	 */
	public ConnectorSpec(ActorAddress from, ActorAddress to, boolean isBidirectional) {
		this.isBidirectional = isBidirectional;
		this.from = from;
		this.to = to;
		this.name = "c1";
	}

	/**
	 * Constructor.
	 * 
	 * @param to
	 *            ActorAddress to request the connector to.
	 * @param isBidirectional
	 *            flag indicating whether the connector should be bidirectional.
	 */
	public ConnectorSpec(String from, String to, boolean isBidirectional) {
		this.isBidirectional = isBidirectional;
		this.from = new ActorAddress(from);
		this.to = new ActorAddress(to);
		this.name = "c1";
	}

	public ActorAddress getFrom() {
		return from;
	}

	public void setFrom(ActorAddress from) {
		this.from = from;
	}

	public boolean isBidirectional() {
		return isBidirectional;
	}

	public void setBidirectional(boolean bidirectional) {
		isBidirectional = bidirectional;
	}

	public void setTo(ActorAddress aa) {
		this.to = aa;
	}

	public ActorAddress getTo() {
		return this.to;
	}

	public void setIsBidirectional(boolean isBidirectional) {
		this.isBidirectional = isBidirectional;
	}

	public boolean getIsBidirectional() {
		return this.isBidirectional;
	}

	public static Vector getAddresses(Vector connectorSpecs) {
		Vector v = new Vector();

		for (int i = 0; i < connectorSpecs.size(); i++) {
			ConnectorSpec cs = (ConnectorSpec) connectorSpecs.elementAt(i);
			v.addElement(cs.getTo());
		}

		return v;
	}

	/**
	 * Check equality. Overridden to allow check for property equality.
	 * 
	 * @param obj
	 * @return boolean
	 */
	public boolean equals(Object obj) {
		if (super.equals(obj)) {
			return true;
		}

		if (!(obj instanceof ConnectorSpec)) {
			return false;
		}

		ConnectorSpec cs = (ConnectorSpec) obj;

		return ((cs.isBidirectional == isBidirectional) && cs.getTo().equals(to) && cs.getFrom().equals(from));
	}

	public Object clone() {
		return new ConnectorSpec((ActorAddress) from.clone(), (ActorAddress) to.clone(), isBidirectional);
	}

	public String toString() {
		return "From: " + from + " To: " + to + " Bidirectional: " + isBidirectional;
	}

	/**
	 * This function must implement the serialization of the object.
	 * 
	 * @return a byte array with the objects data
	 * @throws java.io.IOException
	 */
	public byte[] serialize() throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		dout.writeBoolean(isBidirectional);
		dout.write(ActorAddressHelper.persist(from));
		dout.write(ActorAddressHelper.persist(to));
		dout.flush();

		return bout.toByteArray();
	}

	/**
	 * Use this function for resurrection of the object
	 * 
	 * @param data
	 *            The serialized data containing the object data
	 * @throws java.io.IOException
	 */
	public ByteArrayInputStream deSerialize(byte[] data, AFClassLoader cl) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(data);
		DataInputStream din = new DataInputStream(bin);
		isBidirectional = din.readBoolean();
		from = ActorAddressHelper.resurrect(din);
		to = ActorAddressHelper.resurrect(din);

		return bin;
	}
}
