/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import org.coos.javaframe.messages.Message;

/**
 * The message queue is FIFO queue that stores Actor messages. This concept is
 * not intended for JavaFrame users, but only for JavaFrame
 * 
 * @see se.ericsson.eto.norarc.ejbframe.messages.ActorMsg
 */
public class MailBox {
	private Message firstMessage;
	private Message lastMessage;

	// Invariant: if MailBox is empty, then firstMessage and lastMessage ==
	// null;
	// Invariant: lastMessage.nextMessage == null

	/**
	 * Adds the Message sig to the end of this MailBox.
	 * 
	 * @param sig
	 *            assumed to be != null.
	 */
	public final synchronized void addMessage(Message sig) {
		if (firstMessage == null) {
			firstMessage = sig;
		} else {
			lastMessage.nextMessage = sig;
		}

		lastMessage = sig;
		sig.nextMessage = null;
	}

	/**
	 * Adds the Message sig to the front of this MailBox.
	 * 
	 * @param sig
	 *            assumed to be != null.
	 */
	public final synchronized void addMessageFirst(Message sig) {
		if (firstMessage == null) {
			lastMessage = sig;
		}

		sig.nextMessage = firstMessage;
		firstMessage = sig;
	}

	/**
	 * Moves all the messages in mBox to the front of this mailbox. mBox is
	 * cleared.
	 * 
	 * @param mBox
	 *            assumed to be != null.
	 */
	public final void moveMailBox(MailBox mBox) {
		if (mBox.firstMessage == null) {
			return;
		}

		if (firstMessage == null) {
			lastMessage = mBox.lastMessage;
		} else {
			mBox.lastMessage.nextMessage = firstMessage;
		}

		firstMessage = mBox.firstMessage;
		mBox.firstMessage = null;
		mBox.lastMessage = null;
	}

	/**
	 * Removes the first message of this Mailbox and returns it. If this mailbox
	 * is empty, null is returned.
	 */
	public final synchronized Message removeFirst() {
		if (firstMessage == null) {
			return null;
		}

		Message first = firstMessage;

		if (firstMessage == lastMessage) {
			firstMessage = null;
			lastMessage = null;
		} else {
			firstMessage = firstMessage.nextMessage;
		}

		first.nextMessage = null;

		return first;
	}

	/**
	 * Gets without removing the first message of this Mailbox and returns it.
	 * If this mailbox is empty, null is returned.
	 */
	public final Message getFirst() {
		return firstMessage;
	}

	/**
	 * Check if mailbox is empty
	 * 
	 * @return
	 */
	public final boolean isEmpty() {
		return (lastMessage == null);
	}

	/**
	 * Empty the Mailbox for all messages.
	 */
	public final synchronized void clearMailBox() {
		while (!isEmpty()) {
			removeFirst();
		}
	}
}
