/*
 * Decompiled with CFR 0.152.
 */
package ch.vorburger.mariadb4j;

import ch.vorburger.exec.ManagedProcess;
import ch.vorburger.exec.ManagedProcessBuilder;
import ch.vorburger.exec.ManagedProcessException;
import ch.vorburger.exec.ManagedProcessListener;
import ch.vorburger.exec.OutputStreamLogDispatcher;
import ch.vorburger.mariadb4j.DBConfiguration;
import ch.vorburger.mariadb4j.DBConfigurationBuilder;
import ch.vorburger.mariadb4j.DBShutdownHook;
import ch.vorburger.mariadb4j.MariaDBOutputStreamLogDispatcher;
import ch.vorburger.mariadb4j.Util;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DB {
    private static final Logger logger = LoggerFactory.getLogger(DB.class);
    protected final DBConfiguration configuration;
    private File baseDir;
    private File libDir;
    private File dataDir;
    private File tmpDir;
    private ManagedProcess mysqldProcess;
    protected int dbStartMaxWaitInMS = 30000;
    private static final String ROOT_PASSWORD_PLACEHOLDER = "{root_password}";
    private static final String MYSQL_SECURE_INSTALLATION_INTERACTION = "\nn\ny\n{root_password}\n{root_password}\ny\ny\ny\ny";

    protected DB(DBConfiguration config) {
        this.configuration = config;
    }

    public DBConfiguration getConfiguration() {
        return this.configuration;
    }

    public static DB newEmbeddedDB(DBConfiguration config) throws ManagedProcessException {
        DB db = new DB(config);
        db.prepareDirectories();
        db.unpackEmbeddedDb();
        db.install();
        if (!db.configuration.isSecurityDisabled()) {
            db.runMysqlSecureInstallationScript();
        }
        return db;
    }

    public static DB newEmbeddedDB(int port) throws ManagedProcessException {
        DBConfigurationBuilder config = new DBConfigurationBuilder();
        config.setPort(port);
        return DB.newEmbeddedDB(config.build());
    }

    protected ManagedProcess createDBInstallProcess() throws ManagedProcessException, IOException {
        logger.info("Installing a new embedded database to: " + this.baseDir);
        File installDbCmdFile = this.configuration.getExecutable(DBConfiguration.Executable.InstallDB);
        ManagedProcessBuilder builder = new ManagedProcessBuilder(installDbCmdFile);
        builder.setOutputStreamLogDispatcher(this.getOutputStreamLogDispatcher("mysql_install_db"));
        builder.getEnvironment().put(this.configuration.getOSLibraryEnvironmentVarName(), this.libDir.getAbsolutePath());
        builder.addArgument("--datadir=" + this.dataDir.getAbsolutePath(), false).setWorkingDirectory(this.baseDir);
        if (!this.configuration.isWindows()) {
            builder.addArgument("--auth-root-authentication-method=normal");
            builder.addArgument("--datadir=" + this.dataDir.getAbsolutePath(), false);
            builder.addArgument("--tmpdir=" + this.tmpDir.getAbsolutePath(), false);
            builder.addArgument("--basedir=" + this.baseDir.getAbsolutePath(), false);
            builder.addArgument("--no-defaults");
            builder.addArgument("--force");
            builder.addArgument("--skip-name-resolve");
        } else {
            builder.addFileArgument("--datadir", this.dataDir.getCanonicalFile());
            builder.addFileArgument("--tmpdir", this.tmpDir.getCanonicalFile());
        }
        return builder.build();
    }

    ManagedProcess secureInstallPreparation() throws ManagedProcessException, IOException {
        logger.info("Run mysql_secure_installation for embedded database at: " + this.baseDir);
        File installDbCmdFile = this.newExecutableFile("bin", "mysql_secure_installation");
        if (!installDbCmdFile.exists()) {
            throw new ManagedProcessException("mysql_secure_installation was not found in bin/ under " + this.baseDir.getAbsolutePath());
        }
        ManagedProcessBuilder builder = new ManagedProcessBuilder(installDbCmdFile);
        builder.setOutputStreamLogDispatcher(this.getOutputStreamLogDispatcher("mysql_secure_installation"));
        builder.getEnvironment().put(this.configuration.getOSLibraryEnvironmentVarName(), this.libDir.getAbsolutePath());
        String interaction = StringUtils.replace((String)MYSQL_SECURE_INSTALLATION_INTERACTION, (String)ROOT_PASSWORD_PLACEHOLDER, (String)this.configuration.getDefaultRootPassword());
        builder.setInputStream((InputStream)new ByteArrayInputStream(interaction.getBytes(Charset.forName(StandardCharsets.US_ASCII.name()))));
        builder.addArgument("--basedir=" + this.baseDir.getAbsolutePath(), false).setWorkingDirectory(this.baseDir);
        this.addPortAndMaybeSocketArguments(builder);
        ManagedProcess mysqlInstallProcess = builder.build();
        return mysqlInstallProcess;
    }

    private static File toWindowsPath(File file) throws IOException {
        return new File(file.getCanonicalPath().replace(" ", "%20"));
    }

    protected synchronized void install() throws ManagedProcessException {
        try {
            ManagedProcess mysqlInstallProcess = this.createDBInstallProcess();
            mysqlInstallProcess.start();
            mysqlInstallProcess.waitForExit();
        }
        catch (Exception e) {
            throw new ManagedProcessException("An error occurred while installing the database", (Throwable)e);
        }
        logger.info("Installation complete.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected synchronized void runMysqlSecureInstallationScript() throws ManagedProcessException {
        if (StringUtils.isEmpty((CharSequence)this.configuration.getDefaultRootPassword())) {
            logger.info("*** crafter-set-env.sh hasn't been upgraded within your bundle. The property MARIADB_ROOT_PASSWD is set to empty. The database secure installation script will not be run. ***");
            return;
        }
        ManagedProcess mysqlSecureInstallProcess = null;
        try {
            logger.info("Start secure database script");
            this.start();
            try {
                Class.forName(this.configuration.getDriverClassName());
            }
            catch (Exception e) {
                logger.error("Error connecting to database", (Throwable)e);
            }
            boolean secured = false;
            String jdbcUrl = "jdbc:mysql://localhost:" + this.configuration.getPort() + "/mysql";
            try (Connection conn = DriverManager.getConnection(jdbcUrl, "root", "");){
                secured = false;
                logger.info("Unsecured database detected, running the secure installation script against the database.");
            }
            catch (SQLException e) {
                secured = true;
                logger.info("Secured database detected.");
                logger.debug("Can not connect to database as root without password. Secured database detected.", (Throwable)e);
            }
            catch (Exception e) {
                logger.error("Can not connect to database as root without password.", (Throwable)e);
            }
            if (!secured) {
                mysqlSecureInstallProcess = this.secureInstallPreparation();
                mysqlSecureInstallProcess.start();
                mysqlSecureInstallProcess.waitForExit();
                logger.info("Securing database complete.");
            }
        }
        catch (Exception e) {
            logger.error("Error when trying to secure database.", (Throwable)e);
        }
        finally {
            if (mysqlSecureInstallProcess != null && mysqlSecureInstallProcess.isAlive()) {
                mysqlSecureInstallProcess.destroy();
            }
            this.stop();
        }
    }

    protected String getWinExeExt() {
        return this.configuration.isWindows() ? ".exe" : "";
    }

    public synchronized void start() throws ManagedProcessException {
        logger.info("Starting up the database...");
        boolean ready = false;
        try {
            this.mysqldProcess = this.startPreparation();
            ready = this.mysqldProcess.startAndWaitForConsoleMessageMaxMs(this.getReadyForConnectionsTag(), (long)this.dbStartMaxWaitInMS);
        }
        catch (Exception e) {
            logger.error("failed to start mysqld", (Throwable)e);
            throw new ManagedProcessException("An error occurred while starting the database", (Throwable)e);
        }
        if (!ready) {
            if (this.mysqldProcess != null && this.mysqldProcess.isAlive()) {
                this.mysqldProcess.destroy();
            }
            throw new ManagedProcessException("Database does not seem to have started up correctly? Magic string not seen in " + this.dbStartMaxWaitInMS + "ms: " + this.getReadyForConnectionsTag() + this.mysqldProcess.getLastConsoleLines());
        }
        logger.info("Database startup complete.");
    }

    protected String getReadyForConnectionsTag() {
        return ": ready for connections.";
    }

    synchronized ManagedProcess startPreparation() throws ManagedProcessException, IOException {
        ManagedProcessBuilder builder = new ManagedProcessBuilder(this.configuration.getExecutable(DBConfiguration.Executable.Server));
        builder.setOutputStreamLogDispatcher(this.getOutputStreamLogDispatcher("mysqld"));
        builder.getEnvironment().put(this.configuration.getOSLibraryEnvironmentVarName(), this.libDir.getAbsolutePath());
        builder.addArgument("--no-defaults");
        builder.addArgument("--console");
        if (this.configuration.isSecurityDisabled()) {
            builder.addArgument("--skip-grant-tables");
        }
        if (!this.hasArgument("--max_allowed_packet")) {
            builder.addArgument("--max_allowed_packet=64M");
        }
        builder.addArgument("--basedir=" + this.baseDir.getAbsolutePath(), false).setWorkingDirectory(this.baseDir);
        if (!this.configuration.isWindows()) {
            builder.addArgument("--datadir=" + this.dataDir.getAbsolutePath(), false);
            builder.addArgument("--tmpdir=" + this.tmpDir.getAbsolutePath(), false);
        } else {
            builder.addFileArgument("--datadir", this.dataDir.getCanonicalFile());
            builder.addFileArgument("--tmpdir", this.tmpDir.getCanonicalFile());
        }
        this.addPortAndMaybeSocketArguments(builder);
        for (String arg : this.configuration.getArgs()) {
            builder.addArgument(arg);
        }
        if (StringUtils.isNotBlank((CharSequence)this.configuration.getDefaultCharacterSet())) {
            builder.addArgument("--character-set-server=", this.configuration.getDefaultCharacterSet());
        }
        this.cleanupOnExit();
        builder.setDestroyOnShutdown(false);
        logger.info("mysqld executable: " + builder.getExecutable());
        return builder.build();
    }

    protected boolean hasArgument(String argumentName) {
        for (String argument : this.configuration.getArgs()) {
            if (!argument.startsWith(argumentName)) continue;
            return true;
        }
        return false;
    }

    protected File newExecutableFile(String dir, String exec) {
        return new File(this.baseDir, dir + "/" + exec + this.getWinExeExt());
    }

    protected void addPortAndMaybeSocketArguments(ManagedProcessBuilder builder) throws IOException {
        builder.addArgument("--port=" + this.configuration.getPort());
        if (!this.configuration.isWindows()) {
            builder.addFileArgument("--socket", this.getAbsoluteSocketFile());
        }
    }

    protected void addSocketOrPortArgument(ManagedProcessBuilder builder) throws IOException {
        if (!this.configuration.isWindows()) {
            builder.addFileArgument("--socket", this.getAbsoluteSocketFile());
        } else {
            builder.addArgument("--port=" + this.configuration.getPort());
        }
    }

    protected File getAbsoluteSocketFile() {
        String socket = this.configuration.getSocket();
        File socketFile = new File(socket);
        return socketFile.getAbsoluteFile();
    }

    public void source(String resource) throws ManagedProcessException {
        this.source(resource, null, null, null);
    }

    public void source(InputStream resource) throws ManagedProcessException {
        this.source(resource, null, null, null);
    }

    public void source(String resource, String dbName) throws ManagedProcessException {
        this.source(resource, null, null, dbName);
    }

    public void source(InputStream resource, String dbName) throws ManagedProcessException {
        this.source(resource, null, null, dbName);
    }

    public void source(InputStream resource, String username, String password, String dbName) throws ManagedProcessException {
        this.run("script file sourced from an InputStream", resource, username, password, dbName, false);
    }

    public void source(String resource, String username, String password, String dbName) throws ManagedProcessException {
        this.source(resource, username, password, dbName, false);
    }

    public void source(String resource, String username, String password, String dbName, boolean force) throws ManagedProcessException {
        try (InputStream from = this.getClass().getClassLoader().getResourceAsStream(resource);){
            if (from == null) {
                throw new IllegalArgumentException("Could not find script file on the classpath at: " + resource);
            }
            this.run("script file sourced from the classpath at: " + resource, from, username, password, dbName, force);
        }
        catch (IOException ioe) {
            logger.warn("Issue trying to close source InputStream. Raise warning and continue.", (Throwable)ioe);
        }
    }

    public void run(String command, String username, String password, String dbName) throws ManagedProcessException {
        this.run(command, username, password, dbName, false, true);
    }

    public void run(String command) throws ManagedProcessException {
        this.run(command, null, null, null);
    }

    public void run(String command, String username, String password) throws ManagedProcessException {
        this.run(command, username, password, null);
    }

    public void run(String command, String username, String password, String dbName, boolean force) throws ManagedProcessException {
        this.run(command, username, password, dbName, force, true);
    }

    public void run(String command, String username, String password, String dbName, boolean force, boolean verbose) throws ManagedProcessException {
        try (InputStream from = IOUtils.toInputStream((String)command, (Charset)Charset.defaultCharset());){
            String logInfoText = verbose ? "command: " + command : "command (" + command.length() / 1024 + " KiB long)";
            this.run(logInfoText, from, username, password, dbName, force);
        }
        catch (IOException ioe) {
            logger.warn("Issue trying to close source InputStream. Raise warning and continue.", (Throwable)ioe);
        }
    }

    protected void run(String logInfoText, InputStream fromIS, String username, String password, String dbName, boolean force) throws ManagedProcessException {
        logger.info("Running a " + logInfoText);
        try {
            ManagedProcessBuilder builder = new ManagedProcessBuilder(this.configuration.getExecutable(DBConfiguration.Executable.Client));
            builder.setOutputStreamLogDispatcher(this.getOutputStreamLogDispatcher("mysql"));
            builder.setWorkingDirectory(this.baseDir);
            builder.addArgument("--default-character-set=utf8");
            if (username != null && !username.isEmpty()) {
                builder.addArgument("-u", username);
            }
            if (password != null && !password.isEmpty()) {
                builder.addArgument("-p", password);
            }
            if (dbName != null && !dbName.isEmpty()) {
                builder.addArgument("-D", dbName);
            }
            if (force) {
                builder.addArgument("-f");
            }
            this.addSocketOrPortArgument(builder);
            if (fromIS != null) {
                builder.setInputStream(fromIS);
            }
            if (this.configuration.getProcessListener() != null) {
                builder.setProcessListener(this.configuration.getProcessListener());
            }
            if (this.configuration.getDefaultCharacterSet() != null) {
                builder.addArgument("--default-character-set=", this.configuration.getDefaultCharacterSet());
            }
            ManagedProcess process = builder.build();
            process.start();
            process.waitForExit();
        }
        catch (Exception e) {
            throw new ManagedProcessException("An error occurred while running a " + logInfoText, (Throwable)e);
        }
        logger.info("Successfully ran the " + logInfoText);
    }

    public void createDB(String dbName) throws ManagedProcessException {
        this.run("create database if not exists `" + dbName + "`;");
    }

    public void createDB(String dbName, String username, String password) throws ManagedProcessException {
        this.run("create database if not exists `" + dbName + "`;", username, password);
    }

    protected OutputStreamLogDispatcher getOutputStreamLogDispatcher(String exec) {
        return new MariaDBOutputStreamLogDispatcher();
    }

    public synchronized void stop() throws ManagedProcessException {
        if (this.mysqldProcess != null && this.mysqldProcess.isAlive()) {
            logger.debug("Stopping the database...");
            this.mysqldProcess.destroy();
            logger.info("Database stopped.");
        } else {
            logger.debug("Database was already stopped.");
        }
    }

    protected void unpackEmbeddedDb() {
        if (this.configuration.getBinariesClassPathLocation() == null) {
            logger.info("Not unpacking any embedded database (as BinariesClassPathLocation configuration is null)");
            return;
        }
        try {
            Util.extractFromClasspathToFile(this.configuration.getBinariesClassPathLocation(), this.baseDir);
            if (!this.configuration.isWindows()) {
                Util.forceExecutable(this.configuration.getExecutable(DBConfiguration.Executable.PrintDefaults));
                Util.forceExecutable(this.configuration.getExecutable(DBConfiguration.Executable.InstallDB));
                Util.forceExecutable(this.configuration.getExecutable(DBConfiguration.Executable.Server));
                Util.forceExecutable(this.configuration.getExecutable(DBConfiguration.Executable.Dump));
                Util.forceExecutable(this.configuration.getExecutable(DBConfiguration.Executable.Client));
                Util.forceExecutable(this.newExecutableFile("bin", "mysql_secure_installation"));
                Util.forceExecutable(this.newExecutableFile("bin", "mysql_upgrade"));
                Util.forceExecutable(this.newExecutableFile("bin", "mysqlcheck"));
                Util.forceExecutable(this.newExecutableFile("bin", "resolveip"));
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Error unpacking embedded DB", e);
        }
    }

    protected void prepareDirectories() throws ManagedProcessException {
        this.baseDir = Util.getDirectory(this.configuration.getBaseDir());
        this.libDir = Util.getDirectory(this.configuration.getLibDir());
        this.tmpDir = Util.getDirectory(this.configuration.getTmpDir());
        try {
            String dataDirPath = this.configuration.getDataDir();
            if (Util.isTemporaryDirectory(dataDirPath)) {
                FileUtils.deleteDirectory((File)new File(dataDirPath));
            }
            this.dataDir = Util.getDirectory(dataDirPath);
        }
        catch (Exception e) {
            throw new ManagedProcessException("An error occurred while preparing the data directory", (Throwable)e);
        }
    }

    protected void cleanupOnExit() {
        String threadName = "Shutdown Hook Deletion Thread for Temporary DB " + this.dataDir.toString();
        DB db = this;
        Runtime.getRuntime().addShutdownHook(new DBShutdownHook(threadName, db, () -> this.mysqldProcess, () -> this.baseDir, () -> this.dataDir, () -> this.tmpDir, this.configuration));
    }

    public ManagedProcess dumpXML(File outputFile, String dbName, String user, String password) throws IOException, ManagedProcessException {
        return this.dump(outputFile, Arrays.asList(dbName), true, true, true, user, password);
    }

    public ManagedProcess dumpSQL(File outputFile, String dbName, String user, String password) throws IOException, ManagedProcessException {
        return this.dump(outputFile, Arrays.asList(dbName), true, true, false, user, password);
    }

    protected ManagedProcess dump(File outputFile, List<String> dbNamesToDump, boolean compactDump, boolean lockTables, boolean asXml, String user, String password) throws ManagedProcessException, IOException {
        ManagedProcessBuilder builder = new ManagedProcessBuilder(this.configuration.getExecutable(DBConfiguration.Executable.Dump));
        final BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(outputFile));
        builder.addStdOut((OutputStream)outputStream);
        builder.setOutputStreamLogDispatcher(this.getOutputStreamLogDispatcher("mysqldump"));
        builder.addArgument("--port=" + this.configuration.getPort());
        if (!this.configuration.isWindows()) {
            builder.addFileArgument("--socket", this.getAbsoluteSocketFile());
        }
        if (lockTables) {
            builder.addArgument("--flush-logs");
            builder.addArgument("--lock-tables");
        }
        if (compactDump) {
            builder.addArgument("--compact");
        }
        if (asXml) {
            builder.addArgument("--xml");
        }
        if (StringUtils.isNotBlank((CharSequence)user)) {
            builder.addArgument("-u");
            builder.addArgument(user);
            if (StringUtils.isNotBlank((CharSequence)password)) {
                builder.addArgument("-p" + password);
            }
        }
        builder.addArgument(StringUtils.join(dbNamesToDump, (String)" "));
        builder.setDestroyOnShutdown(true);
        builder.setProcessListener(new ManagedProcessListener(){

            public void onProcessComplete(int i) {
                this.closeOutputStream();
            }

            public void onProcessFailed(int i, Throwable throwable) {
                this.closeOutputStream();
            }

            private void closeOutputStream() {
                try {
                    outputStream.close();
                }
                catch (IOException exception) {
                    logger.error("Problem while trying to close the stream to the file containing the DB dump", (Throwable)exception);
                }
            }
        });
        return builder.build();
    }
}

